/*  
 *  fview/fview.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:46 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_fpoint.h>
#include <gdl/gdl_fview.h>

struct _gdl_fview
{
	const gdl_fview_type * type;
	void * data;
};

gdl_fview *
gdl_fview_alloc (const gdl_fview_type * T)
{
	gdl_fview * g = GDL_MALLOC (gdl_fview, 1);
	
	g->type = T;
	
	g->data = gdl_malloc (T->size);
	
	if (g->data)
	{
		(T->alloc)(g->data);
	}
	else
	{
		GDL_FREE (g);
		g = NULL;
	}
		
	return g;
}

void
gdl_fview_free (gdl_fview * g)
{
	if (g)
	{
		(g->type->free)(g->data);
		GDL_FREE (g->data);
		GDL_FREE (g);
	}	
}

gdl_fview *
gdl_fview_fread (FILE * stream)
{
	if (stream)
	{
		const gdl_fview_type * T;
		gdl_fview * g;
		
		T = gdl_fview_type_fread (stream);
		GDL_FREAD_STATUS (T!=0, 1);
		
		g = gdl_fview_alloc (T);
		(g->type->free)(g->data);
		GDL_FREE (g->data);
		
		g->data = (g->type->fread)(stream);
		GDL_FREAD_STATUS (g->data!=0, 1);
	
		return g;
	}
	
	return NULL;
}

int
gdl_fview_fwrite (FILE * stream, gdl_fview * g)
{
	if (stream && g)
	{
		int status;
		
		status = gdl_fview_type_fwrite (stream, g->type);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = (g->type->fwrite)(stream, g->data);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}

size_t
gdl_fview_factor_size (const gdl_fview * g)
{
	return (g->type->factor_size)(g->data);
}

size_t
gdl_fview_accession_size (const gdl_fview * g)
{
	return (g->type->accession_size)(g->data);	
}

gdl_factor *
gdl_fview_add_factor (gdl_fview * g, const gdl_factor_type * T, const gdl_string * name)
{
	return (g->type->add_factor) (g->data, T, name);
}

gdl_accession *
gdl_fview_add_accession (gdl_fview * g, const gdl_string * name)
{
	return (g->type->add_accession) (g->data, name);
}

gdl_factor *
gdl_fview_get_factor (const gdl_fview * g, size_t i)
{
	return (g->type->get_factor)(g->data, i);
}

gdl_accession *
gdl_fview_get_accession (const gdl_fview * g, size_t i)
{
	return (g->type->get_accession)(g->data, i);	
}

gdl_factor *
gdl_fview_search_factor (const gdl_fview * g, const gdl_string * name)
{
	return (g->type->search_factor)(g->data, name);
}

gdl_accession *
gdl_fview_search_accession (const gdl_fview * g, const gdl_string * name)
{
	return (g->type->search_accession)(g->data, name);
}

gdl_gvalues_get *
gdl_fview_get_new (const gdl_fview * g)
{
	return (g->type->get_new)(g->data);
}

int
gdl_fview_set_fdatapoint (gdl_fview * g, const gdl_accession * va, const gdl_factor * vl, gdl_fdatapoint * gd)
{
	return (g->type->set_fdatapoint)(g->data, va, vl, gd);
}

int
gdl_fview_get_value (const gdl_fview * g, const gdl_accession * a, const gdl_factor * l, gdl_gvalues_get * gb)
{
	return (g->type->get)(g->data, a, l, gb);
}

gdl_boolean
gdl_fview_is_missing (const gdl_fview * g, const gdl_accession * a, const gdl_factor * l)
{
	return (g->type->is_missing)(g->data, a, l);
}

size_t
gdl_fview_accession_missing_size (const gdl_fview * g, const gdl_accession * a)
{
	return gdl_fview_accession_missing_size_f (g, a->idx);
}

size_t
gdl_fview_factor_missing_size (const gdl_fview * g, const gdl_factor * l)
{
	return gdl_fview_factor_missing_size_f (g, l->idx);
}

int
gdl_fview_set_fdatapoint_f (gdl_fview * g, size_t a, size_t l, gdl_fdatapoint * gd)
{
	return (g->type->set_fdatapoint_f)(g->data, a, l, gd);
}

int
gdl_fview_get_value_f (const gdl_fview * g, size_t i, size_t j, gdl_gvalues_get * gb)
{
	return (g->type->get_f)(g->data, i, j, gb);
}

gdl_boolean
gdl_fview_is_missing_f (const gdl_fview * g, size_t i, size_t j)
{
	return (g->type->is_missing_f)(g->data, i, j);
}

size_t
gdl_fview_accession_missing_size_f (const gdl_fview * g, size_t a)
{
	size_t i, nl, nm=0;
	
	nl = gdl_fview_factor_size (g);
	
	for (i = 0; i < nl; i++)
	{
		nm += gdl_fview_is_missing_f (g, a, i);
	}
	
	return nm;	
}

size_t
gdl_fview_factor_missing_size_f (const gdl_fview * g, size_t l)
{
	size_t i, na, nm=0;
	
	na = gdl_fview_accession_size (g);
	
	for (i = 0; i < na; i++)
	{
		nm += gdl_fview_is_missing_f (g, i, l);
	}
	
	return nm;	
}

int
gdl_fview_add (gdl_fview * g, const gdl_string * name, gdl_fview * s, size_t owner)
{
	if (g->type->add)
	{
		return (g->type->add)(g->data, name, s, owner);
	}
	else
	{
		GDL_ERROR_VAL ("Cannot add a fview : fonction not available",
		               GDL_EINVAL,
		               -1);
	}	
}

static void
_gdl_fview_free (void * vg)
{
	if (vg)
	{
		gdl_fview * g = (gdl_fview *) vg;
		gdl_fview_free (g);
	}	
}

static void *
_gdl_fview_fread (FILE * stream)
{
	return gdl_fview_fread (stream);
}

static int
_gdl_fview_fwrite (FILE * stream, const void * vg)
{
	return gdl_fview_fwrite (stream, (gdl_fview *) vg);
}

static const gdl_data_interface _interface=
{
	&_gdl_fview_free,
	NULL,
	NULL,
    &_gdl_fview_fread,
    &_gdl_fview_fwrite
};

const gdl_data_interface * gdl_fview_interface = &_interface;
