/*  
 *  fview/mask.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:46 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_fview.h>
#include <gdl/gdl_fview_mask.h>

size_t
gdl_fview_mask_accession_size (const gdl_fview * fview, const gdl_mask * m)
{
	const gdl_entity_mask * em = gdl_mask_get (m, GDL_ACCESSION);
	
	if (em)
	{
		return gdl_entity_mask_size (em);
	}
	else
	{
		return gdl_fview_accession_size (fview);
	}
}

size_t
gdl_fview_mask_factor_size (const gdl_fview * fview, const gdl_mask * m)
{
	const gdl_entity_mask * em = gdl_mask_get (m, GDL_FACTOR);
	
	if (em)
	{
		return gdl_entity_mask_size (em);
	}
	else
	{
		return gdl_fview_factor_size (fview);
	}	
}

gdl_accession *
gdl_fview_mask_get_accession (const gdl_fview * fview, const gdl_mask * m, size_t i)
{
	return gdl_fview_get_accession (fview, gdl_mask_get_idx (m, GDL_ACCESSION, i));	
}

gdl_factor *
gdl_fview_mask_get_factor (const gdl_fview * fview, const gdl_mask * m, size_t i)
{
	return gdl_fview_get_factor (fview, gdl_mask_get_idx (m, GDL_FACTOR, i));
}

gdl_gvalues_get *
gdl_fview_mask_get_new (const gdl_fview * g, const gdl_mask * m)
{
	gdl_gvalues_get * get;
	size_t i, size, nl;
	
	nl = GDL_FVIEW_FACTOR_SIZE (g, m);
	
	for (size = i = 0; i < nl; i++)
	{
		gdl_factor * vl = GDL_FVIEW_GET_FACTOR (g, m, i);
		if (size < gdl_factor_size (vl))
		{
			size = gdl_factor_size (vl);
		}
	}
	
	return gdl_gvalues_get_alloc (size);
}

int
gdl_fview_mask_get_value (const gdl_fview * fview, const gdl_mask * m, size_t i, size_t j, gdl_gvalues_get * g)
{
	return gdl_fview_get_value_f (fview, gdl_mask_get_idx (m, GDL_ACCESSION, i), gdl_mask_get_idx (m, GDL_FACTOR, j), g);
}

gdl_boolean
gdl_fview_mask_is_missing (const gdl_fview * fview, const gdl_mask * m, size_t i, size_t j)
{
	return gdl_fview_is_missing_f (fview, gdl_mask_get_idx (m, GDL_ACCESSION, i), gdl_mask_get_idx (m, GDL_FACTOR, j));
}

static size_t
gdl_fview_mask_accession_missing_size_f (const gdl_fview * fview, size_t i, const gdl_mask * m)
{
	size_t j, nl, nm=0;
	
	nl = gdl_mask_get_size (m, GDL_FACTOR);
	
	for (j = 0; j < nl; j++)
	{
		nm += gdl_fview_is_missing_f (fview, i, gdl_mask_get_idx (m, GDL_FACTOR, j));
	}
	
	return nm;
}

static size_t
gdl_fview_mask_factor_missing_size_f (const gdl_fview * fview, size_t i, const gdl_mask * m)
{
	size_t j, na, nm=0;
	
	na = gdl_mask_get_size (m, GDL_ACCESSION);
	
	for (j = 0; j < na; j++)
	{
		nm += gdl_fview_is_missing_f (fview, gdl_mask_get_idx (m, GDL_ACCESSION, j), i);
	}
	
	return nm;
}

size_t
gdl_fview_mask_accession_missing_size (const gdl_fview * fview, const gdl_mask * m, size_t i)
{
	const gdl_entity_mask * factor = gdl_mask_get (m, GDL_FACTOR);
	
	if (factor)
	{
		return gdl_fview_mask_accession_missing_size_f (fview, gdl_mask_get_idx (m, GDL_ACCESSION, i), m);
	}
	else
	{
		return gdl_fview_accession_missing_size_f (fview, gdl_mask_get_idx (m, GDL_ACCESSION, i));
	}	
}

size_t
gdl_fview_mask_factor_missing_size (const gdl_fview * fview, const gdl_mask * m, size_t i)
{
	const gdl_entity_mask * accession = gdl_mask_get (m, GDL_ACCESSION);
	
	if (accession)
	{
		return gdl_fview_mask_factor_missing_size_f (fview, gdl_mask_get_idx (m, GDL_FACTOR, i), m);
	}
	else
	{
		return gdl_fview_factor_missing_size_f (fview, gdl_mask_get_idx (m, GDL_FACTOR, i));
	}	
}

size_t
GDL_FVIEW_ACCESSION_SIZE (const gdl_fview * g, const gdl_mask * m)
{
	if (m) return gdl_fview_mask_accession_size (g, m);
    else return gdl_fview_accession_size (g);
}

size_t
GDL_FVIEW_FACTOR_SIZE (const gdl_fview * g, const gdl_mask * m)
{
	if (m) return gdl_fview_mask_factor_size (g, m);
    else return gdl_fview_factor_size (g);
}

gdl_accession * 
GDL_FVIEW_GET_ACCESSION (const gdl_fview * g, const gdl_mask * m, size_t i)
{
	if (m) return gdl_fview_mask_get_accession (g, m, i);
    else return gdl_fview_get_accession (g, i);
}

gdl_factor *
GDL_FVIEW_GET_FACTOR (const gdl_fview * g, const gdl_mask * m, size_t i)
{
	if (m) return gdl_fview_mask_get_factor (g, m, i);
	else return gdl_fview_get_factor (g, i);	
}

gdl_boolean
GDL_FVIEW_IS_MISSING (const gdl_fview * g, const gdl_mask * m, size_t i, size_t j)
{
	if (m) return gdl_fview_mask_is_missing (g, m, i, j);
    else return gdl_fview_is_missing_f (g, i, j);
}

size_t
GDL_FVIEW_ACCESSION_MISSING_SIZE (const gdl_fview * g, const gdl_mask * m, size_t i)
{
	if (m) return gdl_fview_mask_accession_missing_size (g, m, i);
    else return gdl_fview_accession_missing_size_f (g, i);
}

size_t
GDL_FVIEW_FACTOR_MISSING_SIZE (const gdl_fview * g, const gdl_mask * m, size_t i)
{
	if (m) return gdl_fview_mask_factor_missing_size (g, m, i);
    else return gdl_fview_factor_missing_size_f (g, i);
}

gdl_gvalues_get *
GDL_FVIEW_GET_NEW (const gdl_fview * g, const gdl_mask * m)
{
	if (m) return gdl_fview_mask_get_new (g, m);
    else return gdl_fview_get_new (g);	
}

int
GDL_FVIEW_GET_VALUE (const gdl_fview * g, const gdl_mask * m, size_t i, size_t j, gdl_gvalues_get * x)
{
    if (m) return gdl_fview_mask_get_value (g, m, i, j, x);
    else return gdl_fview_get_value_f (g, i, j, x);
}
