/*  
 *  fview/rcont.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:46 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <stdio.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_util.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_fsheet.h>
#include <gdl/gdl_fview.h>
#include <gdl/gdl_fview_reader.h>

static const gdl_fsheet_type STANDARD =
{
	1,
	1,
	"",
	"\t",
	"\n",
	"\t",
	"\t"	
};

static gdl_fsheet_type *
_gdl_fview_reader_continuous_fsheet_type (const gdl_fview_reader_type * type)
{
	gdl_fsheet_type * ftype = gdl_fsheet_type_default ();
	
   ftype->rname  = type->has_acc_name;
	ftype->cname  = type->has_fac_header;
	gdl_string_free (ftype->fsep);
	ftype->fsep   = gdl_string_clone (type->sep);
	gdl_string_free (ftype->rnfsep);
	ftype->rnfsep = gdl_string_clone (type->sep);
	gdl_string_free (ftype->cnfsep);
	ftype->cnfsep = gdl_string_clone (type->sep);
	gdl_string_free (ftype->tsep);
	ftype->tsep   = gdl_string_clone (STANDARD.tsep);
	gdl_string_free (ftype->rsep);
	ftype->rsep   = gdl_string_clone (STANDARD.rsep);
	
	return ftype;
}

static gdl_fsheet *
_gdl_fview_reader_continuous_fsheet (gdl_fview_reader * reader)
{
	gdl_fsheet_type * ftype = _gdl_fview_reader_continuous_fsheet_type (reader->type);
	gdl_fsheet * fs;
	
	fs = gdl_fsheet_alloc (ftype);
	
	gdl_fsheet_init (fs, reader->stream);
	
	return fs;
}

static int
_gdl_fview_reader_continuous_init (gdl_fview_reader * reader, gdl_fsheet * fs)
{
	size_t n, f;
	
	n = gdl_fsheet_nrow (fs);
	f = gdl_fsheet_ncolumn (fs);
	
	if (!n || !f)
	{
		return GDL_EINVAL;
	}
	
	reader->fview = gdl_fview_alloc (gdl_fview_standard);
	
	return GDL_SUCCESS;
}

static gdl_fview_reader_cell *
gdl_fview_reader_continuous_parse_cell (gdl_fview_reader * reader, gdl_string * cell)
{
	if (reader && cell)
	{
		gdl_fview_reader_cell * pcell = NULL;
		
		pcell = gdl_fview_reader_cell_alloc ();
		
		if (strcmp (cell, reader->type->missing))
		{
			pcell->point            = gdl_gvalues_alloc (1);
			pcell->point->values[0] = gdl_gvalue_alloc ();
			pcell->point->values[0]->idx   = 0;
			pcell->point->values[0]->value = (double)atof(cell);
		}
		
		return pcell;
	}
	return NULL;	
}

static int
_gdl_fview_reader_continuous_check (gdl_fview_reader * reader, gdl_fsheet * fs)
{
	int status;
	char * cell, * row, * col;
	size_t ridx, cidx, first = 1;
	gdl_accession  * va;
	gdl_factor      * vl;
	gdl_fview_reader_cell  * pcell;
	
	gdl_fsheet_itr * itr = gdl_fsheet_iterator_c (fs);
	
	do 
	{
		row  = gdl_fsheet_iterator_row (itr);
		col  = gdl_fsheet_iterator_column (itr);
		cell = gdl_fsheet_iterator_cell (itr);
		ridx = gdl_fsheet_iterator_row_idx (itr);
		cidx = gdl_fsheet_iterator_column_idx (itr);
		
		if (gdl_fsheet_iterator_is_new_column (itr))
		{
			if (first)
			{
				first =	0;
			}
			
			vl = gdl_fview_reader_add_factor (reader, gdl_factor_continuous, col, cidx);
			
			if (vl->idx != cidx)
			{
				gdl_fview_reader_error (reader, "Factor %s is not unique", col);
				gdl_fsheet_iterator_free (itr);
				return (1);
			}
					
		}
		
		va = gdl_fview_reader_add_accession (reader, row, ridx);
		
		if (first && va->idx != ridx)
		{
			gdl_fview_reader_error (reader, "Accession %s is not unique", row);
			gdl_fsheet_iterator_free (itr);
			return (1);
		}
		
		pcell = gdl_fview_reader_continuous_parse_cell (reader, cell);
		
		if (pcell == NULL)
		{
			gdl_fview_reader_error (reader, "Invalid cell format [%s] at row [%s,%d] and column [%s,%d]", cell, row, ridx+1, col, cidx+1);
			gdl_fsheet_iterator_free (itr);
			return (1);	
		}
		
	}
	while (gdl_fsheet_iterator_next (itr));
	
	gdl_fsheet_iterator_free (itr);
	
	return GDL_SUCCESS;	
}

static int
_gdl_fview_reader_continuous_perform (gdl_fview_reader * reader, gdl_fsheet * fs)
{
	int status;
	char * cell, * row, * col;
	size_t ridx, cidx, first = 1;
	gdl_accession  * va;
	gdl_factor      * vl = NULL;
	gdl_fview_reader_cell  * pcell;
	
	gdl_fsheet_itr * itr = gdl_fsheet_iterator_c (fs);
	
	do 
	{
		row  = gdl_fsheet_iterator_row (itr);
		col  = gdl_fsheet_iterator_column (itr);
		cell = gdl_fsheet_iterator_cell (itr);
		ridx = gdl_fsheet_iterator_row_idx (itr);
		cidx = gdl_fsheet_iterator_column_idx (itr);
		
		if (gdl_fsheet_iterator_is_new_column (itr))
		{
			vl = gdl_fview_get_factor (reader->fview, cidx);
		}
		
		va = gdl_fview_get_accession (reader->fview, ridx);
		
		pcell = gdl_fview_reader_continuous_parse_cell (reader, cell);
		
		gdl_fview_reader_set_fdatapoint (reader, va, vl, pcell);
		
	}
	while (gdl_fsheet_iterator_next (itr));
	
	gdl_fsheet_iterator_free (itr);
	
	return GDL_SUCCESS;
}

static int
_gdl_fview_reader_continuous_parse (gdl_fview_reader * reader, gdl_fsheet * fs)
{
	int status;
	
	if (_gdl_fview_reader_continuous_init (reader, fs) != GDL_SUCCESS)
	{
		return GDL_EINVAL;
	}
	
	status = _gdl_fview_reader_continuous_check (reader, fs);
	
	if (status != GDL_SUCCESS)
	{
		gdl_fview_free (reader->fview);
		GDL_ERROR_VAL (reader->error, GDL_EINVAL, 0);
	}
	
	status = _gdl_fview_reader_continuous_perform (reader, fs);
	
	if (status != GDL_SUCCESS)
	{
		gdl_fview_free (reader->fview);
		GDL_ERROR_VAL (reader->error, GDL_EINVAL, 0);
	}
	
	return GDL_SUCCESS;
}

static int
_gdl_fview_reader_continuous_read (gdl_fview_reader * reader)
{
	int status;
	
	gdl_fsheet * fs = _gdl_fview_reader_continuous_fsheet (reader);
	
	if (fs == NULL)
	{
		return GDL_EINVAL;
	}
		
	status = _gdl_fview_reader_continuous_parse (reader, fs);
	
	gdl_fsheet_free (fs);
	
	return status;
}

static gdl_fview_reader_type _gdl_fview_reader_continuous = 
{
	"?",
	"\t",
	1,
	1,
	&_gdl_fview_reader_continuous_read
};

const gdl_fview_reader_type * gdl_fview_reader_continuous = &_gdl_fview_reader_continuous;
