/*  
 *  fview/reader.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:46 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_util.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_fsheet.h>
#include <gdl/gdl_fview.h>
#include <gdl/gdl_fview_reader.h>

static gdl_string *
_gdl_fview_reader_get_key (char * name, size_t idx)
{
	if (name)
		return gdl_string_clone (name);
	else
	{
		gdl_string * key = gdl_string_alloc (10);
		sprintf (key, "%d", idx);
		return key;
	}		
}

gdl_fview_reader *
gdl_fview_reader_alloc (gdl_fview_reader_type * type)
{
	gdl_fview_reader * r;
	
	r = GDL_CALLOC (gdl_fview_reader, 1);
	
	r->type = type;
		
	return r;
}

void
gdl_fview_reader_free (gdl_fview_reader * reader)
{
	if (reader)
	{
		gdl_string_free (reader->filename);
		gdl_string_free (reader->error);
		gdl_fview_reader_type_free (reader->type);
		GDL_FREE (reader);
	}
}

int
gdl_fview_reader_open (gdl_fview_reader * reader, 
                          const gdl_string * filename)
{
	if (reader && filename)
	{
		reader->stream = gdl_fileopen (filename, "r");
		if (reader->stream)
		{
			gdl_string_free (reader->filename);
			reader->filename = gdl_string_clone (filename);
			gdl_string_free (reader->error);
			reader->error = NULL;
			return GDL_SUCCESS;
		}
		else
			return GDL_EINVAL;
	}
	return GDL_EINVAL;
}

gdl_fview *
gdl_fview_reader_parse (gdl_fview_reader * reader)
{
	if (reader==0)
	{
		return NULL;
	}
	else if ((reader->type->read)(reader) == GDL_SUCCESS)
	{
		return reader->fview;
	}
	else
	{
		return NULL;
	}
}

int
gdl_fview_reader_close (gdl_fview_reader * reader)
{
	if (reader && reader->stream)
	{
		gdl_fileclose (reader->filename, reader->stream);
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}

int
gdl_fview_reader_error (gdl_fview_reader * reader, char * format, ...)
{
	if (reader == 0)
		return GDL_EINVAL;
	else
	{
		va_list ap; 
		
		va_start (ap, format);
		
		gdl_string_free (reader->error);
		
		reader->error = gdl_string_vsprintf (format, ap);
		
		va_end (ap);
	 	
	 	return GDL_SUCCESS;
	}
		
}

gdl_factor * 
gdl_fview_reader_add_factor (gdl_fview_reader * reader, const gdl_factor_type * T, char * name, size_t idx)
{
	gdl_string * key    = _gdl_fview_reader_get_key (name, idx);
	gdl_factor * factor = gdl_fview_add_factor (reader->fview, T, key);
	gdl_string_free (key);
	return factor;
}

gdl_accession *
gdl_fview_reader_add_accession (gdl_fview_reader * reader, char * name, size_t idx)
{
	gdl_string * key = _gdl_fview_reader_get_key (name, idx);
	gdl_accession * accession = gdl_fview_add_accession (reader->fview, key);
	gdl_string_free (key);
	return accession;
}

static int
gdl_fview_reader_update_level (gdl_factor * factor, const gdl_string * name)
{
	gdl_factor_level * level;
	  
	level = gdl_factor_level_new (name);
	
	gdl_factor_add (factor, &level, 1);
	
	return level->idx;
}

int
gdl_fview_reader_update_factor (gdl_factor * factor, const gdl_string * level)
{
	if (gdl_factor_get_type (factor) == gdl_factor_continuous)
	{
		return -1;
	}
	else 
	{
		return gdl_fview_reader_update_level (factor, level);
	}
}

int 
gdl_fview_reader_set_fdatapoint (gdl_fview_reader * reader, 
                                    gdl_accession * va,
                                    gdl_factor     * vl,
                                    gdl_fview_reader_cell * cell)
{
	gdl_fdatapoint * f = cell->point;
	if (f)
	{
		gdl_fview_set_fdatapoint (reader->fview, va, vl, f);
	}
}

int
gdl_fview_reader_split_mixture_column_name (gdl_string * col, gdl_string ** factor, gdl_string ** level)
{
	if (col)
	{
		size_t i, cl, fl, ll;
		
		fl = ll = 0;
		cl = strlen (col);
		
		for (i = 0; i < cl; i++)
		{
			if (col[i] == ':')
			{
				break;	
			}	
		}
		
		if (i == cl)
		{
			return GDL_FAILURE;
		}
		
		fl = i;
		ll = cl - fl - 1;
		
		*factor = gdl_string_alloc (fl);
		*level  = gdl_string_alloc (ll);
		
		memcpy (*factor, col, sizeof(char)*fl);
		memcpy (*level, &col[fl+1], sizeof(char)*ll);
		
		return GDL_SUCCESS;
	}
	
	return GDL_FAILURE;
}


