/*  
 *  fview/uninfo.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:46 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_fview.h>
#include <gdl/gdl_fview_mask.h>

static gdl_mask * gdl_fview_mask_not_informative_factor (const gdl_fview * g, gdl_mask * m);
static gdl_mask * gdl_fview_mask_not_informative_accession (const gdl_fview * g, gdl_mask * m);

gdl_mask *
gdl_fview_mask_not_informative (const gdl_fview * g,
                                gdl_mask * m,
                                const gdl_entity_type * type)
{
	if (type == GDL_FACTOR)
	{
		return gdl_fview_mask_not_informative_factor (g, m);
	}
	else if (type == GDL_ACCESSION)
	{
		return gdl_fview_mask_not_informative_accession (g, m);
	}
	else
	{
		return NULL;	
	}
}

static gdl_mask *
gdl_fview_mask_not_informative_factor (const gdl_fview * g, gdl_mask * m)
{
	size_t i;
	size_t nl;
	gdl_mask * u = NULL;
	
	nl = GDL_FVIEW_FACTOR_SIZE (g, m);
	
	for (i = 0; i < nl; i++)
	{
		gdl_factor * factor       = GDL_FVIEW_GET_FACTOR (g, m, i);
		const gdl_factor_type * T = gdl_factor_get_type (factor);
		size_t ng                 = gdl_factor_size (factor);
		
		if (T == gdl_factor_continuous || ng >= 2)
		{
			// keep it
			if (u == 0)
			{
				u = gdl_mask_alloc ();
				if (m != 0)
				{
					gdl_mask_set (u, GDL_ACCESSION, gdl_mask_get_clone (m, GDL_ACCESSION), 1);
				}
			}
			gdl_mask_add (u, factor);
		}
	}
	
	return u;
}

static gdl_mask *
gdl_fview_mask_not_informative_accession (const gdl_fview * g, gdl_mask * m)
{
	size_t i, j;
	size_t na, nl, ni;
	gdl_gvalues_get * gbuf;
	const gdl_gvalues * x;
	gdl_mask * u = NULL;
	
	na = GDL_FVIEW_ACCESSION_SIZE (g, m);
	nl = GDL_FVIEW_FACTOR_SIZE (g, m);
	
	gbuf = GDL_FVIEW_GET_NEW (g, m);
	
	for (i = 0; i < na; i++)
	{
		for (ni = j = 0; j < nl; j++)
		{
			GDL_FVIEW_GET_VALUE (g, m, i, j, gbuf);
			x = gdl_gvalues_get_gvalues (gbuf);
			if (x==0)
			{
				ni++;	
			}
		}
		if (ni < nl)
		{
			// keep it
			gdl_accession * a = GDL_FVIEW_GET_ACCESSION (g, m, i);
			if (u == 0)
			{
				u = gdl_mask_alloc ();
				if (m)
				{
					gdl_mask_set (u, GDL_FACTOR, gdl_mask_get_clone (m, GDL_FACTOR), 1);
				}
			}
			gdl_mask_add (u, a);
		}
	}
	
	gdl_gvalues_get_free (gbuf);
	
	return u;
}

gdl_mask *
gdl_fview_mask_not_informative_recursive (const gdl_fview * g)
{
	size_t nl1, na1, nl2, na2;
	gdl_mask * m1 = NULL, * m2;
	
	nl1 = gdl_fview_factor_size (g);
	
	do
	{
		m2  = gdl_fview_mask_not_informative_factor (g, m1);
		gdl_mask_free (m1);
		nl2 = gdl_mask_get_size (m2, GDL_FACTOR);
		if (nl2 == nl1)
		{
			m1=m2;
			break;
		}
		na2 = gdl_mask_get_size (m2, GDL_ACCESSION);
		m1  = gdl_fview_mask_not_informative_accession (g, m2);
		gdl_mask_free (m2);
		na1 = gdl_mask_get_size (m1, GDL_ACCESSION);
		if (na1 == na2)
		{
			break;
		}
		nl1 = gdl_mask_get_size (m1, GDL_FACTOR);
	} while (0==0);
	
	return m1;
}


