/*  
 *  fview/virtual.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:46 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_list.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_factor.h>
#include <gdl/gdl_accession.h>
#include <gdl/gdl_fview.h>
 
#include "ventry.c"

typedef struct
{
	size_t size;
	size_t na;
	size_t nf;
	gdl_hashtable * factor;
	gdl_hashtable * accession;
	gdl_hashtable * inputs;
	gdl_list      * inames;
	gdl_factor    ** _factor;
	gdl_accession ** _accession;
	gdl_string    * _gn;
	gdl_fview     * _g;
	gdl_factor    * _l;
	gdl_accession * _a;
	gdl_gvalues_get * _gb;
} gdl_fview_virtual_t;

static int
_gdl_fview_virtual_grow_factor (gdl_fview_virtual_t * v, size_t n)
{
	if (v->_factor == NULL)
	{
		v->_factor = GDL_MALLOC (gdl_factor *, n);
	}
	else
	{
		gdl_factor ** new = GDL_MALLOC (gdl_factor *, n);
		memcpy (new, v->_factor, sizeof(gdl_factor *)*v->nf);
		GDL_FREE (v->_factor);
		v->_factor = new;
	}
	v->nf = n;
	return (0);
}

static int
_gdl_fview_virtual_grow_accession (gdl_fview_virtual_t * v, size_t n)
{
	if (v->_accession == NULL)
	{
		v->_accession = GDL_MALLOC (gdl_accession *, n);
	}
	else
	{
		gdl_accession ** new = GDL_MALLOC (gdl_accession *, n);
		memcpy (new, v->_accession, sizeof(gdl_accession *)*v->na);
		GDL_FREE (v->_accession);
		v->_accession = new;
	}
	v->na = n;
	return (0);
}

typedef struct
{
	size_t * aidx;
	size_t * gidx;
} _gdl_fview_virtual_gcode;

static void
_gdl_fview_virtual_gcode_free (void * v)
{
	if (v)
	{
		_gdl_fview_virtual_gcode * vc = (_gdl_fview_virtual_gcode *)v;	
		GDL_FREE (vc->aidx);
		GDL_FREE (vc->gidx);
		GDL_FREE (vc);
	}
}

static gdl_data_interface _gdl_fview_virtual_gcode_interface = 
{
	&_gdl_fview_virtual_gcode_free,
	NULL,
	NULL,
	NULL,
	NULL
};

static void
_apply_gdl_fview_virtual_fcode_level (const gdl_factor * factor, gdl_string * name, gdl_gvalues_get * gb)
{
	gdl_hashtable * hash = (gdl_hashtable *) factor->extra;
	_gdl_fview_virtual_gcode * vgc = (_gdl_fview_virtual_gcode *) gdl_hashtable_lookup (hash, name);
	if (vgc)
	{
		size_t i;
		const gdl_gvalues * x = gdl_gvalues_get_gvalues (gb);
		for (i = 0; i < x->size; i++)
		{
			x->values[i]->idx = vgc->aidx[x->values[i]->idx];
		}
	}
}

static _gdl_fview_virtual_gcode *
gdl_fview_virtual_factor_update (gdl_factor * vfactor, const gdl_factor * factor, gdl_boolean * diff)
{
	size_t i, j, na;
	gdl_factor_level * a, * va;
	_gdl_fview_virtual_gcode * vgc = GDL_CALLOC (_gdl_fview_virtual_gcode, 1);
	
	na = gdl_factor_size (factor);
	
	if (na)
	{
		vgc->aidx = GDL_MALLOC (size_t, na);
		for (i = 0; i < na; i++)
		{
			a  = gdl_factor_get (factor, i);
			va = gdl_factor_search (vfactor, a->name);
			if ((va == 0) || (va && va->idx != a->idx))
			{
					*diff = gdl_true;
			}
			if (va == 0)
			{
				va = gdl_entity_clone (a);
				gdl_factor_add (vfactor, &va, 1);
			}
			vgc->aidx[a->idx] = va->idx;
		}
	}
	
	return vgc;
}

static int
_gdl_fview_virtual_add (gdl_fview_virtual_t * v, const gdl_string * name, gdl_fview * g)
{
	size_t n, i, j, gb_size;
	size_t * aidx;
	gdl_boolean diff;
	gdl_string * iname;
	gdl_hashtable * hash;
	gdl_factor * factor;
	gdl_accession * accession;
	gdl_fview_ventry * vfactor, * vac;
	_gdl_fview_virtual_gcode * vgc;
	
	iname = gdl_string_clone (name);
	
	gdl_list_push_front (v->inames, iname, 1);

	n = gdl_fview_factor_size (g);
	
	gb_size = (v->_gb) ? v->_gb->size : 0;
	
	for (i = 0; i < n; i++)
	{
		factor = gdl_fview_get_factor (g, i);
		vfactor  = (gdl_fview_ventry *) gdl_hashtable_lookup (v->factor, factor->name);
		if (vfactor == 0)
		{
			vfactor              = gdl_fview_ventry_alloc ();
			vfactor->entity      = gdl_entity_clone (factor);
			vfactor->entity->idx = gdl_hashtable_size (v->factor);
			_gdl_fview_virtual_grow_factor (v, vfactor->entity->idx+1);
			v->_factor[vfactor->entity->idx] = vfactor->entity;
			gdl_hashtable_add (v->factor, factor->name, vfactor, 1);
		}
		else if (gdl_factor_get_type (factor) != gdl_factor_get_type (vfactor->entity))
		{
			GDL_ERROR_VAL (gdl_string_sprintf ("Unable to merge factor %s : factor type differs!", factor->name),
			               GDL_FAILURE,
			               GDL_FAILURE);
		}
		else if (gdl_factor_get_type (factor) != gdl_factor_continuous)
		{
			diff = gdl_false;
			vgc  = gdl_fview_virtual_factor_update (vfactor->entity, factor, &diff);
			if (diff)
			{
				if (vfactor->entity->extra)
				{
					hash = (gdl_hashtable *) vfactor->entity->extra;
				}
				else
				{
					hash = gdl_hashtable_alloc (&_gdl_fview_virtual_gcode_interface, 0);
					vfactor->entity->extra = hash;
				}
				gdl_hashtable_add (hash, iname, vgc, 1);
			}
			else
			{
				_gdl_fview_virtual_gcode_free (vgc);
			}		
		}
		gdl_fview_ventry_grow (vfactor, vfactor->size+1);
		gdl_fview_ventry_set (vfactor, vfactor->size-1, iname);
		// Check factor size
		if (gdl_factor_size (vfactor->entity) > gb_size)
		{
			gb_size = gdl_factor_size (vfactor->entity);
		}
	}
	
	n = gdl_fview_accession_size (g);
	
	for (i = 0; i < n; i++)
	{
		accession = gdl_fview_get_accession (g, i);
		vac       = (gdl_fview_ventry *) gdl_hashtable_lookup (v->accession, accession->name);
		if (vac == 0)
		{
			vac = gdl_fview_ventry_alloc ();
			vac->entity      = gdl_entity_clone (accession);
			vac->entity->idx = gdl_hashtable_size (v->accession);
			_gdl_fview_virtual_grow_accession (v, vac->entity->idx+1);
			v->_accession[vac->entity->idx] = vac->entity;
			gdl_hashtable_add (v->accession, accession->name, vac, 1);
		}
		gdl_fview_ventry_grow (vac, vac->size+1);
		gdl_fview_ventry_set (vac, vac->size-1, iname);		
	}
	
	gdl_gvalues_get_free (v->_gb);
	v->_gb = gdl_gvalues_get_alloc (gb_size);
	
	v->size++;
	
	return GDL_SUCCESS;
}

static int 
gdl_fview_virtual_get_fview (gdl_fview_virtual_t * g, gdl_fview_ventry * va, gdl_fview_ventry * vl)
{
	size_t i, ii, j, jj;
	
	g->_g = NULL;
	
	for (i = vl->size; i > 0; i--)
	{
		ii = i - 1;
		for (j = va->size; j > 0; j--)
		{
			jj = j -1;
			if (vl->fviews[ii] == va->fviews[jj])
			{
				g->_gn = vl->fviews[ii];
				g->_g  = gdl_hashtable_lookup (g->inputs, g->_gn);
				return (0); 
			}
		}	
	}	
	return (1);
}

static int
gdl_fview_virtual_attach_fview (gdl_fview_virtual_t * g, const gdl_accession * a, const gdl_factor * l)
{
	gdl_fview_ventry * va = gdl_hashtable_lookup (g->accession, a->name);
	gdl_fview_ventry * vl = gdl_hashtable_lookup (g->factor, l->name);
	
	gdl_fview_virtual_get_fview (g, va, vl);
	
	if (g->_g)
	{
		g->_a = gdl_fview_search_accession (g->_g, a->name);
		g->_l = gdl_fview_search_factor (g->_g, l->name);
	}
	
	return GDL_SUCCESS;
}

static int
gdl_fview_virtual_attach_fview_f (gdl_fview_virtual_t * g, size_t a, size_t l)
{
	return gdl_fview_virtual_attach_fview (g, g->_accession[a], g->_factor[l]);
}

int
gdl_fview_virtual_alloc (void * vg)
{
	gdl_fview_virtual_t * g = (gdl_fview_virtual_t *) vg;
	
	g->na = g->nf = g->size = 0;
	g->_factor      = NULL;
	g->_accession  = NULL;
	g->_gb       = NULL;
	g->_g       = NULL;
	g->_gn      = NULL;
	g->_a      = NULL;
	g->_l      = NULL;
	
	g->factor     = gdl_hashtable_alloc (gdl_fview_ventry_interface, 0);
	g->accession = gdl_hashtable_alloc (gdl_fview_ventry_interface, 0);
	g->inputs    = gdl_hashtable_alloc (gdl_fview_interface, 0);
	g->inames    = gdl_list_alloc (gdl_list_default);

	return GDL_SUCCESS;
}

void
gdl_fview_virtual_free (void * vg)
{
	if (vg)
	{
		gdl_fview_virtual_t * g = (gdl_fview_virtual_t *) vg;
		
		gdl_gvalues_get_free (g->_gb);
		gdl_hashtable_free (g->factor);
		gdl_hashtable_free (g->accession);
		gdl_hashtable_free (g->inputs);
		gdl_list_free (g->inames);
		GDL_FREE (g->_factor);
		GDL_FREE (g->_accession);
	}	
}

size_t
gdl_fview_virtual_factor_size (const void * vg)
{
	return ((gdl_fview_virtual_t *)vg)->nf;
}

size_t
gdl_fview_virtual_accession_size (const void * vg)
{
	return ((gdl_fview_virtual_t *)vg)->na;
}

gdl_factor *
gdl_fview_virtual_add_factor (void * vg, const gdl_factor_type * T, const gdl_string * name)
{
	GDL_ERROR_VAL ("Cannot add a factor to a virtual fview",
	               GDL_EINVAL,
	               0);
}

gdl_accession *
gdl_fview_virtual_add_accession (void * vg, const gdl_string * name)
{
	GDL_ERROR_VAL ("Cannot add an accession to a virtual fview",
	               GDL_EINVAL,
	               0);
}

gdl_factor *
gdl_fview_virtual_get_factor (const void * vg, size_t idx)
{
	return ((gdl_fview_virtual_t *)vg)->_factor[idx];
}

gdl_accession *
gdl_fview_virtual_get_accession (const void * vg, size_t idx)
{
	return ((gdl_fview_virtual_t *)vg)->_accession[idx];
}

gdl_factor *
gdl_fview_virtual_search_factor (const void * vg, const char * name)
{
	gdl_fview_ventry * ve = (gdl_fview_ventry *) gdl_hashtable_lookup (((gdl_fview_virtual_t *)vg)->factor, name);
	if (ve)
	{
		return ve->entity;
	}
	return NULL;
}

gdl_accession *
gdl_fview_virtual_search_accession (const void * vg, const char * name)
{
	gdl_fview_ventry * ve = (gdl_fview_ventry *) gdl_hashtable_lookup (((gdl_fview_virtual_t *)vg)->accession, name);
	if (ve)
	{
		return ve->entity;
	}
	return NULL;
}

int
gdl_fview_virtual_set_fdatapoint (void * vg, const gdl_accession * a, const gdl_factor * l, gdl_fdatapoint * gd)
{
	gdl_fview_virtual_t * g = (gdl_fview_virtual_t *) vg;
	gdl_fview_virtual_attach_fview (g, a, l);
	return gdl_fview_set_fdatapoint (g->_g, g->_a, g->_l, gd);
}

gdl_gvalues_get *
gdl_fview_virtual_get_new (const void * gv)
{
	gdl_gvalues_get * get;
	size_t i, j, max=0;
	gdl_fview_virtual_t * g = (gdl_fview_virtual_t *) gv;
	
	for (i = 0; i < g->nf; i++)
	{
		gdl_factor * vl = gdl_fview_virtual_get_factor(g, i);
		if (gdl_factor_size (vl) > max)
		{
			max = gdl_factor_size (vl);
		}
	}
	return gdl_gvalues_get_alloc (max);
}

int
gdl_fview_virtual_get (const void * vg, const gdl_accession * a, const gdl_factor * l, gdl_gvalues_get * gd)
{
	gdl_fview_virtual_t * g = (gdl_fview_virtual_t *) vg;
	gdl_fview_virtual_attach_fview (g, a, l);
	if (g->_g)
	{
		gdl_fview_get_value (g->_g, g->_a, g->_l, gd);
		_apply_gdl_fview_virtual_fcode_level (l, g->_gn, gd);
	}
	else
	{
		gd->na = gdl_true;
	}
	return GDL_SUCCESS;	
}

gdl_boolean
gdl_fview_virtual_is_missing (const void * vg, const gdl_accession * a, const gdl_factor * l)
{
	gdl_fview_virtual_t * g = (gdl_fview_virtual_t *) vg;
	gdl_fview_virtual_attach_fview (g, a, l);
	if (g->_g)
	{
		return gdl_fview_is_missing (g->_g, g->_a, g->_l);
	}
	else
	{
		return gdl_true;	
	}
}

int
gdl_fview_virtual_set_fdatapoint_f (void * vg, size_t i, size_t j, gdl_fdatapoint * gd)
{
	gdl_fview_virtual_t * g = (gdl_fview_virtual_t *) vg;
	return gdl_fview_virtual_set_fdatapoint (g, g->_accession[i], g->_factor[j], gd);
}

int
gdl_fview_virtual_get_f (const void * vg, size_t i, size_t j, gdl_gvalues_get * gb)
{
	gdl_fview_virtual_t * g = (gdl_fview_virtual_t *) vg;
	return gdl_fview_virtual_get (g, g->_accession[i], g->_factor[j], gb);
}

gdl_boolean
gdl_fview_virtual_is_missing_f (const void * vg, size_t i, size_t j)
{
	gdl_fview_virtual_t * g = (gdl_fview_virtual_t *) vg;
	return gdl_fview_virtual_is_missing (g, g->_accession[i], g->_factor[j]);
}

void *
gdl_fview_virtual_fread (FILE * stream)
{
	if (stream == 0)
	{
		return NULL;
	}
	else
	{
		int status;
		gdl_fview_virtual_t * v;
		
		v = GDL_CALLOC (gdl_fview_virtual_t, 1);
		
		gdl_fview_virtual_alloc (v);
		
		status = gdl_hashtable_fread (stream, v->inputs);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		if (gdl_hashtable_size (v->inputs))
		{
			gdl_hashtable_itr * itr 
				= gdl_hashtable_iterator (v->inputs);
			do
			{
				gdl_string * gn = gdl_hashtable_iterator_key (itr);
				gdl_fview  * g  = (gdl_fview *) gdl_hashtable_iterator_value (itr);
				_gdl_fview_virtual_add (v, gn, g);
			}
			while (gdl_hashtable_iterator_next (itr));
			
			gdl_hashtable_iterator_free (itr);
		}
				
		return v;
	}	
}

int
gdl_fview_virtual_fwrite (FILE * stream, void * gv)
{
	if (stream == 0 || gv == 0)
	{
		return GDL_EINVAL;
	}
	else
	{
		int status;
		gdl_fview_virtual_t * v = (gdl_fview_virtual_t *) gv;
		
		status = gdl_hashtable_fwrite (stream, v->inputs);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}	
}

int
gdl_fview_virtual_add (void * vg, const gdl_string * name, void * s, size_t owner)
{
	gdl_fview_virtual_t * g = (gdl_fview_virtual_t *) vg;
	gdl_fview * gv = (gdl_fview *) s;
	_gdl_fview_virtual_add (g, name, gv);
	gdl_hashtable_add (g->inputs, name, gv, owner);
	
	return GDL_SUCCESS;
}

static const gdl_fview_type _virtual = 
{
	"gdl_fview_virtual",
	sizeof (gdl_fview_virtual_t),
	&gdl_fview_virtual_alloc,
	&gdl_fview_virtual_free,
	&gdl_fview_virtual_add_factor,
	&gdl_fview_virtual_add_accession,
	&gdl_fview_virtual_factor_size,
	&gdl_fview_virtual_accession_size,
	&gdl_fview_virtual_get_factor,
	&gdl_fview_virtual_get_accession,
	&gdl_fview_virtual_search_factor,
	&gdl_fview_virtual_search_accession,
	&gdl_fview_virtual_get_new,
	&gdl_fview_virtual_set_fdatapoint,
	&gdl_fview_virtual_get,
	&gdl_fview_virtual_is_missing,
	&gdl_fview_virtual_set_fdatapoint_f,
	&gdl_fview_virtual_get_f,
	&gdl_fview_virtual_is_missing_f,
	&gdl_fview_virtual_fread,
	&gdl_fview_virtual_fwrite,
	&gdl_fview_virtual_add
};

const gdl_fview_type * gdl_fview_virtual = &_virtual;
