/*  
 * 	gblock/freq.c 
 *  
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:04 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_frequency_block.h>

gdl_frequency_block *
gdl_frequency_block_alloc  (const size_t l, const size_t * na)
{
	size_t i;
	gdl_frequency_block * f;
	
	f = GDL_CALLOC (gdl_frequency_block, 1);
	f->l   = l;
	f->tna = GDL_MALLOC (size_t, l);
	f->tna[0]=0;
	for (i = 1; i < l; i++)
	{
		f->tna[i] = f->tna[i-1] + na[i-1];
	}
	f->t = f->tna[l-1] + na[l-1];
	f->data = GDL_CALLOC (double, f->t);
	
	return f;
}

void
gdl_frequency_block_free (gdl_frequency_block * f)
{
	if (f)
	{
		GDL_FREE (f->tna);
		GDL_FREE (f->data);
		GDL_FREE (f);
	}	
}

double
gdl_frequency_block_get (const gdl_frequency_block * f, size_t l, size_t a)
{
	return f->data[f->tna[l]+a];
}

void
gdl_frequency_block_set (const gdl_frequency_block * f, size_t l, size_t a, double x)
{
	f->data[f->tna[l]+a]=x;	
}

gdl_frequency_block *
gdl_frequency_block_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		gdl_frequency_block * b;
		
		b = GDL_MALLOC (gdl_frequency_block, 1);
		
		status=fread (&b->l, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status=fread (&b->t, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		b->tna = GDL_MALLOC (size_t, b->l);
		
		status=fread (b->tna, sizeof(size_t), b->l, stream);
		GDL_FREAD_STATUS (status, b->l);
		
		b->data = GDL_MALLOC (double, b->t);
		
		status=fread (b->data, sizeof(double), b->t, stream);
		GDL_FREAD_STATUS (status, b->t);
		
		return b;
	}
	else
	{
		return 0;	
	}
}

int
gdl_frequency_block_fwrite (FILE * stream, const gdl_frequency_block * b)
{
	if (stream && b)
	{
		int status;
		
		status=fwrite (&b->l, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status=fwrite (&b->t, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status=fwrite (b->tna, sizeof(size_t), b->l, stream);
		GDL_FWRITE_STATUS (status, b->l);
		status=fwrite (b->data, sizeof(double), b->t, stream);
		GDL_FWRITE_STATUS (status, b->t);
		
		return GDL_SUCCESS;
	}
	else
	{
		return GDL_EINVAL;	
	}	
}

gdl_frequencies_block *
gdl_frequencies_block_alloc  (const size_t k, const size_t l, const size_t * na)
{
	size_t i;
	gdl_frequencies_block * b;
	
	b = GDL_MALLOC (gdl_frequencies_block, 1);
	
	b->k      = k;
	b->blocks = GDL_CALLOC (gdl_frequency_block *, k);
	for (i = 0; i < k; i++)
	{
		b->blocks[i] = gdl_frequency_block_alloc (l, na);
	}
	
	return b;
}

void
gdl_frequencies_block_free (gdl_frequencies_block * b)
{
	if (b)
	{
		size_t i;
		for (i = 0; i < b->k; i++)
		{
			gdl_frequency_block_free (b->blocks[i]);
		}
		GDL_FREE (b->blocks);
		GDL_FREE (b);
	}	
}

double
gdl_frequencies_block_get (const gdl_frequencies_block * b, size_t k, size_t l, size_t a)
{
	return gdl_frequency_block_get (b->blocks[k], l, a);	
}

void
gdl_frequencies_block_set (const gdl_frequencies_block * b, size_t k, size_t l, size_t a, double x)
{
	gdl_frequency_block_set (b->blocks[k], l, a, x);
}

gdl_frequencies_block *
gdl_frequencies_block_fread (FILE * stream)
{
	if (stream)
	{
		size_t i;
		int status;
		gdl_frequencies_block * b;
	
		b = GDL_MALLOC (gdl_frequencies_block, 1);
	
		status = fread (&(b->k), sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		b->blocks = GDL_MALLOC (gdl_frequency_block *, b->k);
		for (i = 0; i < b->k; i++)
		{
			b->blocks[i] = gdl_frequency_block_fread (stream);
			GDL_FREAD_STATUS (b->blocks[i]!=0, 1);
		}
		
		return b;
	}
	else
	{
		return 0;	
	}	
}

int
gdl_frequencies_block_fwrite (FILE * stream, const gdl_frequencies_block * b)
{
	if (stream && b)
	{
		size_t i;
		int status;
		
		status = fwrite (&(b->k), sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < b->k; i++)
		{
			status = gdl_frequency_block_fwrite (stream, b->blocks[i]);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);
		}
		return GDL_SUCCESS;	
	}
	else
	{
		return GDL_EINVAL;	
	}	
}

