/*  
 * 	block/test.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:04 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_test.h>
#include <gdl/gdl_gblock.h>

void
test_block_init(void)
{
  gdl_block *b;
  size_t *dim;
  
  dim = GDL_MALLOC(size_t, 3);
  dim[0] = 1;
  dim[1] = 2;
  dim[2] = 3;

  b = gdl_block_alloc(3, dim);
  
  gdl_test (b->data == 0,     "gdl_block_alloc returns valid pointer");
  gdl_test (b->size != 1*2*3, "gdl_block_alloc returns valid size");
  
  b = gdl_block_alloc2(3, 1, 2, 3);
  
  gdl_test (b->data == 0,     "gdl_block_alloc2 returns valid pointer");
  gdl_test (b->size != 1*2*3, "gdl_block_alloc2 returns valid size");
  
  GDL_FREE(dim);
  gdl_block_free (b);       /* free whatever is in v */
}

void
test_block_value(void)
{
  gdl_block *b;
  int status;
  size_t i, j, k;
  
  b = gdl_block_alloc2(3, 5, 5, 5);
  
  status = 0;
  
  for (i = 0; i < 5; i++)
  {
  	for (j = 0; j < 5; j++)
  	{
  		for (k = 0; k < 5; k++)
  		{
  			double x = gdl_block_get(b, i, j, k);
  			status   = (x != 0);
  			gdl_block_set(b, i, j, k, (double) i+j+k);
  		}  		
  	}  
  }
  
  gdl_test (status, "gdl_block_get returns 0 values just after allocation");
  
  status = 0;
  
  for (i = 0; i < 5; i++)
  {
  	for (j = 0; j < 5; j++)
  	{
  		for (k = 0; k < 5; k++)
  		{
  		   double x = gdl_block_get(b, i, j, k);
  		   status = (x != (double)i+j+k);
  		}
  	}
  }
  
  gdl_test (status, "gdl_block_get returns correct values after setting");
  
  gdl_block_free (b);       /* free whatever is in v */
}

void
test_block_io(void)
{
  gdl_block *b,*b1;
  int status;
  size_t i, j, k;
  FILE *stream;
  
  b = gdl_block_alloc2(3, 5, 5, 5);
  
  status = 0;
  
  for (i = 0; i < 5; i++)
  {
  	for (j = 0; j < 5; j++)
  	{
  		for (k = 0; k < 5; k++)
  		{
  			gdl_block_set(b, i, j, k, (double) i+j+k);
  		}  		
  	}  
  }	
  
  stream = gdl_fileopen("test.dat", "w");
  
  status = gdl_block_fwrite(stream, b);
  
  gdl_test (status, "gdl_block_fwrite returns ok");  
  
  gdl_fileclose("test.dat", stream);
  
  stream = gdl_fileopen("test.dat", "r");
  
  b1 = gdl_block_fread(stream);
  
  gdl_fileclose("test.dat", stream);
  
  gdl_test (b1 == 0, "gdl_block_fread returns not null block");  
  
  status  = 1;
  status &= (b1->p == b->p);
  status &= (b1->size == b->size);
  
  gdl_test (!status, "gdl_block_fread returns correct block parameters"); 
  
}

void
test_nblock_init(void)
{
  gdl_nblock *b;
  size_t i, j, *dim, **nested;
  
  dim = GDL_MALLOC(size_t, 2);
  dim[0] = 5;
  dim[1] = 5;
  
  nested = GDL_MATRIX_ALLOC(size_t, 2, 5);
  
  for ( i = 0; i < 2 ; i++)
  {
  	  for ( j = 0; j < 5; j++)
  	  {
  	  	nested[i][j] = 2;
  	  }
  }

  b = gdl_nblock_alloc(2, dim, nested);
  
  gdl_test (b->data == 0,     "gdl_nblock_alloc returns valid pointer");
  gdl_test (b->size != 100,   "gdl_nblock_alloc returns valid size");
  
  GDL_FREE(dim);
  GDL_MATRIX_FREE(nested, 2);
  gdl_nblock_free (b);      
}

void
test_nblock_value(void)
{
  gdl_nblock *b;
  int status;
  size_t i, ii, j, jj, *dim, **nested;
  
  dim = GDL_MALLOC (size_t, 2);
  dim[0] = 5;
  dim[1] = 5;
  
  nested = GDL_MATRIX_ALLOC (size_t, 2, 5);
  
  for ( i = 0; i < 2 ; i++)
  {
  	  for ( j = 0; j < 5; j++)
  	  {
  	  	nested[i][j] = 2;
  	  }
  }

  b = gdl_nblock_alloc (2, dim, nested);
  
  status = 0;
  
  for ( i = 0; i < 5 ; i++)
  {
  	  for( ii = 0; ii < 2; ii++)
  	  {
	  	  for ( j = 0; j < 5; j++)
	  	  {
	  	  	for( jj = 0; jj < 2; jj++)
  	  		{
  	  			double x = gdl_nblock_get (b, i, ii, j, jj);
  	  			if (x != 0) status = 1;
  	  			x = (double) (i + ii + j + jj);
  	  			gdl_nblock_set (b, i, ii, j, jj, x);
  	  		}
	  	  }
  	  }
  }
  
  gdl_test (status, "gdl_nblock_get returns 0 values just after allocation");
  
  status = 0;
  
  for ( i = 0; i < 5 ; i++)
  {
  	  for( ii = 0; ii < 2; ii++)
  	  {
	  	  for ( j = 0; j < 5; j++)
	  	  {
	  	  	for( jj = 0; jj < 2; jj++)
  	  		{
  	  			double x = gdl_nblock_get (b, i, ii, j, jj);
  	  			if (x != (double) i + ii + j + jj) status = 1;
  	  		}
	  	  }
  	  }
  }
  
  gdl_test (status, "gdl_nblock_get returns correct values after setting");	
  
  GDL_FREE(dim);
  GDL_MATRIX_FREE(nested, 2);
  gdl_nblock_free (b);      
}

void
test_nblock_io(void)
{
  gdl_nblock * b, * b1;
  int status;
  size_t i, ii, j, jj, *dim, **nested;
  FILE *stream;
  
  dim = GDL_MALLOC (size_t, 2);
  dim[0] = 5;
  dim[1] = 5;
  
  nested = GDL_MATRIX_ALLOC (size_t, 2, 5);
  
  for ( i = 0; i < 2 ; i++)
  {
  	  for ( j = 0; j < 5; j++)
  	  {
  	  	nested[i][j] = 2;
  	  }
  }

  b = gdl_nblock_alloc (2, dim, nested);
  
  status = 0;
  
  for ( i = 0; i < 5 ; i++)
  {
  	  for( ii = 0; ii < 2; ii++)
  	  {
	  	  for ( j = 0; j < 5; j++)
	  	  {
	  	  	for( jj = 0; jj < 2; jj++)
  	  		{
  	  			double x = gdl_nblock_get (b, i, ii, j, jj);
  	  			if (x != 0) status = 1;
  	  			x = (double) (i + ii + j + jj);
  	  			gdl_nblock_set (b, i, ii, j, jj, x);
  	  		}
	  	  }
  	  }
  }
  
  stream = gdl_fileopen("test.dat", "w");
  
  status = gdl_nblock_fwrite(stream, b);
  
  gdl_test (status, "gdl_nblock_fwrite returns ok");  
  
  gdl_fileclose("test.dat", stream);
  
  stream = gdl_fileopen("test.dat", "r");
  
  b1 = gdl_nblock_fread(stream);
  
  gdl_fileclose("test.dat", stream);
  
  gdl_test (b1 == 0, "gdl_nblock_fread returns valid pointer");  
  
  status  = 1;
  status &= (b1->p == b->p);
  status &= (b1->size == b->size);
  
  gdl_test (!status, "gdl_nblock_fread returns correct block parameters"); 
  
  GDL_FREE(dim);
  GDL_MATRIX_FREE(nested, 2);
  gdl_nblock_free (b); 
}

void
test_hnblock_init(void)
{
  gdl_hnblock *b;
  size_t i, j, *dim, **nested;
  
  dim = GDL_MALLOC(size_t, 4);
  for ( i = 0; i < 4 ; i++)
  {
  	dim[i] = 5;
  }
  
  nested = GDL_MATRIX_ALLOC(size_t, 2, 5);
  
  for ( i = 0; i < 2 ; i++)
  {
  	  for ( j = 0; j < 5; j++)
  	  {
  	  	nested[i][j] = 2;
  	  }
  }

  b = gdl_hnblock_alloc(2, 2, dim, nested);
   
  gdl_test (b->data == 0,      "gdl_hnblock_alloc returns valid pointer");
  gdl_test (b->size != 2500,   "gdl_hnblock_alloc returns valid size");
  
  GDL_FREE(dim);
  GDL_MATRIX_FREE(nested, 2);
  gdl_hnblock_free (b);      
}

void
test_hnblock_value(void)
{
  gdl_hnblock *b;
  int status;
  size_t i, ii, j, jj, k, kk, l, ll, *dim, **nested;
  
  dim = GDL_MALLOC(size_t, 4);
  for ( i = 0; i < 4 ; i++)
  {
  	dim[i] = 5;
  }
  
  nested = GDL_MATRIX_ALLOC(size_t, 2, 5);
  
  for ( i = 0; i < 2 ; i++)
  {
  	  for ( j = 0; j < 5; j++)
  	  {
  	  	nested[i][j] = 2;
  	  }
  }

  b = gdl_hnblock_alloc(2, 2, dim, nested);
  status = 0;
  
  for ( i = 0; i < 5 ; i++)
  {
  	  for( ii = 0; ii < 5; ii++)
  	  {
	  	  for ( j = 0; j < 5; j++)
	  	  {
	  	  	for( jj = 0; jj < 2; jj++)
  	  		{
  	  			double x = gdl_hnblock_get (b, i, ii, j, jj, 0, 0);
  	  			if (x != 0) status = 1;
  	  			x = (double) (i + ii + j + jj);
  	  			gdl_hnblock_set (b, i, ii, j, jj, 0, 0, x);
  	  		}
	  	  }
  	  }
  }
  
  gdl_test (status, "gdl_hnblock_get returns 0 values just after allocation");
  
  status = 0;
  
  for ( i = 0; i < 5 ; i++)
  {
  	  for( ii = 0; ii < 5; ii++)
  	  {
	  	  for ( j = 0; j < 5; j++)
	  	  {
	  	  	for( jj = 0; jj < 2; jj++)
  	  		{
  	  			double x = gdl_hnblock_get (b, i, ii, j, jj, 0, 0);
  	  			if (x != (double) i + ii + j + jj) status = 1;
  	  		}
	  	  }
  	  }
  }
  
  gdl_test (status, "gdl_hnblock_get returns correct values after setting");	
  
  GDL_FREE(dim);
  GDL_MATRIX_FREE(nested, 2);
  gdl_hnblock_free (b);      
}

void
test_hnblock_io(void)
{
  gdl_hnblock * b, * b1;
  int status;
  size_t i, ii, j, jj, k, kk, l, ll, *dim, **nested;
  FILE *stream;
  
  dim = GDL_MALLOC(size_t, 4);
  for ( i = 0; i < 4 ; i++)
  {
  	dim[i] = 5;
  }
  
  nested = GDL_MATRIX_ALLOC(size_t, 2, 5);
  
  for ( i = 0; i < 2 ; i++)
  {
  	  for ( j = 0; j < 5; j++)
  	  {
  	  	nested[i][j] = 2;
  	  }
  }

  b = gdl_hnblock_alloc(2, 2, dim, nested);
  status = 0;
  
  for ( i = 0; i < 5 ; i++)
  {
  	  for( ii = 0; ii < 5; ii++)
  	  {
	  	  for ( j = 0; j < 5; j++)
	  	  {
	  	  	for( jj = 0; jj < 2; jj++)
  	  		{
  	  			double x = gdl_hnblock_get (b, i, ii, j, jj, 0, 0);
  	  			if (x != 0) status = 1;
  	  			x = (double) (i + ii + j + jj);
  	  			gdl_hnblock_set (b, i, ii, j, jj, 0, 0, x);
  	  		}
	  	  }
  	  }
  } 
   
  stream = gdl_fileopen("test.dat", "w");
  
  status = gdl_hnblock_fwrite(stream, b);
  
  gdl_test (status, "gdl_hnblock_fwrite returns ok");  
  
  gdl_fileclose("test.dat", stream);
  
  stream = gdl_fileopen("test.dat", "r");
  
  b1 = gdl_hnblock_fread(stream);
  
  gdl_fileclose("test.dat", stream);
  
  gdl_test (b1 == 0, "gdl_hnblock_fread returns valid pointer");  
  
  status  = 1;
  status &= (b1->p == b->p);
  status &= (b1->size == b->size);
  
  gdl_test (!status, "gdl_hnblock_fread returns correct block parameters"); 
  
  GDL_FREE(dim);
  GDL_MATRIX_FREE(nested, 2);
  gdl_hnblock_free (b);
}

int
main (void)
{
	test_block_init();
	test_block_value();
	test_block_io();
	test_nblock_init();
	test_nblock_value();
	test_nblock_io();
	test_hnblock_init();
	test_hnblock_value();
	test_hnblock_io();
	exit(gdl_test_summary());
}

