/* gcoord/chunk.c
 * 
 * Copyright (C) 2008 Jean-Baptiste Veyrieras
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
 
#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_list.h> 
#include <gdl/gdl_hash.h>
#include <gdl/gdl_math.h> 
#include <gdl/gdl_genome_coord.h>

gdl_chunk_coord *
gdl_chunk_coord_alloc ()
{
	gdl_chunk_coord * c;
	
	c = GDL_CALLOC (gdl_chunk_coord, 1);
	
	return c;	
}

void
gdl_chunk_coord_free (gdl_chunk_coord * c)
{
	if (c)
	{
		GDL_FREE (c);
	}	
}

gdl_boolean
gdl_chunk_coord_is_inversion (const gdl_chunk_coord * c)
{
	return (c->strand == '+') ? gdl_false : gdl_true;	
}

gdl_boolean
gdl_chunk_coord_is_insertion (const gdl_chunk_coord * c)
{
	return (c->type == 'i') ? gdl_true : gdl_false;	
}

gdl_boolean
gdl_chunk_coord_is_deletion (const gdl_chunk_coord * c)
{
	return (c->type == 'd') ? gdl_true : gdl_false;
}

gdl_boolean
gdl_chunk_coord_is_translocation (const gdl_chunk_coord * c)
{
	return (c->relation == 't') ? gdl_true : gdl_false;	
}

gdl_chunk_coord *
gdl_chunk_coord_sscanf (const gdl_string * line, const size_t n, gdl_dictionary * seqid_dico)
{  
	gdl_string * tok;
	size_t i=0,j=0;
	gdl_chunk_coord * c = gdl_chunk_coord_alloc ();
	                    
	// ref_id ref_start ref_end relation strand type qry_id qry_strand qry_start qry_end
	tok = gdl_string_next_token (line, n, &i, &j);
	c->ref_id = gdl_dictionary_populate (seqid_dico, "ref", tok);
	gdl_string_free (tok);
	tok = gdl_string_next_token (line, n, &i, &j);
	c->ref_start = atol(tok);
	gdl_string_free (tok);
	tok = gdl_string_next_token (line, n, &i, &j);
	c->ref_end = atol(tok);
	gdl_string_free (tok);
	tok = gdl_string_next_token (line, n, &i, &j);
	c->relation = tok[0];
	gdl_string_free (tok);
	tok = gdl_string_next_token (line, n, &i, &j);
	c->strand = tok[0];
	gdl_string_free (tok);
	tok = gdl_string_next_token (line, n, &i, &j);
	c->type = tok[0];
	gdl_string_free (tok);
	tok = gdl_string_next_token (line, n, &i, &j);
	c->qry_id = gdl_dictionary_populate (seqid_dico, "qry", tok);
	gdl_string_free (tok);
	tok = gdl_string_next_token (line, n, &i, &j);
	c->qry_strand = atoi(tok);
	gdl_string_free (tok);
	tok = gdl_string_next_token (line, n, &i, &j);
	c->qry_start = atol(tok);
	gdl_string_free (tok);
	tok = gdl_string_next_token (line, n, &i, &j);
	c->qry_end = atol(tok);
	gdl_string_free (tok);
	
	if (c->qry_start > c->qry_end)
	{
		long tmp = c->qry_start;
		c->qry_start = c->qry_end;
		c->qry_end   = tmp;
	}
	//gdl_chunk_coord_fprintf (stdout, c, seqid_dico);
	
	return c;
}

int
gdl_chunk_coord_fprintf (FILE * stream, const gdl_chunk_coord * c, const gdl_dictionary * dico)
{
	fprintf(stream, "%s %d %d %c %c %c %s %d %d %d\n", gdl_dictionary_get (dico, "ref", c->ref_id), c->ref_start, c->ref_end, c->relation, c->strand, c->type, gdl_dictionary_get (dico, "qry", c->qry_id), c->qry_strand, c->qry_start, c->qry_end);   
}

size_t
gdl_chunk_coord_size (const gdl_chunk_coord * c, const gdl_string * which)
{
	if (!strcmp (which, "ref"))
		return c->ref_end-c->ref_start+1;
	else 
		return c->qry_end-c->qry_start+1;
}

