/* gcoord/gdl_genome_coord.h
 * 
 * Copyright (C) 2008 Jean-Baptiste Veyrieras
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef __GDL_GENOME_COORD_H__
#define __GDL_GENOME_COORD_H__

#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_list.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_dictionary.h>
#include <gdl/gdl_gfeatures.h>
#include <gdl/gdl_gfeatures_gff3.h>

__BEGIN_DECLS

typedef struct
{
	size_t ref_id;
	size_t qry_id;
	long ref_start;
	long ref_end;
	long qry_start;
	long qry_end;
	int qry_strand;	
	unsigned char relation;
	unsigned char strand;
	unsigned char type;
} gdl_chunk_coord;

gdl_chunk_coord * gdl_chunk_coord_alloc ();
void gdl_chunk_coord_free (gdl_chunk_coord * c);

gdl_chunk_coord * gdl_chunk_coord_sscanf (const gdl_string * line, const size_t n, gdl_dictionary * seqid_dico);
int gdl_chunk_coord_fprintf (FILE * stream, const gdl_chunk_coord * c, const gdl_dictionary * dico);

size_t gdl_chunk_coord_size (const gdl_chunk_coord * c, const gdl_string * which); 
gdl_boolean gdl_chunk_coord_is_inversion (const gdl_chunk_coord * c);
gdl_boolean gdl_chunk_coord_is_insertion (const gdl_chunk_coord * c);
gdl_boolean gdl_chunk_coord_is_deletion (const gdl_chunk_coord * c);
gdl_boolean gdl_chunk_coord_is_translocation (const gdl_chunk_coord * c);

typedef struct
{
	const gdl_string * seq_id;
	gdl_boolean is_ref;
	long start;
	long end;
	size_t nchunk;
	gdl_chunk_coord ** chunks;
	const gdl_dictionary  * seqid_dico;
} gdl_chunk_coord_cluster;

gdl_chunk_coord_cluster * gdl_chunk_coord_cluster_alloc (const gdl_string * seq_id, const gdl_string * which);
void gdl_chunk_coord_cluster_free (gdl_chunk_coord_cluster * c, void (*free)(gdl_chunk_coord * c));
int gdl_chunk_coord_cluster_add (gdl_chunk_coord_cluster * cluster, gdl_chunk_coord * chunk);
void gdl_chunk_coord_cluster_sort (gdl_chunk_coord_cluster * cluster);
int gdl_chunk_coord_cluster_goto (const gdl_chunk_coord_cluster * cluster, const long position, const size_t start_idx);
int gdl_chunk_coord_cluster_fprintf (FILE * stream, const gdl_chunk_coord_cluster * c, gdl_dictionary * dico);
gdl_boolean gdl_chunk_coord_cluster_has_inversion (const gdl_chunk_coord_cluster * cluster, const size_t from, const size_t to);
gdl_boolean gdl_chunk_coord_cluster_has_translocation (const gdl_chunk_coord_cluster * cluster, const size_t from, const size_t to);
gdl_boolean gdl_chunk_coord_cluster_has_rearrangement (const gdl_chunk_coord_cluster * cluster, const size_t from, const size_t to);
gdl_boolean gdl_chunk_coord_cluster_is_inversion (const gdl_chunk_coord_cluster * cluster, const size_t from, const size_t to);
gdl_boolean gdl_chunk_coord_cluster_is_translocation (const gdl_chunk_coord_cluster * cluster, const size_t from, const size_t to);

long gdl_chunk_coord_cluster_map (const gdl_chunk_coord_cluster * cluster, long from_position, int start_chunk);
void gdl_chunk_coord_cluster_indel_size (const gdl_chunk_coord_cluster * chunk_cluster, const size_t chunk_start, const size_t chunk_end, const long region_start, const long region_end, long * del_size, long * ins_size);
gdl_gfeatures_gff3_record * gdl_chunk_coord_cluster_map_gff3_record (const gdl_chunk_coord_cluster * cluster, const gdl_gfeatures_gff3_record * rec, int chunk_start, int chunk_end);

typedef struct
{
	size_t nchunk;
	size_t nref;
	size_t nqry;
	gdl_dictionary  * seqid_dico;
	gdl_chunk_coord_cluster ** ref_chunks;
	gdl_chunk_coord_cluster ** qry_chunks;
	gdl_chunk_coord ** chunks;
} gdl_genome_coord;

gdl_genome_coord * gdl_genome_coord_alloc (const gdl_string * file);
void gdl_genome_coord_free (gdl_genome_coord * gc);
gdl_chunk_coord_cluster * gdl_genome_coord_lookup (const gdl_genome_coord * gc, const gdl_string * which, const gdl_string * seq_id);
gdl_chunk_coord_cluster * gdl_genome_coord_get (const gdl_genome_coord * gc, const gdl_string * which, const size_t seq_idx);

typedef struct 
{
	const gdl_string * ref_seq_id;
	const gdl_string * qry_seq_id;
	long ref_start;
	long ref_end;
	long qry_start;
	long qry_end;
	int  qry_strand;
	gdl_boolean is_continuous;
} gdl_genome_coord_info;
										
void gdl_chunk_coord_cluster_get_info (gdl_chunk_coord_cluster * chunk_cluster, const size_t from, const size_t to, gdl_genome_coord_info * info);

__END_DECLS

#endif
