/* gdl_math.h
 * 
 * Copyright (C) 1996, 1997, 1998, 1999, 2000, 2004 Gerard Jungman, Brian Gough
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __GDL_MATH_H__
#define __GDL_MATH_H__

#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_sys.h>
#include <gdl/gdl_nan.h>
#include <gdl/gdl_pow_int.h>

#ifndef M_E
#define M_E        2.71828182845904523536028747135      /* e */
#endif

#ifndef M_LOG2E
#define M_LOG2E    1.44269504088896340735992468100      /* log_2 (e) */
#endif

#ifndef M_LOG10E
#define M_LOG10E   0.43429448190325182765112891892      /* log_10 (e) */
#endif

#ifndef M_SQRT2
#define M_SQRT2    1.41421356237309504880168872421      /* sqrt(2) */
#endif

#ifndef M_SQRT1_2
#define M_SQRT1_2  0.70710678118654752440084436210      /* sqrt(1/2) */
#endif


#ifndef M_SQRT3
#define M_SQRT3    1.73205080756887729352744634151      /* sqrt(3) */
#endif

#ifndef M_PI
#define M_PI       3.14159265358979323846264338328      /* pi */
#endif

#ifndef M_PI_2
#define M_PI_2     1.57079632679489661923132169164      /* pi/2 */
#endif

#ifndef M_PI_4
#define M_PI_4     0.78539816339744830966156608458      /* pi/4 */
#endif

#ifndef M_SQRTPI
#define M_SQRTPI   1.77245385090551602729816748334      /* sqrt(pi) */
#endif

#ifndef M_2_SQRTPI
#define M_2_SQRTPI 1.12837916709551257389615890312      /* 2/sqrt(pi) */
#endif

#ifndef M_1_PI
#define M_1_PI     0.31830988618379067153776752675      /* 1/pi */
#endif

#ifndef M_2_PI
#define M_2_PI     0.63661977236758134307553505349      /* 2/pi */
#endif

#ifndef M_LN10
#define M_LN10     2.30258509299404568401799145468      /* ln(10) */
#endif

#ifndef M_LN2
#define M_LN2      0.69314718055994530941723212146      /* ln(2) */
#endif

#ifndef M_LNPI
#define M_LNPI     1.14472988584940017414342735135      /* ln(pi) */
#endif

#ifndef M_EULER
#define M_EULER    0.57721566490153286060651209008      /* Euler constant */
#endif

__BEGIN_DECLS

/* other needlessly compulsive abstractions */

#define GDL_IS_ODD(n)  ((n) & 1)
#define GDL_IS_EVEN(n) (!(GDL_IS_ODD(n)))
#define GDL_SIGN(x)    ((x) >= 0.0 ? 1 : -1)

/* Return nonzero if x is a real number, i.e. non NaN or infinite. */
#define GDL_IS_REAL(x) (gdl_finite(x))

/* Define MAX and MIN macros/functions if they don't exist. */

/* plain old macros for general use */
#define GDL_MAX(a,b) ((a) > (b) ? (a) : (b))
#define GDL_MIN(a,b) ((a) < (b) ? (a) : (b))
#define GDL_SQR(a)   ((a) == 0.0 ? 0.0 : (a)*(a))

/* function versions of the above, in case they are needed */
double gdl_max (double a, double b);
double gdl_min (double a, double b);

/* inline-friendly strongly typed versions */
#ifdef HAVE_INLINE

extern inline int GDL_MAX_INT (int a, int b);
extern inline int GDL_MIN_INT (int a, int b);
extern inline int GDL_SQR_INT (int a);
extern inline double GDL_MAX_DBL (double a, double b);
extern inline double GDL_MIN_DBL (double a, double b);
extern inline double GDL_SQR_DBL (double a);
extern inline long double GDL_MAX_LDBL (long double a, long double b);
extern inline long double GDL_MIN_LDBL (long double a, long double b);

extern inline int
GDL_MAX_INT (int a, int b)
{
  return GDL_MAX (a, b);
}

extern inline int
GDL_MIN_INT (int a, int b)
{
  return GDL_MIN (a, b);
}

extern inline int
GDL_SQR_INT (int a)
{
  return GDL_SQR(a);
}

extern inline double
GDL_MAX_DBL (double a, double b)
{
  return GDL_MAX (a, b);
}

extern inline double
GDL_MIN_DBL (double a, double b)
{
  return GDL_MIN (a, b);
}

extern inline double
GDL_SQR_DBL (double a)
{
  return GDL_SQR (a);
}

extern inline long double
GDL_MAX_LDBL (long double a, long double b)
{
  return GDL_MAX (a, b);
}

extern inline long double
GDL_MIN_LDBL (long double a, long double b)
{
  return GDL_MIN (a, b);
}
#else
#define GDL_MAX_INT(a,b)   GDL_MAX(a,b)
#define GDL_MIN_INT(a,b)   GDL_MIN(a,b)
#define GDL_SQR_INT(a)     GDL_SQR(a)
#define GDL_MAX_DBL(a,b)   GDL_MAX(a,b)
#define GDL_MIN_DBL(a,b)   GDL_MIN(a,b)
#define GDL_SQR_DBL(a)     GDL_SQR(a)
#define GDL_MAX_LDBL(a,b)  GDL_MAX(a,b)
#define GDL_MIN_LDBL(a,b)  GDL_MIN(a,b)
#endif /* HAVE_INLINE */

/* Definition of an arbitrary function with parameters */

struct gdl_function_struct 
{
  double (* function) (double x, void * params);
  void * params;
};

typedef struct gdl_function_struct gdl_function ;

#define GDL_FN_EVAL(F,x) (*((F)->function))(x,(F)->params)

/* Definition of an arbitrary function returning two values, r1, r2 */

struct gdl_function_fdf_struct 
{
  double (* f) (double x, void * params);
  double (* df) (double x, void * params);
  void (* fdf) (double x, void * params, double * f, double * df);
  void * params;
};

typedef struct gdl_function_fdf_struct gdl_function_fdf ;

#define GDL_FN_FDF_EVAL_F(FDF,x) (*((FDF)->f))(x,(FDF)->params)
#define GDL_FN_FDF_EVAL_DF(FDF,x) (*((FDF)->df))(x,(FDF)->params)
#define GDL_FN_FDF_EVAL_F_DF(FDF,x,y,dy) (*((FDF)->fdf))(x,(FDF)->params,(y),(dy))


/* Definition of an arbitrary vector-valued function with parameters */

struct gdl_function_vec_struct 
{
  int (* function) (double x, double y[], void * params);
  void * params;
};

typedef struct gdl_function_vec_struct gdl_function_vec ;

#define GDL_FN_VEC_EVAL(F,x,y) (*((F)->function))(x,y,(F)->params)


__END_DECLS

#endif /* __GDL_MATH_H__ */
