/*
 *  genex/block.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:52 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_vector_uint.h>
#include <gdl/gdl_genex_block.h>

gdl_genex_block *
gdl_genex_block_alloc (const long start, const long end, const unsigned char strand)
{
	gdl_genex_block * b;
	
	b = GDL_CALLOC (gdl_genex_block, 1);
	
	b->start = start;
	b->end = end;
	b->strand = strand;
	b->ignore = 'n';
	
	return b;
}

void
gdl_genex_block_free (gdl_genex_block * b)
{
	if (b)
	{
		if (b->record)
		{
			size_t i;
			for (i = 0; i < b->size; i++)
			{
				gdl_list_free (b->record[i]);
			}
			GDL_FREE (b->record);
		}
		GDL_FREE (b->probes);
		GDL_FREE (b->pval);
		GDL_FREE (b);
	}
}

void
gdl_genex_block_add_probe (gdl_genex_block * b, gdl_genex_probe * probe)
{
	if (b->size)
	{
		gdl_genex_probe ** tmp = GDL_MALLOC (gdl_genex_probe *, b->size+1);
		memcpy (tmp, b->probes, sizeof(gdl_genex_probe *)*b->size);
		GDL_FREE (b->probes);
		b->probes = tmp;
		b->size++;
	}
	else
	{
		b->probes = GDL_MALLOC (gdl_genex_probe *, 1);
		b->size=1;
	}
	b->probes[b->size-1]=probe;
}

void
gdl_genex_block_rm_all_probe (gdl_genex_block * b)
{
	if (b->size)
	{
		b->size=0;
		GDL_FREE (b->probes);
	}
}

void
gdl_genex_block_init_permut_storage (gdl_genex_block * b)
{
	size_t n = b->snp_down-b->snp_up+1;
	if (n && !b->pval)
	{
		b->pval   = GDL_MATRIX_ALLOC (double, b->size, n);
		b->permut = GDL_MATRIX_ALLOC (size_t, b->size, n);
	}	
}

void
gdl_genex_block_clean_permut_storage (gdl_genex_block * b)
{
	GDL_MATRIX_FREE (b->pval, b->size);
	GDL_MATRIX_FREE (b->permut, b->size);	
}

void
gdl_genex_block_set_obs_pval (gdl_genex_block * b, size_t probe, size_t snp, double pval)
{
	b->pval[probe][snp-b->snp_up] = pval;
} 

double
gdl_genex_block_get_obs_pval (gdl_genex_block * b, size_t probe, size_t snp)
{
	return b->pval[probe][snp-b->snp_up];
}

void
gdl_genex_block_set_permut_pval (gdl_genex_block * b, size_t probe, size_t snp, double pval)
{
	if (pval < b->pval[probe][snp-b->snp_up])
	{
		b->permut[probe][snp-b->snp_up]++;
	}
}

double
gdl_genex_block_get_permut_pval (gdl_genex_block * b, size_t probe, size_t snp, size_t size)
{
	return ((double)b->permut[probe][snp-b->snp_up])/size;
}

void
gdl_genex_annot_dico_add_count (const gdl_genex_block * b, const gdl_snp * snp, gdl_snp_annot_dico * dico)
{
	size_t l;
	gdl_snp_annot * annot = snp->annot;
	
	if (annot)
	{
		if (snp->position >= b->start && snp->position <= b->end)
		{
			if (annot->gene)
			{
				dico->func[annot->gene->func_id]->count++;
				dico->count++;
			}
			if (annot->prot)
			{
				for (l = 0; l < annot->prot->size; l++)
				{
					dico->prot[annot->prot->ids[l]]->count++;
					dico->count++;
				}
			}
		}
		else
		{
			if (annot->geno)
			{
				for (l = 0; l < annot->geno->size; l++)
				{
					dico->geno[annot->geno->ids[l]]->count++;
					dico->count++;
				}
			}
			if (annot->tran)
			{
				for (l = 0; l < annot->tran->size; l++)
				{
					dico->tran[annot->tran->ids[l]]->count++;
					dico->count++;
				}
			}			
		}
	}
	else
	{
		dico->unknown->count++;
		dico->count++;
	}
}

void
gdl_genex_block_record_snp (gdl_genex_block * block, const size_t probe, const size_t snp)
{
	if (!block->record)
	{
		block->record = GDL_CALLOC (gdl_list *, block->size);
		block->record[probe] = gdl_list_alloc (gdl_interface_uint);
	}
	size_t * r = GDL_MALLOC (size_t, 1);
	*r = snp;
	gdl_list_push_back (block->record[probe], r, 1);
}

void
gdl_genex_block_clean_record_snp (gdl_genex_block * b)
{
	if (b->record)
	{
		size_t i;
		for (i = 0; i < b->size; i++)
		{
			gdl_list_free (b->record[i]);
		}
		GDL_FREE (b->record);
		b->record=0;
	}
}

gdl_vector_uint *
gdl_genex_block_get_record_snp (gdl_genex_block * block, const size_t probe)
{
	gdl_vector_uint * snps=0;
	
	if (block->record && block->record[probe])
	{
		size_t i=0;
		gdl_list_itr * itr;
		
		snps = gdl_vector_uint_alloc (gdl_list_size (block->record[probe]));
		itr = gdl_list_iterator_front (block->record[probe]);
		do
		{
			size_t * r = (size_t *) gdl_list_iterator_value (itr);
			gdl_vector_uint_set (snps, i++, *r);
		}
		while (gdl_list_iterator_next (itr));
		gdl_list_iterator_free (itr);		
	}
	
	return snps;
}

gdl_genex_block *
gdl_genex_block_fread (FILE * stream, gdl_genex_probe ** probes)
{
	if (stream && probes)
	{
		size_t i, * pidx, n;
		long start, end;
		unsigned char has, strand;
		int status;
		gdl_genex_block * b;
		
		status = fread (&start, sizeof(long), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&end, sizeof(long), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&strand, sizeof(unsigned char), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		b = gdl_genex_block_alloc (start, end, strand);
		
		status = fread (&b->size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		pidx = GDL_MALLOC (size_t, b->size);
		
		status = fread (pidx, sizeof(size_t), b->size, stream);
		GDL_FREAD_STATUS (status, b->size);
		
		b->probes = GDL_MALLOC (gdl_genex_probe *, b->size);
		
		for (i = 0; i < b->size; i++)
		{
			b->probes[i] = probes[pidx[i]];
		}
		
		GDL_FREE (pidx);
		
		status = fread (&b->snp_start, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&b->snp_end, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&b->snp_up, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&b->snp_down, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&b->ignore, sizeof(unsigned char), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		status = fread (&has, sizeof(unsigned char), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		if (has == 'y')
		{
			b->record = GDL_CALLOC (gdl_list *, b->size);
			for (i = 0; i < b->size; i++)
			{
				status = fread (&n, sizeof(size_t), 1, stream);
				GDL_FREAD_STATUS (status, 1);
				if (n)
				{
					b->record[i] = gdl_list_alloc (gdl_interface_uint);
					status = gdl_list_fread (stream, b->record[i]);
					GDL_FREAD_STATUS (status, GDL_SUCCESS);
				}
			}
		}
		
		return b;
	}
	return 0;
}

int
gdl_genex_block_fwrite (FILE * stream, const gdl_genex_block * b)
{
	if (stream && b)
	{
		int status;
		unsigned char has;
		size_t i, n, * pidx;
		
		status = fwrite (&b->start, sizeof(long), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&b->end, sizeof(long), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&b->strand, sizeof(unsigned char), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&b->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		pidx = GDL_MALLOC (size_t, b->size);
		for (i = 0; i < b->size; i++)
		{
			pidx[i] = b->probes[i]->idx;
		}
		status = fwrite (pidx, sizeof(size_t), b->size, stream);
		GDL_FWRITE_STATUS (status, b->size);
		
		GDL_FREE (pidx);
		
		status = fwrite (&b->snp_start, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&b->snp_end, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&b->snp_up, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&b->snp_down, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&b->ignore, sizeof(unsigned char), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		has = (b->record) ? 'y' : 'n';
		status = fwrite (&has, sizeof(unsigned char), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		if (has == 'y')
		{
			for (i = 0; i< b->size; i++)
			{
				n = (b->record[i]) ? gdl_list_size (b->record[i]) : 0;
				status = fwrite (&n, sizeof(size_t), 1, stream);
				GDL_FWRITE_STATUS (status, 1);
				if (n)
				{
					status = gdl_list_fwrite (stream, b->record[i]);
					GDL_FWRITE_STATUS (status, GDL_SUCCESS);
				}
			}
		}
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;	
}
