/*  
 * 	entity/accession.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:03 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_entity.h>
#include <gdl/gdl_accession.h>

enum _gdl_accession_gmode
{
  gdl_accession_mode_unknown,
  gdl_accession_mode_unique,
  gdl_accession_mode_pooled
};

typedef struct 
{
	size_t              size;              // size of the pool
	gdl_boolean         is_phased;         // phase status
} _gdl_accession;

gdl_accession *
gdl_accession_alloc (void)
{
	return gdl_entity_alloc (GDL_ACCESSION);
}

gdl_accession *
gdl_accession_new (const char * name)
{
	gdl_accession * a = gdl_entity_alloc (GDL_ACCESSION);
	gdl_entity_set_name (a, name);
	return a;
}

void
gdl_accession_free (gdl_accession * g)
{
	gdl_entity_free (g);
}

size_t
gdl_accession_get_size (gdl_accession * g)
{
	return (((_gdl_accession *)g->state)->size);
}

void
gdl_accession_set_size (gdl_accession * g, size_t size)
{
	((_gdl_accession *)g->state)->size = size;
}

gdl_boolean
gdl_accession_is_phased (gdl_accession * g)
{
	return ((_gdl_accession *)g->state)->is_phased;	
}

void
gdl_accession_set_phased (gdl_accession * g, gdl_boolean status)
{
	((_gdl_accession *)g->state)->is_phased = status;	
}



int
_gdl_accession_alloc (void * vstate)
{
	if (vstate == 0)
		return GDL_EINVAL;
	else
	{
		_gdl_accession * ga = (_gdl_accession *) vstate;
		ga->size      = 0;
		ga->is_phased = gdl_false;
	}
	return GDL_SUCCESS;
}

int
_gdl_accession_compare (const void * vstate1, const void * vstate2)
{
	if (vstate1 == 0 || vstate2 == 0)
		return (-1);
	if (vstate1 == vstate2)
		return (0);
	else
	{
		_gdl_accession * g1 = (_gdl_accession *) vstate1;
		_gdl_accession * g2 = (_gdl_accession *) vstate2;
		
		if (g1->size != g2->size)
			return (-1);
		if (g1->is_phased != g2->is_phased)
			return (-1);
			
		return (0);
	}
}

int
_gdl_accession_copy (void * vstate1, const void * vstate2)
{
	if (vstate1 == 0 || vstate2 == 0)
		return GDL_EINVAL;
	else
	{
		_gdl_accession * g1 = (_gdl_accession *) vstate1;
		_gdl_accession * g2 = (_gdl_accession *) vstate2;
		
		g1->size = g2->size;
		g1->is_phased = g2->is_phased;
	}
	
	return GDL_SUCCESS;
}

int
_gdl_accession_fread (FILE * stream, void * vstate)
{
	if (stream == 0 || vstate == 0)
		return GDL_EINVAL;
	else
	{
		size_t status;
		
		_gdl_accession * g = (_gdl_accession *) vstate;
		
		status = fread (&g->size, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&g->is_phased, sizeof (gdl_boolean), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		return GDL_SUCCESS;	
	}
}

int
_gdl_accession_fwrite (FILE * stream, const void * vstate)
{
	if (stream == 0 || vstate == 0)
		return GDL_EINVAL;
	else
	{
		size_t status;
		
		_gdl_accession * g = (_gdl_accession *) vstate;
		
		status = fwrite (&g->size, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&g->is_phased, sizeof (gdl_boolean), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		return GDL_SUCCESS;	
	}
}

static const gdl_entity_type _gdl_accession_type =
{
	sizeof(_gdl_accession),
	"A",
	&_gdl_accession_alloc,
	NULL,
	&_gdl_accession_compare,
	&_gdl_accession_copy,
	&_gdl_accession_fread,
	&_gdl_accession_fwrite
};

static const gdl_accession_gmode  _unknown  = gdl_accession_mode_unknown;
static const gdl_accession_gmode  _unique   = gdl_accession_mode_unique;
static const gdl_accession_gmode  _pooled   = gdl_accession_mode_pooled;

const gdl_entity_type  * GDL_ACCESSION         = &_gdl_accession_type;
const gdl_accession_gmode * gdl_accession_unknown = &_unknown;
const gdl_accession_gmode * gdl_accession_unique  = &_unique;
const gdl_accession_gmode * gdl_accession_pooled  = &_pooled;
