/*  
 * 	entity/chrom.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:04 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_locus.h>
#include <gdl/gdl_chromosome.h>

typedef struct
{
	gdl_locus * locus;
	size_t idx;
	size_t owner;
} _gdl_chromosome_entry;

static void
_gdl_chromosome_entry_free (_gdl_chromosome_entry * e)
{
	if (e->owner)
		gdl_locus_free (e->locus);
	GDL_FREE (e);
}

#define INDEX(n, i, j)((i)*((n)-1)-(i)*((i)-1)/2+(j)-(i)-1)

typedef struct 
{
	gdl_hashtable          * table;
	_gdl_chromosome_entry  ** entries;
	gdl_gdistance         *_gd;
	double     * dists;
	size_t     * types;
	size_t     size;
	size_t     _size;
} _gdl_chromosome;

static int
_gdl_chromosome_alloc (void * vg)
{
	_gdl_chromosome * g = (_gdl_chromosome *) vg;
	
	g->size    = 0;
	g->_size    = 0;
	g->dists   = NULL;
	g->types   = NULL;
	g->entries = NULL;
	g->table   = NULL;
	g->_gd     = NULL;
	
	return GDL_SUCCESS;
}

int
_gdl_chromosome_grow (_gdl_chromosome * c, size_t size)
{
	size_t i;
	
	if (c->_size && c->_size < size)
	{
		size_t j, * nt;
		double * nd;
		_gdl_chromosome_entry ** ne;
		
//		nt = GDL_CALLOC (size_t, size*(size-1)/2);
//		nd = GDL_CALLOC (double, size*(size-1)/2);
		nt = GDL_CALLOC (size_t, size);
		nd = GDL_CALLOC (double, size);
		
		for (i = 0; i < c->_size; i++)
		{
//			for (j = i + 1; j < c->_size; j++)
//			{
//				nt[INDEX(size, i, j)] = c->types[INDEX(c->_size, i, j)];
//				nd[INDEX(size, i, j)] = c->dists[INDEX(c->_size, i, j)];
//			}
			nt[i] = c->types[i];
			nd[i] = c->dists[i];
		}
		
		ne = GDL_CALLOC (_gdl_chromosome_entry *, size);
		
		memcpy (ne, c->entries, sizeof(_gdl_chromosome_entry *)*c->_size); 
		
		for (i = c->_size; i < size; i++)
		{
			ne[i]        = GDL_CALLOC (_gdl_chromosome_entry, 1);
			ne[i]->idx   = i;
			ne[i]->owner = 0;
		}
		
		GDL_FREE (c->types);
		c->types = nt;
		GDL_FREE (c->dists);
		c->dists = nd;
		GDL_FREE (c->entries);
		c->entries = ne;
	}
	else
	{
//		c->types   = GDL_CALLOC (size_t, size*(size-1)/2);
//		c->dists   = GDL_CALLOC (double, size*(size-1)/2);
		c->types   = GDL_CALLOC (size_t, size);
		c->dists   = GDL_CALLOC (double, size);
		c->entries = GDL_CALLOC (_gdl_chromosome_entry *, size);
		for (i = 0; i < size; i++)
		{
			c->entries[i] = GDL_CALLOC (_gdl_chromosome_entry, 1);
			c->entries[i]->idx = i;
			c->entries[i]->owner = 0;
		}
		c->table   = gdl_hashtable_alloc (gdl_hash_default, 0);
		c->_gd     = GDL_CALLOC (gdl_gdistance, 1);
	}
	
	c->_size = size;
	
	return GDL_SUCCESS;
}

static void
_gdl_chromosome_free (void * vg)
{
	if (vg)
	{
		_gdl_chromosome * g = (_gdl_chromosome *) vg;
		
		if (g->size)
		{
			size_t i;
			GDL_FREE (g->dists);
			g->dists = NULL;
			GDL_FREE (g->types);
			g->types = NULL;
			gdl_hashtable_free (g->table);
			g->table   = NULL;
			for (i = 0; i < g->_size; i++)
			{
				_gdl_chromosome_entry_free (g->entries[i]);
			}
			GDL_FREE (g->entries);
			g->entries = NULL;		
			GDL_FREE (g->_gd);
			g->size    = 0;	
			g->_size   = 0;
		}
	}
}

static int
_gdl_chromosome_copy (void * vdest, const void * vsrc)
{
	if (vdest == 0 || vsrc == 0)
		return (-1);
	else
	{
		size_t i;
		
		_gdl_chromosome * dest = (_gdl_chromosome *) vdest;
		_gdl_chromosome * src  = (_gdl_chromosome *) vsrc;
		
		_gdl_chromosome_free (dest);
		
		_gdl_chromosome_grow (dest, src->size);
		dest->size = src->size;
		
		memcpy (dest->dists, src->dists, sizeof(double)*dest->size);
		memcpy (dest->types, src->types, sizeof(size_t)*dest->size);
		
		for (i = 0; i < dest->size; i++)
		{
			dest->entries[i]->owner = src->entries[i]->owner;
			
			if (src->entries[i]->owner)
			{
				dest->entries[i]->locus = gdl_entity_clone (src->entries[i]->locus);
			}
			else
			{
				dest->entries[i]->locus = src->entries[i]->locus;
			}
			
			gdl_hashtable_add (dest->table,
			                      dest->entries[i]->locus->name,
			                      dest->entries[i],
			                      0);
		}
		
		return GDL_SUCCESS;
	}	
}

static int
_gdl_chromosome_fread (FILE * stream, void * vg)
{
	if (stream == 0 || vg == 0)
		return GDL_EINVAL;
	else
	{
		size_t size, status;
		_gdl_chromosome * g = (_gdl_chromosome *) vg;
		
		_gdl_chromosome_free (g);
		
		status = fread (&size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		if (size)
		{
			size_t i;
			
			_gdl_chromosome_grow (g, size);
			
			g->size = size;
			
//			status = fread (g->dists, sizeof(double), g->size*(g->size-1)/2, stream);
//			GDL_FREAD_STATUS (status, g->size*(g->size-1)/2);
//			status = fread (g->types, sizeof(size_t), g->size*(g->size-1)/2, stream);
//			GDL_FREAD_STATUS (status, g->size*(g->size-1)/2);

			status = fread (g->dists, sizeof(double), g->size, stream);
			GDL_FREAD_STATUS (status, g->size);
			status = fread (g->types, sizeof(size_t), g->size, stream);
			GDL_FREAD_STATUS (status, g->size);
			
			for (i = 0; i < g->size; i++)
			{
				g->entries[i]->owner = 1;
				g->entries[i]->locus = gdl_entity_fread (stream);
				GDL_FWRITE_STATUS (g->entries[i]->locus == 0, 0);
				gdl_hashtable_add (g->table, g->entries[i]->locus->name, g->entries[i], 0);
			}
			
		}
		return GDL_SUCCESS;
	}
}

static int
_gdl_chromosome_fwrite (FILE * stream, const void * vg)
{
	if (stream == 0 || vg == 0)
		return GDL_EINVAL;
	else
	{
		size_t status;
		_gdl_chromosome * g = (_gdl_chromosome *) vg;
		
		status = fwrite (&g->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		if (g->size)
		{
			size_t i;
			
//			status = fwrite (g->dists, sizeof(double), g->size*(g->size-1)/2, stream);
//			GDL_FWRITE_STATUS (status, g->size*(g->size-1)/2);
//			status = fwrite (g->types, sizeof(size_t), g->size*(g->size-1)/2, stream);
//			GDL_FWRITE_STATUS (status, g->size*(g->size-1)/2);
			
			status = fwrite (g->dists, sizeof(double), g->size, stream);
			GDL_FWRITE_STATUS (status, g->size);
			status = fwrite (g->types, sizeof(size_t), g->size, stream);
			GDL_FWRITE_STATUS (status, g->size);
			
			for (i = 0; i < g->size; i++)
			{
				status = gdl_entity_fwrite (stream, g->entries[i]->locus);
				GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			}		  
		}
				
		return GDL_SUCCESS;	
	}
}

static void
_gdl_chromosome_add_distance (_gdl_chromosome * c, size_t i, const gdl_gdistance * d)
{
	gdl_gdistance * cd;
	
	cd = gdl_gdistance_convert (d, gdl_gdistance_morgan);

	if (!cd)
	{
		cd = gdl_gdistance_convert (d, gdl_gdistance_kilo_base);
		if  (cd)
		{
			c->dists[i] = cd->value;
			c->types[i] = gdl_gdistance_kilo_base;
		}
		gdl_gdistance_free (cd);
	}
	else
	{
		c->dists[i] = cd->value;
		c->types[i] = gdl_gdistance_morgan;
		gdl_gdistance_free (cd);
	}
	
//	if (mode)
//	{
//		for (ii = 0; ii < i; ii++)
//		{
//			jj = INDEX(c->_size, ii, i);
//			if (c->types[jj] == c->types[idx])
//			{
//				x = c->dists[jj];
//				jj = INDEX(c->_size, ii, j);
//				c->dists[jj]  = x + d->value;
//				c->types[jj] = d->type;
//			}
//		}
//	}
}                                      

gdl_chromosome *
gdl_chromosome_alloc (void)
{
	gdl_chromosome * ce = gdl_entity_alloc (GDL_CHROMOSOME);
	return ce;
}

gdl_chromosome *
gdl_chromosome_new (const char * name)
{
	gdl_chromosome * ce = gdl_chromosome_alloc ();
	gdl_entity_set_name (ce, name);
	return ce;
}

void
gdl_chromosome_free (gdl_chromosome * g)
{
	gdl_entity_free (g);
}

int
gdl_chromosome_init (gdl_chromosome * vc, size_t size)
{
	_gdl_chromosome * c = (_gdl_chromosome *) vc->state;
	return _gdl_chromosome_grow (c, size);
}

int
gdl_chromosome_push (gdl_chromosome * g, gdl_locus * l, size_t owner)
{
	_gdl_chromosome * c = (_gdl_chromosome *) g->state;
	
	if (c->_size < c->size + 1)
	{
		_gdl_chromosome_grow (c, c->_size+1);
	}
	
	c->size++;
	
	c->entries[c->size-1]->owner = owner;
	
	c->entries[c->size-1]->locus = l;
		
	gdl_hashtable_add (c->table, l->name, c->entries[c->size-1], 0);	
	
	return c->size-1;
}

int
gdl_chromosome_add (gdl_chromosome * g, gdl_locus * l, const gdl_gdistance * d, size_t owner)
{
	int i;
	
	_gdl_chromosome * c = (_gdl_chromosome *) g->state;
	
	i = gdl_chromosome_push (g, l, owner);
	
	if (c->size > 1)
	{
		_gdl_chromosome_add_distance (c, c->size-2, d);
	}
	
	return i;
}

size_t
gdl_chromosome_size (const gdl_chromosome * g)
{
	return ((_gdl_chromosome *)g->state)->size;
}

gdl_locus *
gdl_chromosome_get (const gdl_chromosome * g, size_t i)
{
	return ((_gdl_chromosome *)g->state)->entries[i]->locus;
}

int
gdl_chromosome_search (const gdl_chromosome * g, const gdl_locus * l)
{
	_gdl_chromosome * c = (_gdl_chromosome *) g->state;
	
	_gdl_chromosome_entry * e 
		= (_gdl_chromosome_entry *)gdl_hashtable_lookup (c->table, l->name);
	
	if (e)
		return e->idx;
	else
		return -1;
}

gdl_gdistance *
gdl_chromosome_get_distance (const gdl_chromosome * g,
                            size_t i,
                            size_t j)
{
   size_t ii = GDL_MIN(i,j);
   size_t jj = GDL_MAX(i,j);
   size_t kk;
   gdl_gdistance * d;
   
   _gdl_chromosome * c = (_gdl_chromosome *) g->state;
   
   if (ii != jj)
   {
		d = gdl_gdistance_alloc (c->types[ii]);
		
	   for (kk = ii; kk < jj; kk++)
	   {
	   	  if (c->types[kk] == c->types[kk+1] || kk == c->size-2)
	   	  {
	   	  	   d->value += c->dists[kk];
	   	  }
	   	  else
	   	  {
	   	  	 gdl_gdistance_free (d);
	   	  	 return NULL;
	   	  }
	   }
	   
	   return d;
   }
   
   return NULL;
}                            

gdl_gdistance *
gdl_chromosome_search_distance (const gdl_chromosome * g,
                            const gdl_locus * f,
                            const gdl_locus * t)
{
   size_t i  = gdl_chromosome_search (g, f);
   size_t j  = gdl_chromosome_search (g, t);
   return gdl_chromosome_get_distance (g, i, j);
}

gdl_locus_mask *
gdl_chromosome_get_mask (const gdl_chromosome * vc, const gdl_locus_mask * mask)
{
	size_t i;
	_gdl_chromosome * c = (_gdl_chromosome *) vc->state;
	gdl_entity_mask * m = gdl_entity_mask_new (c->size);
	
	for (i = 0; i < c->size; i++)
	{
		gdl_locus * l = c->entries[i]->locus;
		gdl_entity_mask_add (m, l);
	}
	
	if (mask)
	{
		gdl_entity_mask_inter (m, mask);
	}
	
	return m;
}

int
gdl_chromosome_fprintf (FILE * stream, const gdl_chromosome * vc)
{
	if (vc == 0 || stream == 0)
	{
		return GDL_EINVAL;
	}
	else
	{
		size_t i;
		const gdl_gdistance * d;
		_gdl_chromosome * c = (_gdl_chromosome *) vc->state;
		
		for (i = 0; i < c->size; i++)
		{
			fprintf (stream, "%s\t%d\t%s", vc->name, c->entries[i]->idx, c->entries[i]->locus->name);
			if (i < c->size - 1)
			{
				 
				d = gdl_chromosome_get_distance (vc, i, i+1);
				if (d->type)
				{
					fprintf (stream, "\t%g", d->value);
					d = gdl_chromosome_get_distance (vc, 0, i);
					fprintf (stream, "\t%g\n", (i>0) ? d->value : 0.0);
				}
				else
				{
					fprintf (stream, "\t-\t-\n");
				}
			}
			else
			{
				d = gdl_chromosome_get_distance (vc, 0, i);
				fprintf (stream, "\t-\t%g\n", d->value);
			}		
		}
		
		return GDL_SUCCESS;
	}
}

static const gdl_entity_type _gdl_chromosome_type =
{
	sizeof (_gdl_chromosome),
	"C",
	&_gdl_chromosome_alloc,
	&_gdl_chromosome_free,
	NULL,
	&_gdl_chromosome_copy,
    &_gdl_chromosome_fread,
    &_gdl_chromosome_fwrite
};

const gdl_entity_type * GDL_CHROMOSOME = &_gdl_chromosome_type;

#undef INDEX
