/*  
 * 	entity/dist.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:03 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_locus.h>
#include <gdl/gdl_chromosome.h>

gdl_gdistance *
gdl_gdistance_alloc (const gdl_gdistance_type type)
{
	gdl_gdistance * d;
	
	d = GDL_MALLOC (gdl_gdistance, 1);
	
	d->type  = type;
	d->value = 0;
	
	return d;
}

void
gdl_gdistance_free (gdl_gdistance * d)
{
	GDL_FREE (d);
}

gdl_gdistance_type
gdl_gdistance_type_parse (const gdl_string * string)
{
	if (!strcmp (string, "M"))
	{
		return gdl_gdistance_morgan;
	}
	else if (!strcmp (string, "cM"))
	{
		return gdl_gdistance_centi_morgan;
	}
	else if (!strcmp (string, "bp"))
	{
		return gdl_gdistance_base;
	}
	else if (!strcmp (string, "kbp"))
	{
		return gdl_gdistance_kilo_base;
	}
	else if (!strcmp (string, "-"))
	{
		return gdl_gdistance_undefined;
	}
	
	return gdl_gdistance_unknown;
}

int
gdl_gdistance_fprintf (FILE * stream, const gdl_gdistance * d)
{
	if ((d && d->type != gdl_gdistance_unknown)
	    && d->type != gdl_gdistance_undefined)
	{
		fprintf (stream, "%g\t", d->value);
		switch (d->type)
		{
			case gdl_gdistance_morgan :
				return fprintf (stream, "M");
			case gdl_gdistance_centi_morgan :
				return fprintf (stream, "cM");
			case gdl_gdistance_base :
				return fprintf (stream, "bp");
			case gdl_gdistance_kilo_base :
				return fprintf (stream, "kbp");
			default :
				return fprintf (stream, "?");
		}
	}
	else
	{
		fprintf (stream, "-\t-");
	}	
}

gdl_gdistance *
gdl_gdistance_convert (const gdl_gdistance * d, const gdl_gdistance_type type)
{
	if (d && type)
	{
		gdl_gdistance * c = GDL_MALLOC (gdl_gdistance, 1);
		
		c->type = type;
		
		switch (d->type)
		{
			case gdl_gdistance_morgan :
				switch (type)
				{
					case gdl_gdistance_morgan :
						c->value = d->value;
						return c;
					case gdl_gdistance_centi_morgan :
						c->value = d->value*100;
						return c;
					default :
						GDL_FREE (c);
						return NULL; 
				}
			case gdl_gdistance_centi_morgan :
				switch (type)
				{
					case gdl_gdistance_morgan :
						c->value = d->value/100;
						return c;
					case gdl_gdistance_centi_morgan :
						c->value = d->value;
						return c;
					default :
						GDL_FREE (c);
						return NULL; 
				}
			case gdl_gdistance_base :
				switch (type)
				{
					case gdl_gdistance_base :
						c->value = d->value;
						return c;
					case gdl_gdistance_kilo_base :
						c->value = d->value/1000;
						return c;
					default :
						GDL_FREE (c);
						return NULL; 
				}
			case gdl_gdistance_kilo_base :
				switch (type)
				{
					case gdl_gdistance_base :
						c->value = d->value*1000;
						return c;
					case gdl_gdistance_kilo_base :
						c->value = d->value;
						return c;
					default :
						GDL_FREE (c);
						return NULL; 
				}
		}
		
		return NULL;
	}
	
	return NULL;	
}

gdl_gdistance *
gdl_gdistance_fscanf (FILE * stream)
{
	if (stream)
	{
		size_t n;
		gdl_string * line = NULL;
		gdl_gdistance * d;
		
		gdl_getline (stream, &line, &n);
		
		if (n)
		{
			d = gdl_gdistance_sscanf (line);
			gdl_string_free (line);
			return d;
		}
		
		return NULL;
	}
	
	return NULL;
}

gdl_gdistance *
gdl_gdistance_sscanf (const gdl_string * string)
{
	if (string)
	{
		size_t i, j, k, n;
		gdl_string * buf;
		gdl_gdistance_type type=gdl_gdistance_unknown;
		gdl_gdistance * d;
		
		n = strlen (string);
		
		for (j = GDL_MIN (3, n); j >= 1; j--)
		{ 
			type = gdl_gdistance_type_parse (&string[n-j]);
			if (type!=gdl_gdistance_unknown)
			{
				break;
			}
		}
		if (type == gdl_gdistance_unknown)
		{
			return NULL;
		}
		else
		{
			for (k = 0; k < n-j && isspace(string[k]); k++);
			if (k == n-j)
			{
				return NULL;	
			}
			for (i = 0; k+i < n-j && !isspace(string[k+i]); i++);
			buf = gdl_string_alloc (i+1);
			strncpy (buf, &string[k], i+1);
			d   = gdl_gdistance_alloc (type);
			d->value = (double) atof (buf);
			gdl_string_free (buf);
		}
		
		return d;
	}	
}

gdl_boolean
gdl_gdistance_is_defined (const gdl_gdistance * d)
{
	return (d->type==gdl_gdistance_unknown || d->type==gdl_gdistance_undefined) ? gdl_false : gdl_true;
}
