/*  
 * 	entity/factor.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:04 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_list.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_factor.h>
#include <gdl/gdl_factor_level.h>

enum _gdl_factor_type
{
	_gdl_factor_categorial,
	_gdl_factor_mixture,
	_gdl_factor_continuous
};

typedef struct
{
	size_t _nl;
	gdl_hashtable        * levels;
	gdl_factor_level   ** _levels;
	const gdl_factor_type  * type;
} _gdl_factor;

static int
_gdl_factor_levels_grow (_gdl_factor * vl, size_t n)
{
	if (vl->_levels == NULL)
	{
		vl->_levels = GDL_MALLOC (gdl_factor_level *, n);
	}
	else
	{
		gdl_factor_level ** new = GDL_MALLOC (gdl_factor_level *, n);
		memcpy (new, vl->_levels, sizeof(gdl_factor_level *)*vl->_nl);
		GDL_FREE (vl->_levels);
		vl->_levels = new;
	}
	vl->_nl = n;
}

static int
_gdl_factor_alloc (void * vfactor)
{
	_gdl_factor * va = (_gdl_factor *) vfactor;
	
	if (va)
	{
		va->levels = gdl_hashtable_alloc (gdl_entity_interface, 0);
		va->_levels = NULL;		
		va->_nl = 0;
	}
	
	return GDL_SUCCESS;
}

static void
_gdl_factor_free (void * vfactor)
{
	if (vfactor)
	{
		_gdl_factor * va = (_gdl_factor *) vfactor;
		gdl_hashtable_free (va->levels);
		GDL_FREE (va->_levels);
	}
}

static int
_gdl_factor_copy (void * vdest, const void * vsrc)
{
	if (vdest == 0 || vsrc == 0)
		return GDL_EINVAL;
	else
	{
		size_t i;
		
		_gdl_factor * dest = (_gdl_factor *) vdest;
		_gdl_factor * src = (_gdl_factor *) vsrc;
		
		_gdl_factor_free (dest);
		_gdl_factor_alloc (dest);
		
		dest->type = src->type;
		
		if (src->_nl)
		{
			_gdl_factor_levels_grow (dest, src->_nl);
			for (i = 0; i < dest->_nl; i++)
			{
				dest->_levels[i] = gdl_entity_clone (src->_levels[i]);
				gdl_hashtable_add (dest->levels, dest->_levels[i]->name, dest->_levels[i], 1);			
			}
		}
		else
		{
			dest->_nl = 0;	
		}
		
		return GDL_SUCCESS;
	}
}

static int
_gdl_factor_compare (const void * v1, const void * v2)
{
	if (v1 == 0 || v2 == 0)
	{
		return GDL_EINVAL;
	}
	else if (v1 == v2)
	{
		return 0;
	}
	else
	{
		_gdl_factor * l1 = (_gdl_factor *) v1;
		_gdl_factor * l2 = (_gdl_factor *) v2;
		
		// TODO
		
		return 0;
	}
}

const gdl_factor_type *
gdl_factor_type_fread (FILE * stream)
{
	size_t type;
	
	int status = fread (&type, sizeof(size_t), 1, stream);
	GDL_FREAD_STATUS (status, 1);
	
	if (type == _gdl_factor_categorial)
	{
		return gdl_factor_categorial;
	}
	else if (type == _gdl_factor_mixture)
	{
		return gdl_factor_mixture;	
	}
	else if (type == _gdl_factor_continuous)
	{
		return gdl_factor_continuous;	
	}
	
	return NULL;
}

static int
_gdl_factor_fread (FILE * stream, void * vfactor)
{
	if (stream == 0 || vfactor == 0)
		return GDL_EINVAL;
	else
	{
		size_t status;
		_gdl_factor * vl;
		
		vl = (_gdl_factor *) vfactor;
		
		vl->type = gdl_factor_type_fread (stream);
		GDL_FREAD_STATUS (vl->type!=0, 1);
		status = gdl_hashtable_fread (stream, vl->levels);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		vl->_nl = gdl_hashtable_size (vl->levels);
		if (vl->_nl)
		{
			gdl_hashtable_itr * itr 
				= gdl_hashtable_iterator (vl->levels);
			
			_gdl_factor_levels_grow (vl, vl->_nl);
			
			do
			{
				gdl_factor_level * a =
					gdl_hashtable_iterator_value (itr);
				vl->_levels[a->idx] = a; 
			}
			while (gdl_hashtable_iterator_next (itr));
			
			gdl_hashtable_iterator_free (itr);
		}
		
		return GDL_SUCCESS;
	}
}

int
gdl_factor_type_fwrite (FILE * stream, const gdl_factor_type * T)
{
	int status = fwrite (T, sizeof (size_t), 1, stream);
	GDL_FWRITE_STATUS (status, 1);
	return GDL_SUCCESS;
}

static int
_gdl_factor_fwrite (FILE * stream, const void * vfactor)
{
	if (stream == 0 || vfactor == 0)
		return GDL_EINVAL;
	else
	{
		size_t status;
		_gdl_factor * vl = (_gdl_factor *) vfactor;
		
		status = gdl_factor_type_fwrite (stream, vl->type);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_hashtable_fwrite (stream, vl->levels);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
				
		return GDL_SUCCESS;	
	}
}

gdl_factor *
gdl_factor_alloc (const gdl_factor_type * T)
{
	gdl_factor * f = gdl_entity_alloc (GDL_FACTOR);
	
	((_gdl_factor*)(f->state))->type = T;
	
	return f;
}

gdl_factor *
gdl_factor_new (const gdl_factor_type * T, const char * name)
{
	gdl_factor * f = gdl_factor_alloc (T);
	gdl_entity_set_name (f, name);
	return f;
}

void
gdl_factor_free (gdl_factor * l)
{
	gdl_entity_free (l);	
}

size_t
gdl_factor_size (const gdl_factor * vl)
{
	if (((_gdl_factor *)vl->state)->type == gdl_factor_continuous)
	{
		return 1.0;	
	}
	else
	{
		return ((_gdl_factor *)vl->state)->_nl;
	}
}

const gdl_factor_type *
gdl_factor_get_type (const gdl_factor * vl)
{
	return ((_gdl_factor *)vl->state)->type;
}

void
gdl_factor_add (gdl_factor * vl, gdl_factor_level ** va, size_t owner)
{
	if (vl == 0 || va == 0)
		return;
	else
	{
		_gdl_factor   * l = (_gdl_factor *) vl->state;
		gdl_factor_level   * a = (gdl_factor_level *) gdl_hashtable_lookup (l->levels, (*va)->name);
		
		if (a == NULL)
		{
			gdl_hashtable_add (l->levels, (*va)->name, *va, owner);
			(*va)->idx = gdl_hashtable_size (l->levels) - 1;
			_gdl_factor_levels_grow (l, (*va)->idx+1);
			l->_levels[(*va)->idx] = (*va);
		}
		else if (owner)
		{
			gdl_entity_free (*va);
			*va = a;
		}
	}
}

gdl_factor_level *
gdl_factor_get (const gdl_factor * vl, size_t idx)
{
	if (((_gdl_factor *)vl->state)->type == gdl_factor_continuous)
	{
		return NULL;
	}
	else
	{
		return ((_gdl_factor *)vl->state)->_levels[idx];
	}
}

gdl_factor_level *
gdl_factor_search (const gdl_factor * vl, const gdl_string * name)
{
	if (((_gdl_factor *)vl->state)->type == gdl_factor_continuous)
	{
		return NULL;
	}
	else
	{
		return (gdl_factor_level * ) gdl_hashtable_lookup (((_gdl_factor *)vl->state)->levels, name);
	}
	return NULL;
}

int
gdl_factor_fprintf (FILE * stream, const gdl_factor * l)
{
	if (stream == 0 || l == 0)
		return (-1);
	else
	{	
		gdl_hashtable_itr * itr;
		_gdl_factor * vl = (_gdl_factor *) l->state;
		
		fprintf (stream, "FACTOR %s %d\n", l->name, l->idx);
		
		if (gdl_hashtable_size (vl->levels))
		{
			itr = gdl_hashtable_iterator (vl->levels);
		
			do
			{
				gdl_factor_level * a = (gdl_factor_level *) gdl_hashtable_iterator_value (itr);
				fprintf (stream, "\tLEVEL %s %d\n", a->name, a->idx);
			}
			while (gdl_hashtable_iterator_next (itr));
			
			gdl_hashtable_iterator_free (itr);
		}
	}
}

static const gdl_entity_type _gdl_factor_type =
{
	sizeof(_gdl_factor),
	"F",
	&_gdl_factor_alloc,
	&_gdl_factor_free,
	&_gdl_factor_compare,
	&_gdl_factor_copy,
	&_gdl_factor_fread,
	&_gdl_factor_fwrite
};

const gdl_entity_type * GDL_FACTOR = &_gdl_factor_type;

static const gdl_factor_type _factor_categorial = _gdl_factor_categorial;
static const gdl_factor_type _factor_mixture    = _gdl_factor_mixture;
static const gdl_factor_type _factor_continuous = _gdl_factor_continuous;

const gdl_factor_type * gdl_factor_categorial = &_factor_categorial;
const gdl_factor_type * gdl_factor_mixture    = &_factor_mixture;
const gdl_factor_type * gdl_factor_continuous = &_factor_continuous;

