/*  
 * 	entity/mask.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:03 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_entity.h>

static const size_t _MASK_SLOT = 5;

static void
_gdl_entity_mask_add (gdl_entity_mask * m, size_t i)
{
	if (m->idx == NULL)
	{
		m->slot  = 1;
		m->_size = _MASK_SLOT;
		m->idx   = GDL_MALLOC (size_t, m->_size);
	} 
	else if ((m->_size+1)/_MASK_SLOT == m->slot)
	{
		size_t * nidx;
		m->slot++;
		m->_size = _MASK_SLOT*m->slot;
		nidx = GDL_MALLOC (size_t, m->_size);
		memcpy (nidx, m->idx, sizeof(size_t)*m->size);
		GDL_FREE (m->idx);
		m->idx = nidx;
	}	
	m->idx[m->size] = i;
	m->size++;
}

static void
_gdl_entity_mask_remove (gdl_entity_mask * m, size_t i)
{
	size_t j;
	
	for (j = i + 1; j < m->size; j++)
	{
		m->idx[j - 1] = m->idx[j];
	}
	m->size--;
}

gdl_entity_mask *
gdl_entity_mask_alloc ()
{
	gdl_entity_mask * m = GDL_CALLOC (gdl_entity_mask, 1);
	return m;
}

gdl_entity_mask *
gdl_entity_mask_new (size_t size)
{
	size_t n;
	gdl_entity_mask * m = gdl_entity_mask_alloc ();
	
	if (size)
	{
		n = size / _MASK_SLOT;
		if (size > n*_MASK_SLOT)
		{
			n++;
		}
		m->slot  = n;
		m->_size = m->slot*_MASK_SLOT;
		m->idx   = GDL_MALLOC (size_t, m->_size);
	}
	
	return m;
}

gdl_entity_mask *
gdl_entity_mask_clone (const gdl_entity_mask * m)
{
	if (m == 0)
	{
		return NULL;
	}
	else
	{
		gdl_entity_mask * c = gdl_entity_mask_new (m->size);
		
		if (m->size)
		{
			c->size = m->size;
			memcpy (c->idx, m->idx, sizeof(size_t)*c->size);
		}
		
		return c;
	}
}

gdl_entity_mask *
gdl_entity_mask_ident (size_t size)
{
	size_t i;
	gdl_entity_mask * c = gdl_entity_mask_new (size);
	
	c->size = size;
	
	for (i = 0; i < size; i++)
	{
		c->idx[i] = i;
	}
	
	return c;
}

void
gdl_entity_mask_free (gdl_entity_mask * m)
{
	if (m)
	{
		GDL_FREE (m->idx);
		GDL_FREE (m);
	}
}

int
gdl_entity_mask_add (gdl_entity_mask * m, const gdl_entity * e)
{
	if (m && e)
	{
		_gdl_entity_mask_add (m, e->idx);
		return m->size-1;
	}
	return -1;
}

int
gdl_entity_mask_add_idx (gdl_entity_mask * m, size_t idx)
{
	if (m)
	{
		_gdl_entity_mask_add (m, idx);
		return m->size-1;
	}
	return -1;
}

void
gdl_entity_mask_remove (gdl_entity_mask * m, size_t idx)
{
	if (m && m->size)
	{
		size_t i;
		for (i = idx; i < m->size - 1; i++)
		{
			m->idx[i] = m->idx[i+1];
		}
		(m->size)--;
	}
}

void
gdl_entity_mask_union (gdl_entity_mask * dest, const gdl_entity_mask * src)
{
	if (dest && src)
	{
		size_t i, j;
		
		for (j = 0; j < src->size; j++)
		{
			for (i = 0; i < dest->size; i++)
			{
				if (dest->idx[i]==src->idx[j])
				{
					break;	
				}
			}
			if (i == dest->size)
			{
				_gdl_entity_mask_add (dest, src->idx[j]);
			}
		}
	}	
}

void
gdl_entity_mask_inter (gdl_entity_mask * dest, const gdl_entity_mask * src)
{
	if (dest && src)
	{
		size_t i, j;
		
		for (i = 0; i < dest->size; i++)
		{
			for (j = 0; j < src->size; j++)
			{
				if (dest->idx[i]==src->idx[j])
				{
					break;
				}
			}
			if (j == src->size)
			{
				_gdl_entity_mask_remove (dest, i);
				i--;
			}
		}
	}	
}

void
gdl_entity_mask_inter_over (gdl_entity_mask * dest, const gdl_entity_mask * src, gdl_entity_mask * m1, gdl_entity_mask * m2)
{
	if (dest && src)
	{
		size_t i, j, ii;
		
		for (ii = i = 0; i < dest->size; i++, ii++)
		{
			for (j = 0; j < src->size; j++)
			{
				if (dest->idx[i]==src->idx[j])
				{
					break;
				}
			}
			if (j < src->size)
			{
				_gdl_entity_mask_add (m1, ii);
				_gdl_entity_mask_add (m2, j);
			}
			else
			{
				_gdl_entity_mask_remove (dest, i);
				i--;
			}
		}
	}
}

size_t 
gdl_entity_mask_size (const gdl_entity_mask * m)
{
	if (m)
	{
		return m->size;
	}
	else
	{
		return 0;	
	}
}

size_t
gdl_entity_mask_idx (const gdl_entity_mask * m, size_t i)
{
	if (m)
	{
		return m->idx[i];
	}
	else
	{
		return i;	
	}
}

int
gdl_entity_mask_ridx (const gdl_entity_mask * m, size_t i)
{
	size_t j;
	
	for (j = 0; j < m->size; j++)
	{
		if (m->idx[j] == i)
		{
			return (j);	
		}	
	}
	return (-1);
}

gdl_entity_mask *
gdl_entity_mask_over (const gdl_entity_mask * src, const gdl_entity_mask * sub)
{
	size_t i, j;
	gdl_entity_mask * o = gdl_entity_mask_new (sub->size);
	
	for (i = 0; i < sub->size; i++)
	{
		for (j = 0; j < src->size; j++)
		{
			if (sub->idx[i]==src->idx[j])
			{
				break;
			}
		}
		if (j < src->size)
		{
			_gdl_entity_mask_add (o, j);
		}
	}
	
	return o;
}

int
gdl_entity_mask_compare (const gdl_entity_mask * m1, const gdl_entity_mask * m2)
{
	if (m2 == NULL && m1)
	{
		return -1;
	}
	if (m1 == NULL && m2)
	{
		return 1;
	}
	if (m1 == NULL && m2 == NULL)
	{
		return 0;
	}
	else if (m1->size > m2->size)
	{
		return 1;
	}
	else
	{
		size_t i, j;
		for (i = 0; i < m2->size; i++)
		{
			for (j = 0; j < m1->size; j++)
			{
				if (m1->idx[j]==m2->idx[i])
				{
					break;
				}
			}
			if (j == m1->size)
			{
				return 1;
			}
		}
		if (m1->size == m2->size)
		{
			return 0;	
		}
		else
		{
			return -1;	
		}
	}
}

gdl_entity_mask *
gdl_entity_mask_reverse (const gdl_entity_mask * m, size_t n)
{
	size_t i, j, l;
	gdl_entity_mask * r;
	
	l = gdl_entity_mask_size (m);
	
	r = gdl_entity_mask_new (n-l);
	
	for (i = 0; i < n; i++)
	{
		for (j = 0; j < l; j++)
		{
			if (gdl_entity_mask_idx (m, j) == i) break;
		}
		if (j == l)
		{
			gdl_entity_mask_add_idx (r, i);
		}
	}
	
	return r;
}

gdl_entity_mask *
gdl_entity_mask_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t _size;
		gdl_entity_mask * m = gdl_entity_mask_alloc ();
		
		status = fread (&_size, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		m = gdl_entity_mask_new (_size);
		status = fread (&m->size, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		if (m->size)
		{
			status = fread (m->idx, sizeof (size_t), m->size, stream);
			GDL_FREAD_STATUS (status, m->size);
		}
		
		return m;
	}
	else
	{
		return NULL;	
	}
}

int
gdl_entity_mask_fwrite (FILE * stream, const gdl_entity_mask * m)
{
	if (stream && m)
	{
		int status;
		
		status = fwrite (&m->_size, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&m->size, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);		
		if (m->size)
		{
			status = fwrite (m->idx, sizeof (size_t), m->size, stream);
			GDL_FWRITE_STATUS (status, m->size);
		}
				
		return GDL_SUCCESS;
	}
	else
	{
		return GDL_EINVAL;	
	}	
}

static void
_gdl_entity_mask_free (void * vva)
{
	gdl_entity_mask * va = (gdl_entity_mask *) vva;
	gdl_entity_mask_free (va);
}

static void *
_gdl_entity_mask_clone (const void * vva)
{
	gdl_entity_mask * va = (gdl_entity_mask *) vva;
	return gdl_entity_mask_clone (va);
}

static void *
_gdl_entity_mask_fread (FILE * stream)
{
	return gdl_entity_mask_fread (stream);
}

static int
_gdl_entity_mask_fwrite (FILE * stream, const void * vva)
{
	gdl_entity_mask * va = (gdl_entity_mask *) vva;
	return gdl_entity_mask_fwrite (stream, va);
} 

static const gdl_data_interface _gdl_entity_mask_interface =
{
	&_gdl_entity_mask_free,
	&_gdl_entity_mask_clone,
	NULL,
    &_gdl_entity_mask_fread,
    &_gdl_entity_mask_fwrite
};

const gdl_data_interface * gdl_entity_mask_interface = &_gdl_entity_mask_interface;
