/*  
 * 	gentity/test.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:22:04 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_util.h>
#include <gdl/gdl_test.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_gentity.h>

static const int NGENO  = 1;
static const int NCHROM = 2;
static const int NLOCUS = 5;

static gdl_chromosome *
test_chromosome (const char * name)
{
	size_t i;
	gdl_chromosome * c 
		= gdl_chromosome_new (name);
	
	gdl_chromosome_init (c, NLOCUS);
		
	for (i = 0; i < NLOCUS; i++)
	{
		gdl_string * name = gdl_string_sprintf ("locus_%d", i);
		gdl_locus * l = gdl_locus_new (name);
		gdl_gdistance d;
		
		d.value = 0.1;
		d.type  = gdl_gdistance_morgan;
		
		if (i)
		{
			gdl_chromosome_add (c, l, &d, 1);
		}
		else
		{
			gdl_chromosome_push (c, l, 1);
		}
		
		gdl_string_free (name);
	}
	
	return c;
}

static gdl_genome *
test_genome (const char * name)
{
	size_t i;
	gdl_genome * g = gdl_genome_new (name);
	
	for (i = 0; i < NCHROM; i++)
	{
		gdl_string * name  = gdl_string_sprintf ("chromosome_%d", i);
		
		gdl_chromosome * c = test_chromosome (name);
		
		size_t j = gdl_genome_add (g, c);
		
		gdl_string_free (name);
	}
	
	return g;
}

static void
test_alloc (void)
{
	size_t i;
	
	for (i = 0; i < NGENO; i++)
	{
		gdl_string * name  = gdl_string_sprintf ("genome_%d", i);
		
		gdl_genome * g = test_genome (name);
		
		gdl_test (g==NULL, "gdl_genome_alloc returns valid pointer");
		
		gdl_genome_free (g);
		
		gdl_string_free (name);
	}
}

static void
test_fprintf (void)
{
	size_t i;
	FILE * stream;
	
	for (i = 0; i < NGENO; i++)
	{
		gdl_string * name  = gdl_string_sprintf ("genome_%d", i);
		
		gdl_genome * g = test_genome (name);
		
		stream = gdl_fileopen ("test.txt", "w");
		
		gdl_test (gdl_genome_fprintf (stream, g) != GDL_SUCCESS, "gdl_genome_fprintf returns ok");
		
		gdl_fileclose ("test.txt", stream);
		
		gdl_genome_free (g);
		
		gdl_string_free (name);
	}
}

static void
test_fwrite (void)
{
	size_t i;
	FILE * stream;
	
	for (i = 0; i < NGENO; i++)
	{
		gdl_string * name  = gdl_string_sprintf ("genome_%d", i);
		
		gdl_genome * g = test_genome (name);
		
		stream = gdl_fileopen ("test.dat", "w");
		
		gdl_test (gdl_entity_fwrite (stream, g) != GDL_SUCCESS, "gdl_genome_fwrite returns ok");
		
		gdl_fileclose ("test.dat", stream);
		
		gdl_genome_free (g);
		
		gdl_string_free (name);
	}
}

static void
test_fread (void)
{
	size_t i;
	FILE * stream;
	
	for (i = 0; i < NGENO; i++)
	{
		gdl_genome * g;
		
		stream = gdl_fileopen ("test.dat", "r");
		
		g = gdl_entity_fread (stream);
		
		gdl_test (g == NULL, "gdl_genome_fread returns ok");
		
		gdl_fileclose ("test.dat", stream);
		
		gdl_genome_free (g);
	}
}

int
main (void)
{
	test_alloc ();
	test_fprintf ();
	test_fwrite ();
	test_fread ();
	exit (gdl_test_summary());
}
