/* gfeatures/gfeatures.c
 * 
 * Copyright (C) 2008 Jean-Baptiste Veyrieras
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
 
#include <sys/types.h> 
#include <sys/stat.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_hash.h> 
#include <gdl/gdl_gfeatures.h>

gdl_gfeatures *
gdl_gfeatures_alloc (const gdl_gfeatures_type * T, const gdl_string * file)
{
	gdl_gfeatures * g;
	
	g = GDL_CALLOC (gdl_gfeatures, 1);
	
	g->type = T;
	g->dico = gdl_dictionary_alloc ();
	g->landmarks = (g->type->alloc)(file, &(g->size), g->dico);
	
	if (!g->landmarks)
	{
		GDL_FREE (g);
		GDL_ERROR_VAL ("Unable to parse genomic features !", GDL_FAILURE, 0); 
	}
	
	size_t i;
	
	g->landmark_id2idx = gdl_hashtable_alloc (gdl_interface_uint, g->size);
	
	for(i = 0; i < g->size; i++)
	{
		size_t * idx = GDL_MALLOC (size_t, 1);
		*idx = i;
		gdl_hashtable_add (g->landmark_id2idx, g->landmarks[i]->seqid, idx, 1);
	}
	
	return g;
}

gdl_gfeatures *
gdl_gfeatures_alloc_light (const gdl_gfeatures_type * T, const gdl_string * file, const gdl_string * light_dir)
{
	DIR * dir;
	gdl_gfeatures * g;
	
	g = GDL_CALLOC (gdl_gfeatures, 1);
	
	g->type            = T;
	g->dico            = gdl_dictionary_alloc ();
	g->light_dir       = gdl_string_clone (light_dir);
	// Create the directory if not exist
	if ((dir=opendir(light_dir))==0)
		mkdir (light_dir, S_IRWXU);
	g->light_landmarks = (g->type->alloc_light)(file, light_dir, &(g->size), g->dico);
	
	if (!g->light_landmarks)
	{
		GDL_FREE (g);
		GDL_ERROR_VAL ("Unable to parse genomic features !", GDL_FAILURE, 0); 
	}
	
	g->landmarks = GDL_CALLOC (gdl_gfeatures_landmark *, g->size);
	
	size_t i;
	
	g->landmark_id2idx = gdl_hashtable_alloc (gdl_interface_uint, g->size);
	
	for(i = 0; i < g->size; i++)
	{
		size_t * idx = GDL_MALLOC (size_t, 1);
		*idx = i;
		gdl_hashtable_add (g->landmark_id2idx, g->light_landmarks[i], idx, 1);
	}
	
	return g; 	
}

void
gdl_gfeatures_free (gdl_gfeatures * g)
{
	if (g)
	{
		size_t i;
		if (g->light_dir)
		{
			GDL_FREE (g->light_dir);
			GDL_MATRIX_FREE (g->light_landmarks, g->size);
		}
		for(i = 0; i < g->size; i++)
			(g->type->free)(g->landmarks[i]);
		GDL_FREE (g->landmarks);
		gdl_hashtable_free (g->landmark_id2idx);
		gdl_dictionary_free (g->dico);
		GDL_FREE (g);
	}	
}

size_t
gdl_gfeatures_size (const gdl_gfeatures * g)
{
	return g->size;	
}

gdl_gfeatures_landmark *
gdl_gfeatures_get (const gdl_gfeatures * g, size_t i)
{
	if (g->light_dir && !g->landmarks[i])
	{
		gdl_string * file = GDL_GFEATURES_LIGHT_FILE (g->light_dir, g->light_landmarks[i]);
		FILE * stream     = gdl_fileopen (file, "r");
		g->landmarks[i]   = (g->type->fread)(stream);
		gdl_fileclose (file, stream);
		gdl_string_free (file);
	}
	g->landmarks[i]->dico = g->dico;
	return g->landmarks[i];
}

gdl_gfeatures_landmark *
gdl_gfeatures_lookup (const gdl_gfeatures * g, const gdl_string * seqid)
{
	size_t * idx = gdl_hashtable_lookup (g->landmark_id2idx, seqid);
	if (idx)
	{
		return gdl_gfeatures_get (g, *idx);
	}
	return 0;	
}

int
gdl_gfeatures_update (gdl_gfeatures * g, gdl_gfeatures_landmark * l)
{
	size_t * idx;
	if ((idx=gdl_hashtable_lookup(g->landmark_id2idx, l->seqid)) != 0)
	{
		if (g->light_dir)
		{
			int status;
			gdl_string * file = GDL_GFEATURES_LIGHT_FILE (g->light_dir, l->seqid);
			FILE * stream     = gdl_fileopen (file, "w");
			status = (g->type->fwrite)(stream, l);
			gdl_fileclose (file, stream);
			gdl_string_free (file);
			return status;
		}
		g->landmarks[*idx]=l;
	}
	return GDL_FAILURE;
}

void
gdl_gfeatures_clean (const gdl_gfeatures * g, gdl_gfeatures_landmark * l)
{
	size_t * idx;
	if ((idx=gdl_hashtable_lookup(g->landmark_id2idx, l->seqid)) != 0)
		if (l == g->landmarks[*idx])g->landmarks[*idx]=0;
	(g->type->free)(l);
}

static const gdl_gfeatures_type *
gdl_gfeatures_type_fread (FILE * stream)
{
	gdl_string * name = gdl_string_fread (stream);
	GDL_FREAD_STATUS (name != 0, 1);
	if (!strcmp (name, gdl_gfeatures_gff3->name))
		return gdl_gfeatures_gff3;
	return 0;
}

gdl_gfeatures *
gdl_gfeatures_fread (FILE * stream)
{
	if (stream)
	{
		size_t i;
		int status;
		gdl_gfeatures * g;
		
		g = GDL_CALLOC (gdl_gfeatures, 1);
		
		g->type = gdl_gfeatures_type_fread (stream);
		GDL_FREAD_STATUS (g->type != 0, 1);
		status = fread (&g->size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		g->light_dir = gdl_string_fread (stream);
		GDL_FREAD_STATUS (g->light_dir != 0, 1);
		
		g->landmarks = GDL_CALLOC (gdl_gfeatures_landmark *, g->size);
		g->landmark_id2idx = gdl_hashtable_alloc (gdl_interface_uint, g->size);
		
		if (!strcmp (g->light_dir, ""))
		{
			GDL_FREE (g->light_dir);
			g->light_dir=0;
			for(i = 0; i < g->size; i++)
			{
				g->landmarks[i] = (g->type->fread)(stream);
				GDL_FREAD_STATUS (g->landmarks[i] != 0, 1);
				size_t * idx = GDL_MALLOC (size_t, 1);
				*idx=i;
				gdl_hashtable_add (g->landmark_id2idx, g->landmarks[i]->seqid, idx, 1);
			}
		}
		else
		{
			g->light_landmarks = GDL_MALLOC (gdl_string *, g->size);
			for(i = 0; i < g->size; i++)
			{
				g->light_landmarks[i] = gdl_string_fread (stream);
				GDL_FREAD_STATUS (g->light_landmarks[i] != 0, 1);
				size_t * idx = GDL_MALLOC (size_t, 1);
				*idx=i;
				gdl_hashtable_add (g->landmark_id2idx, g->light_landmarks[i], idx, 1);
			}
		}
		
		g->dico = gdl_dictionary_fread (stream);
		GDL_FREAD_STATUS (g->dico != 0, 1);
		
		return g;
	}
	return 0;	
}

static int
gdl_gfeatures_type_fwrite (FILE * stream, const gdl_gfeatures_type * T)
{
	if (T)
		return gdl_string_fwrite (stream, T->name);
	else
		return GDL_EINVAL;
}

int
gdl_gfeatures_fwrite (FILE * stream, const gdl_gfeatures * g)
{
	if (stream && g)
	{
		size_t i;
		int status;
		
		status = gdl_gfeatures_type_fwrite (stream, g->type);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = fwrite (&g->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		if (!g->light_dir)
		{
			status = gdl_string_fwrite (stream, "");
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			for(i = 0; i < g->size; i++)
			{
				status = (g->type->fwrite)(stream, g->landmarks[i]);
				GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			}
		}
		else
		{
			status = gdl_string_fwrite (stream, g->light_dir);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			for(i = 0; i < g->size; i++)
			{
				status = gdl_string_fwrite (stream, g->light_landmarks[i]);
				GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			}
		}
		
		status = gdl_dictionary_fwrite (stream, g->dico);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;	
}

int
gdl_gfeatures_fprintf (FILE * stream, const gdl_gfeatures * g)
{	
	if (stream && g)
	{
		size_t i;
		
		fprintf (stream, "##gff-version 3\n");
		for(i = 0; i < g->size; i++)
		{
			gdl_gfeatures_landmark * l = gdl_gfeatures_get (g, i);
			
			l->dico = g->dico;
			
			(g->type->fprintf)(stream, l);
			
			if (g->light_dir) gdl_gfeatures_clean (g, l);
		}
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}

gdl_gfeatures_landmark *
gdl_gfeatures_landmark_alloc (const gdl_string * seqid)
{
	gdl_gfeatures_landmark * l;
	
	l = GDL_CALLOC (gdl_gfeatures_landmark, 1);
	
	l->seqid = gdl_string_clone (seqid);
	
	return l;
}

