/* gfeatures/gff3.c
 * 
 * Copyright (C) 2008 Jean-Baptiste Veyrieras
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
 
#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_list.h> 
#include <gdl/gdl_hash.h>
#include <gdl/gdl_math.h> 
#include <gdl/gdl_sort_long.h> 
#include <gdl/gdl_gfeatures.h>
#include <gdl/gdl_gfeatures_gff3.h>

gdl_gfeatures_gff3_record *
gdl_gfeatures_gff3_record_alloc (void)
{
	gdl_gfeatures_gff3_record * r;
	
	r = GDL_CALLOC (gdl_gfeatures_gff3_record, 1);
	r->attributes = gdl_hashtable_alloc (gdl_string_interface, 0);
	
	return r;	
}

void
gdl_gfeatures_gff3_record_free (gdl_gfeatures_gff3_record * r)
{
	if (r)
	{
		gdl_hashtable_free (r->attributes);
		GDL_FREE (r->parents);
		GDL_FREE (r->children);
		GDL_FREE (r);
	}
}	

int
gdl_gfeatures_gff3_record_add_parent (gdl_gfeatures_gff3_record * child, gdl_gfeatures_gff3_record * parent)
{
	if (child && parent)
	{
		// look if this parent has not been previously added
		size_t i;
		for(i = 0; i < child->nparent; i++)
		{
			if (child->parents[i]==parent)
				break;
		}
		if (i < child->nparent)
		{
			return 0;
		}
		// child 2 parent
		if (!child->nparent)
			child->parents = GDL_MALLOC (gdl_gfeatures_gff3_record *, 1);
		else
		{
			gdl_gfeatures_gff3_record ** tmp = GDL_MALLOC (gdl_gfeatures_gff3_record *, child->nparent+1);
			memcpy (tmp, child->parents, sizeof(gdl_gfeatures_gff3_record *)*child->nparent);
			GDL_FREE (child->parents);
			child->parents=tmp;
		}
		child->parents[child->nparent]=parent;
		(child->nparent)++;
		// parent 2 child
		if (!parent->nchild)
			parent->children = GDL_MALLOC (gdl_gfeatures_gff3_record *, 1);
		else
		{
			gdl_gfeatures_gff3_record ** tmp = GDL_MALLOC (gdl_gfeatures_gff3_record *, parent->nchild+1);
			memcpy (tmp, parent->children, sizeof(gdl_gfeatures_gff3_record *)*parent->nchild);
			GDL_FREE (parent->children);
			parent->children=tmp;
		}
		parent->children[parent->nchild]=child;
		return (parent->nchild)++;
	}
	return -1;
}

static int
gdl_gfeatures_gff3_record_add_attribute (const gdl_gfeatures_landmark * landmark, gdl_gfeatures_gff3_record * rec, const gdl_string * key, const gdl_string * value)
{
	if (!strcmp(key, "Parent"))
	{
		// parents must have been previously defined...
		gdl_gfeatures_gff3_record * prec;
		gdl_string * pid;
		size_t i,j,len;
		
		len = strlen (value);
		i = j = 0;
		while (j < len)
		{
			for(j = i; j < len && value[j] != (char)GFF3_ATTRIBUTE_MULTISEP; j++);
			pid = gdl_string_alloc (j-i);
			strncpy (pid, &value[i], j-i);
			prec = gdl_hashtable_lookup (landmark->_rec_id_buffer, pid);
			//printf ("rec %p look for parent %s ==> %p\n", rec, pid, prec);
			if (prec)
			{
			   gdl_gfeatures_gff3_record_add_parent (rec, prec);
			}
			//printf (" rec %p has %d parent(s)\n", rec, rec->nparent);
			//printf (" rec %p has %d child(ren)\n", prec, prec->nchild);
			gdl_string_free (pid);
			i = j+1;
		}
	}
	else if (!strcmp(key, "ID"))
	{
		if (gdl_hashtable_lookup (landmark->_rec_id_buffer, value))
			GDL_ERROR_VAL (gdl_string_sprintf("ID=%s is not unique\n",value),GDL_EINVAL,GDL_EINVAL);
		gdl_hashtable_add (landmark->_rec_id_buffer, value, rec, 0);
	}
	
	//printf("%s=>%s\n",key,value);
	gdl_hashtable_add (rec->attributes, key, gdl_string_clone (value), 1);
	
	return GDL_SUCCESS;
} 

static void
gdl_gfeatures_gff3_record_check_relationship (const gdl_gfeatures_landmark * landmark,
                                              gdl_gfeatures_gff3_record * rec)
{
	gdl_string * value;
	if ((value=gdl_hashtable_lookup (rec->attributes, "Parent"))!=0)
	{
		gdl_gfeatures_gff3_record * prec;
		gdl_string * pid;
		size_t i,j,len;
		len = strlen (value);
		i = j = 0;
		while (j < len)
		{
			for(j = i; j < len && value[j] != (char)GFF3_ATTRIBUTE_MULTISEP; j++);
			pid = gdl_string_alloc (j-i);
			strncpy (pid, &value[i], j-i);
			prec = gdl_hashtable_lookup (landmark->_rec_id_buffer, pid);
			//printf ("rec %p look for parent %s ==> %p\n", rec, pid, prec);
			if (!prec)
			{
				gdl_string_free (pid);
				GDL_ERROR_VOID (gdl_string_sprintf("Unable to find parent ID=%s for Parent attribute: [%s]\n",pid,value),GDL_FAILURE);
			}
			else
			{
			   gdl_gfeatures_gff3_record_add_parent (rec, prec);
			}
			//printf (" rec %p has %d parent(s)\n", rec, rec->nparent);
			//printf (" rec %p has %d child(ren)\n", prec, prec->nchild);
			gdl_string_free (pid);
			i = j+1;
		}
	}
}

static gdl_gfeatures_landmark ** 
gdl_gfeatures_gff3_alloc (const gdl_string * file, size_t * nlandmark, gdl_dictionary * dico)
{
	FILE * stream;
	
	stream = gdl_fileopen (file, "r");
	
	if (!stream)
	{
		return 0;
	}
	else
	{
		gdl_gfeatures_landmark ** landmarks;
		gdl_gfeatures_landmark * landmark;
		gdl_gfeatures_gff3_record  * rec;
		gdl_hashtable * buffer;
		size_t i,j,k,l,n,m,o,p;
		gdl_string * line = 0, * tok, * key, * value;
		
		buffer = gdl_hashtable_alloc (gdl_hash_default, 0);
		gdl_getline (&line, &n, stream);
		if (strncmp (line, "##gff-version 3", 15))
			GDL_ERROR_VAL ("Invalid file format",GDL_EINVAL,0);
		
		gdl_string_free (line);line=0;
		l = 1;
		// Add the two required entries to the dictionary
		gdl_dictionary_add (dico, "source");
		gdl_dictionary_add (dico, "type");
		
		while(gdl_getline (&line, &n, stream)!=-1)
		{
			if (line[0] == '#')
			{
				gdl_string_free (line);line=0;
				continue;
			}
			l++;
			i=j=k=0;
			while(k < 9 && (i < n && j < n))
			{
				for(j = i;j < n && line[j] != (char)GFF3_SEPARATOR; j++);
				if (k < 8)
				{
					tok = gdl_string_unescape (&line[i], j-i, 0);
					switch(k)
					{
						case 0: // seqid
							rec = gdl_gfeatures_gff3_record_alloc();
							landmark = gdl_hashtable_lookup (buffer, tok);
							if (!landmark)
							{
								landmark = gdl_gfeatures_landmark_alloc (tok);
								landmark->_rec_buffer    = gdl_list_alloc (gdl_list_default);
								landmark->_rec_id_buffer = gdl_hashtable_alloc (gdl_hash_default, 0);
								gdl_hashtable_add (buffer, tok, landmark, 0); 
							}
							rec->idx = gdl_list_size (landmark->_rec_buffer);
							gdl_list_push_back (landmark->_rec_buffer, rec, 0);
							break;
						case 1: // source
							rec->source_d = gdl_dictionary_populate (dico, "source", tok);
							break;
						case 2: // type
							rec->type_d = gdl_dictionary_populate (dico, "type", tok);
							break;
						case 3: // start
							rec->start = atol(tok);
							break; 
						case 4: // end
							rec->end = atol(tok);
							if (rec->start > rec->end)
								GDL_ERROR_VAL (gdl_string_sprintf ("In line %d <start> position is greater than <end> position\n",l),GDL_EINVAL,0);
							break;
						case 5: // score
							if (!strcmp(tok, (char *)GFF3_UNDEFINED_FIELD))
								rec->score = GDL_NAN;
							else
								rec->score = (double)atof(tok);								
							break;
						case 6: // strand
							if (!strcmp(tok, (char *)GFF3_UNDEFINED_FIELD))
								rec->strand = '?';
							else 
								rec->strand = (unsigned char)tok[0];
							break;
						case 7: // phase
							if (!strcmp(tok, (char *)GFF3_UNDEFINED_FIELD))
								rec->phase = -1;
							else
								rec->phase = atoi(tok);
							break;
					}
					gdl_string_free (tok);
				}
				else // attributes
				{
					m = i;
					while(m < j)
					{
						o = p = m;
						for(;m < j && line[m] != (char)GFF3_ATTRIBUTE_SEPARATOR; m++);
						for(;p < m && line[p] != (char)GFF3_ATTRIBUTE_DEFINE; p++);
						if (p == m)
							GDL_ERROR_VAL (gdl_string_sprintf ("Line %d has an invalid attribute declaration (9th field)\n",l),GDL_EINVAL,0);
						key   = gdl_string_unescape (&line[o], p-o, 0);
						value = gdl_string_unescape (&line[p+1], m-p-1, 0);
						gdl_gfeatures_gff3_record_add_attribute (landmark, rec, key, value);
						gdl_string_free (key);
						gdl_string_free (value);
						m++;
					}
				}
				i=j+1;
				k++;
			}
			if (k < 9) 
				GDL_ERROR_VAL (gdl_string_sprintf ("Line %d does not contain 9 fields separated by a tabulation\n",l),GDL_EINVAL,0);
			gdl_string_free (line);line=0;
		}
		
		// Now, re-format the buffered landmarks
		 
		{
			if ((*nlandmark=gdl_hashtable_size (buffer))!=0)
			{
				i=0;
				gdl_list_itr * ritr;
				gdl_hashtable_itr * itr = gdl_hashtable_iterator (buffer);
				
				landmarks = GDL_MALLOC (gdl_gfeatures_landmark *, *nlandmark);
				do
				{
					size_t * idx;
					long * pos;
					void ** tmp;
					landmarks[i]         = gdl_hashtable_iterator_value (itr);
					landmarks[i]->size   = gdl_list_size (landmarks[i]->_rec_buffer);
					landmarks[i]->records= GDL_MALLOC (void *, landmarks[i]->size);
					pos = GDL_MALLOC (long, landmarks[i]->size); 
					ritr = gdl_list_iterator_front (landmarks[i]->_rec_buffer);
					j = 0;
					do
					{
						landmarks[i]->records[j]=gdl_list_iterator_value (ritr);
						gdl_gfeatures_gff3_record_check_relationship (landmarks[i], landmarks[i]->records[j]);
						gdl_gfeatures_gff3_record  * rec = (gdl_gfeatures_gff3_record  *)landmarks[i]->records[j]; 
						pos[j]=rec->start;
						j++;						
					}
					while (gdl_list_iterator_next (ritr));
					gdl_list_iterator_free (ritr);
					// clean the buffers
					gdl_list_free (landmarks[i]->_rec_buffer);landmarks[i]->_rec_buffer=0;
					gdl_hashtable_free (landmarks[i]->_rec_id_buffer);landmarks[i]->_rec_id_buffer=0;
					// sort the records
					idx = GDL_MALLOC (size_t, landmarks[i]->size);
					gdl_sort_long_index (idx, pos, 1, landmarks[i]->size);
					tmp = GDL_MALLOC (void *, landmarks[i]->size);
					for(j = 0; j <  landmarks[i]->size; j++)
					{
						tmp[j] = landmarks[i]->records[idx[j]];	
						gdl_gfeatures_gff3_record  * rec = (gdl_gfeatures_gff3_record  *)tmp[j]; 
					}
					GDL_FREE (landmarks[i]->records);
					landmarks[i]->records=tmp;
					GDL_FREE (idx);
					GDL_FREE (pos);
					i++;
				}
				while(gdl_hashtable_iterator_next (itr));
				gdl_hashtable_iterator_free (itr);				
			}
		}
		
		gdl_hashtable_free (buffer);
		gdl_fileclose(file, stream);
		
		return landmarks;
	}
}

static gdl_string ** 
gdl_gfeatures_gff3_alloc_light (const gdl_string * file, const gdl_string * odir, size_t * nlandmark, gdl_dictionary * dico)
{
	// TODO :-)
}

static void 
gdl_gfeatures_gff3_free (gdl_gfeatures_landmark * l)
{
	if (l)
	{
		size_t i;
		for (i = 0; i < l->size; i++)
			gdl_gfeatures_gff3_record_free (l->records[i]);
		GDL_FREE (l->records);
		GDL_FREE (l);
	}
}

gdl_gfeatures_landmark *
gdl_gfeatures_gff3_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i,j,pid,cid;
		gdl_gfeatures_landmark * l;
		
		l = GDL_CALLOC (gdl_gfeatures_landmark, 1);
		
		l->seqid = gdl_string_fread (stream);
		GDL_FREAD_STATUS (l->seqid != 0, 1);
		status   = fread (&(l->size), sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		l->records = GDL_CALLOC (void *, l->size);
		for(i = 0; i < l->size; i++)
		{
			gdl_gfeatures_gff3_record * rec = l->records[i] = gdl_gfeatures_gff3_record_alloc ();
			rec->idx = i;
			status = fread (&(rec->source_d), sizeof(size_t), 1, stream);
			GDL_FREAD_STATUS (status, 1);
			status = fread (&(rec->type_d), sizeof(size_t), 1, stream);
			GDL_FREAD_STATUS (status, 1);
			status = fread (&(rec->start), sizeof(long), 1, stream);
			GDL_FREAD_STATUS (status, 1);
			status = fread (&(rec->end), sizeof(long), 1, stream);
			GDL_FREAD_STATUS (status, 1);
			status = fread (&(rec->score), sizeof(double), 1, stream);
			GDL_FREAD_STATUS (status, 1);
			status = fread (&(rec->strand), sizeof(unsigned char), 1, stream);
			GDL_FREAD_STATUS (status, 1);
			status = fread (&(rec->phase), sizeof(int), 1, stream);
			GDL_FREAD_STATUS (status, 1);	
			status = gdl_hashtable_fread (stream, rec->attributes);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);			
		}
		for(i = 0; i < l->size; i++)
		{
			gdl_gfeatures_gff3_record * rec = l->records[i];
			status = fread (&(rec->nparent), sizeof(size_t), 1, stream);
			GDL_FREAD_STATUS (status, 1);
			if (rec->nparent)
			{
				rec->parents = GDL_MALLOC (gdl_gfeatures_gff3_record *, rec->nparent);
				for(j = 0; j < rec->nparent; j++)
				{
					status = fread (&pid, sizeof(size_t), 1, stream);
					GDL_FREAD_STATUS (status, 1);
					rec->parents[j] = l->records[pid]; 
				}
			}
			status = fread (&(rec->nchild), sizeof(size_t), 1, stream);
			GDL_FREAD_STATUS (status, 1);
			if (rec->nchild)
			{
				rec->children = GDL_MALLOC (gdl_gfeatures_gff3_record *, rec->nchild);
				for(j = 0; j < rec->nchild; j++)
				{
					status = fread (&cid, sizeof(size_t), 1, stream);
					GDL_FREAD_STATUS (status, 1);
					rec->children[j] = l->records[cid]; 
				}
			}
		}
		
		return l; 
	}
	return 0;
}

int 
gdl_gfeatures_gff3_fwrite (FILE * stream, const gdl_gfeatures_landmark * l)
{
	if (stream && l)
	{
		int status;
		size_t i,j;
		
		status = gdl_string_fwrite (stream, l->seqid);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status   = fwrite (&(l->size), sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		for(i = 0; i < l->size; i++)
		{
			gdl_gfeatures_gff3_record * rec = l->records[i];
			status = fwrite (&(rec->source_d), sizeof(size_t), 1, stream);
			GDL_FWRITE_STATUS (status, 1);
			status = fwrite (&(rec->type_d), sizeof(size_t), 1, stream);
			GDL_FWRITE_STATUS (status, 1);
			status = fwrite (&(rec->start), sizeof(long), 1, stream);
			GDL_FWRITE_STATUS (status, 1);
			status = fwrite (&(rec->end), sizeof(long), 1, stream);
			GDL_FWRITE_STATUS (status, 1);
			status = fwrite (&(rec->score), sizeof(double), 1, stream);
			GDL_FWRITE_STATUS (status, 1);
			status = fwrite (&(rec->strand), sizeof(unsigned char), 1, stream);
			GDL_FWRITE_STATUS (status, 1);
			status = fwrite (&(rec->phase), sizeof(int), 1, stream);
			GDL_FWRITE_STATUS (status, 1);
			status = gdl_hashtable_fwrite (stream, rec->attributes);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);	
		}
		for(i = 0; i < l->size; i++)
		{
			gdl_gfeatures_gff3_record * rec = l->records[i];
			status = fwrite (&(rec->nparent), sizeof(size_t), 1, stream);
			GDL_FWRITE_STATUS (status, 1);
			if (rec->nparent)
			{
				for(j = 0; j < rec->nparent; j++)
				{
					status = fwrite (&(rec->parents[j]->idx), sizeof(size_t), 1, stream);
					GDL_FWRITE_STATUS (status, 1);
				}
			}
			status = fwrite (&(rec->nchild), sizeof(size_t), 1, stream);
			GDL_FWRITE_STATUS (status, 1);
			if (rec->nchild)
			{
				for(j = 0; j < rec->nchild; j++)
				{
					status = fwrite (&(rec->children[j]->idx), sizeof(size_t), 1, stream);
					GDL_FWRITE_STATUS (status, 1);
				}
			}
		}
		return GDL_SUCCESS; 
	}
	return GDL_EINVAL;
}


int
gdl_gfeatures_gff3_record_fprintf (FILE * stream, const gdl_gfeatures_landmark * l, const gdl_gfeatures_gff3_record * rec)
{
	size_t j,n,sep;
	
	fprintf(stream, "%s", l->seqid);
	if (l->dico)
		fprintf(stream, "%c%s%c%s", (char)GFF3_SEPARATOR, gdl_dictionary_get(l->dico, "source", rec->source_d), (char)GFF3_SEPARATOR, gdl_dictionary_get(l->dico, "type", rec->type_d));
	else
		fprintf(stream, "%c%d%c%d", (char)GFF3_SEPARATOR, rec->source_d, (char)GFF3_SEPARATOR, rec->type_d);
	fprintf(stream, "%c%ld%c%ld", (char)GFF3_SEPARATOR, rec->start, (char)GFF3_SEPARATOR, rec->end);
	if (gdl_isnan (rec->score)) fprintf(stream, "%c%s", (char)GFF3_SEPARATOR, (char *)GFF3_UNDEFINED_FIELD);
	else fprintf(stream, "%c%e", (char)GFF3_SEPARATOR, rec->score);
	if (rec->strand=='?') fprintf(stream, "%c%s", (char)GFF3_SEPARATOR, (char *)GFF3_UNDEFINED_FIELD);
	else fprintf(stream, "%c%c", (char)GFF3_SEPARATOR, rec->strand);
	if (rec->phase==-1) fprintf(stream, "%c%s", (char)GFF3_SEPARATOR, (char *)GFF3_UNDEFINED_FIELD);
	else fprintf(stream, "%c%d", (char)GFF3_SEPARATOR, rec->phase);
	fprintf(stream, "%c", (char)GFF3_SEPARATOR);
	sep=0;
	if (rec->nparent)
	{
		fprintf(stream, "Parent%c",(char)GFF3_ATTRIBUTE_DEFINE);
		for(j = 0; j < rec->nparent; j++)
		{
			fprintf(stream, "%s", gdl_hashtable_lookup (rec->parents[j]->attributes, "ID"));
			if (j < rec->nparent-1) fprintf(stream, "%c", (char)GFF3_ATTRIBUTE_MULTISEP);
		}
		sep=1;
	}
	if ((n=gdl_hashtable_size (rec->attributes))!=0)
	{
		gdl_hashtable_itr * itr = gdl_hashtable_iterator (rec->attributes);
		j=0;
		do
		{
			if (sep) fprintf (stream, "%c", (char)GFF3_ATTRIBUTE_SEPARATOR);
			const gdl_string * key   = gdl_hashtable_iterator_key (itr);
			const gdl_string * value = gdl_hashtable_iterator_value (itr);
			gdl_string * evalue      = gdl_string_escape (value, ",:");
			fprintf (stream, "%s%c%s", key, (char)GFF3_ATTRIBUTE_DEFINE, evalue);
			gdl_string_free (evalue);
			if (!j) sep=1;
			j++;											
		}
		while(gdl_hashtable_iterator_next (itr));
		gdl_hashtable_iterator_free (itr);
	}
	fprintf (stream, "\n");			
}

int
gdl_gfeatures_gff3_record_fprintf2 (FILE * stream, const gdl_string * new_landmark, const gdl_gfeatures_landmark * l, const gdl_gfeatures_gff3_record * rec)
{
	size_t j,n,sep;
	
	fprintf(stream, "%s", new_landmark);
	if (l->dico)
		fprintf(stream, "%c%s%c%s", (char)GFF3_SEPARATOR, gdl_dictionary_get(l->dico, "source", rec->source_d), (char)GFF3_SEPARATOR, gdl_dictionary_get(l->dico, "type", rec->type_d));
	else
		fprintf(stream, "%c%d%c%d", (char)GFF3_SEPARATOR, rec->source_d, (char)GFF3_SEPARATOR, rec->type_d);
	fprintf(stream, "%c%ld%c%ld", (char)GFF3_SEPARATOR, rec->start, (char)GFF3_SEPARATOR, rec->end);
	if (gdl_isnan (rec->score)) fprintf(stream, "%c%s", (char)GFF3_SEPARATOR, (char *)GFF3_UNDEFINED_FIELD);
	else fprintf(stream, "%c%e", (char)GFF3_SEPARATOR, rec->score);
	if (rec->strand=='?') fprintf(stream, "%c%s", (char)GFF3_SEPARATOR, (char *)GFF3_UNDEFINED_FIELD);
	else fprintf(stream, "%c%c", (char)GFF3_SEPARATOR, rec->strand);
	if (rec->phase==-1) fprintf(stream, "%c%s", (char)GFF3_SEPARATOR, (char *)GFF3_UNDEFINED_FIELD);
	else fprintf(stream, "%c%d", (char)GFF3_SEPARATOR, rec->phase);
	fprintf(stream, "%c", (char)GFF3_SEPARATOR);
	sep=0;
	if (rec->nparent)
	{
		fprintf(stream, "Parent%c",(char)GFF3_ATTRIBUTE_DEFINE);
		for(j = 0; j < rec->nparent; j++)
		{
			fprintf(stream, "%s", gdl_hashtable_lookup (rec->parents[j]->attributes, "ID"));
			if (j < rec->nparent-1) fprintf(stream, "%c", (char)GFF3_ATTRIBUTE_MULTISEP);
		}
		sep=1;
	}
	if ((n=gdl_hashtable_size (rec->attributes))!=0)
	{
		gdl_hashtable_itr * itr = gdl_hashtable_iterator (rec->attributes);
		j=0;
		do
		{
			if (sep) fprintf (stream, "%c", (char)GFF3_ATTRIBUTE_SEPARATOR);
			const gdl_string * key   = gdl_hashtable_iterator_key (itr);
			const gdl_string * value = gdl_hashtable_iterator_value (itr);
			gdl_string * evalue      = gdl_string_escape (value, ",:");
			fprintf (stream, "%s%c%s", key, (char)GFF3_ATTRIBUTE_DEFINE, evalue);
			gdl_string_free (evalue);
			if (!j) sep=1;
			j++;											
		}
		while(gdl_hashtable_iterator_next (itr));
		gdl_hashtable_iterator_free (itr);
	}
	fprintf (stream, "\n");			
}

int 
gdl_gfeatures_gff3_fprintf (FILE * stream, const gdl_gfeatures_landmark * l)
{
	if (stream && l)
	{
		size_t i;
		for(i = 0; i < l->size; i++)
		{
			gdl_gfeatures_gff3_record_fprintf (stream, l, l->records[i]);	
		}
		return GDL_SUCCESS; 
	}
	return GDL_EINVAL;
}

int
gdl_gfeatures_gff3_record_and_children_fprintf (FILE * stream, const gdl_gfeatures_landmark * l, const gdl_gfeatures_gff3_record * rec)
{
	size_t c;
	gdl_gfeatures_gff3_record_fprintf (stream, l, rec);
	for(c = 0; c < rec->nchild; c++)
		gdl_gfeatures_gff3_record_fprintf (stream, l, rec->children[c]);
}

gdl_boolean
gdl_gfeatures_gff3_record_is_of_type (const gdl_gfeatures * feature, const  gdl_gfeatures_gff3_record * rec, const gdl_string * type)
{
	const gdl_string * rec_type = gdl_dictionary_get (feature->dico, "type", rec->type_d);
	if (strcmp(rec_type,type))
		return gdl_false;
	return gdl_true;
}

size_t
gdl_gfeatures_gff3_record_children_of_type (const gdl_gfeatures * feature, const  gdl_gfeatures_gff3_record * rec, const gdl_string * type)
{
	size_t c,n=0;
	for(c = 0; c < rec->nchild; c++)
		if (gdl_gfeatures_gff3_record_is_of_type (feature, rec->children[c], type))
			n++;	
	return n;
}

gdl_boolean
gdl_gfeatures_gff3_record_is_inside (const gdl_gfeatures_gff3_record * rec1, const gdl_gfeatures_gff3_record * rec2)
{
	return (rec2->start >= rec1->start && rec2->end <= rec1->end) ? gdl_true : gdl_false;
}

size_t
gdl_gfeatures_gff3_record_size (const gdl_gfeatures_gff3_record * rec)
{
	return (rec->end-rec->start+1);
}

int
gdl_gfeatures_gff3_record_parse_target (const gdl_gfeatures_gff3_record * rec, gdl_string ** target_id, long * target_start, long * target_end, unsigned char * target_strand)
{
	const gdl_string * Target = gdl_hashtable_lookup (rec->attributes, "Target");
	if (Target)
	{
		gdl_string * tmp;
		size_t i,j,n=strlen(Target);
		
		for(i=0;i<n && isspace(Target[i]);i++);
		if (i==n) return 0;
		for(j=i;j<n && !isspace(Target[j]);j++);
		if (j==n) return 0;
		if (*target_id) gdl_string_free (*target_id);
		*target_id = gdl_string_alloc (j-i);
		memcpy (*target_id, &Target[i], j-i);
		for(i=j;i<n && isspace(Target[i]);i++);
		if (i==n) return 0;
		for(j=i;j<n && !isspace(Target[j]);j++);
		if (j==n) return 0;
		tmp = gdl_string_alloc (j-i);
		memcpy (tmp, &Target[i], j-i);
		*target_start = atol(tmp);
		gdl_string_free (tmp);
		for(i=j;i<n && isspace(Target[i]);i++);
		if (i==n) return 0;
		for(j=i;j<n && !isspace(Target[j]);j++);
		if (j==n) return 0;
		tmp = gdl_string_alloc (j-i);
		memcpy (tmp, &Target[i], j-i);
		*target_end = atol(tmp);
		gdl_string_free (tmp);
		for(i=j;i<n && isspace(Target[i]);i++);
		if (i==n) return 1;
		*target_strand = Target[i];
		
		return 1;
	}
	return 0;
}

int
gdl_gfeatures_gff3_within (gdl_gfeatures_landmark * l, const long from, const long to, int * from_idx, int * to_idx)
{
	size_t i;
	
	*from_idx = *to_idx = -1;
	
	for(i = 0; i < l->size; i++)
	{
		gdl_gfeatures_gff3_record * rec = (gdl_gfeatures_gff3_record *) l->records[i];
		if (((rec->start >= from && rec->end <= to)
		    || (rec->start >= from && rec->start < to)
		    || (rec->end >= from && rec->end < to))
		    && *from_idx==-1)
		{
			*from_idx=*to_idx=i;
		}
		if (rec->start > to)
		{
			if (*from_idx != -1) *to_idx=i-1;
			break;
		}
	}
	
	return (*from_idx==-1 || *to_idx==-1) ? 0 : 1;
}

gdl_gfeatures_landmark *
gdl_gfeatures_gff3_convert (const gdl_gfeatures_landmark * l, const gdl_gfeatures_type * TO)
{
	// TODO
}

static gdl_gfeatures_type _gdl_gfeatures_gff3 =
{
	"gff3",
	&gdl_gfeatures_gff3_alloc,
	&gdl_gfeatures_gff3_alloc_light,
	&gdl_gfeatures_gff3_free,
	&gdl_gfeatures_gff3_fread,
	&gdl_gfeatures_gff3_fwrite,
	&gdl_gfeatures_gff3_fprintf,
	&gdl_gfeatures_gff3_convert,
};

const gdl_gfeatures_type * gdl_gfeatures_gff3 = &_gdl_gfeatures_gff3;
