/* gglm/gglm.c
 * 
 * Copyright (C) 2006 Jean-Baptiste Veyrieras
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <stdlib.h>
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_multireg.h>
#include <gdl/gdl_gglm.h>

struct _gdl_gglm_workspace
{
  gdl_gglm_model * model;
  const gdl_gmatrix_type  * gcode;
  const gdl_fview_wrapper * tdata;
  const gdl_fview_wrapper * odata;
  const gdl_gview_wrapper * gdata;
  gdl_list * heap;
  FILE * logger;
};

gdl_gglm_workspace *
gdl_gglm_workspace_alloc (const gdl_fview_wrapper * t, const gdl_gview_wrapper * g, const gdl_fview_wrapper * o)
{
	size_t i, tn, on, gn;
	gdl_gglm_workspace * w;
	
	if (!t || !g)
	{
		GDL_ERROR_VAL ("No trait data defined", GDL_EINVAL, 0);
	}
	
	tn = gdl_fview_wrapper_accession_size (t);
	gn = gdl_gview_wrapper_accession_size (g);
	
	if (tn != gn)
	{
		GDL_ERROR_VAL ("Trait and Genotypic Data doesn't share the same number of accessions", GDL_EINVAL, 0);	
	}
	if (o)
	{
		on = gdl_fview_wrapper_accession_size (o);
		if (on != tn)
		{
			GDL_ERROR_VAL ("Covariable Data doesn't match the number of accessions of Trait and Genotypic Data", GDL_EINVAL, 0);	
		}
	}
	tn = gdl_fview_wrapper_factor_size (t);
	for (i = 0; i < tn; i++)
	{
		gdl_factor * factor = gdl_fview_wrapper_get_factor (t, i);
		if (gdl_factor_get_type (factor) != gdl_factor_continuous)
		{
			GDL_ERROR_VAL (gdl_string_sprintf ("Trait [ %d ] [ %s ] is not a continuous variable", i+1, gdl_entity_get_name (factor)), GDL_EINVAL, 0);		
		}
	}
	
	w = GDL_CALLOC (gdl_gglm_workspace, 1);
	
	w->tdata = t;
	w->odata = o;
	w->gdata = g;
	
	w->heap  = gdl_list_alloc (gdl_list_default);
	
	return w;
}

void
gdl_gglm_workspace_free (gdl_gglm_workspace * w)
{
	if (w)
	{
		if (gdl_list_size (w->heap))
		{
			gdl_list_itr * itr;
			itr = gdl_list_iterator_front (w->heap);
			do
			{
			   gdl_gglm_static_model_free ((gdl_gglm_static_model *)gdl_list_iterator_value (itr)); 
			}
			while (gdl_list_iterator_next (itr));
			gdl_list_iterator_free (itr);
		}
		gdl_list_free (w->heap);
		GDL_FREE (w);	
	}	
}

const gdl_gview_wrapper *
gdl_gglm_workspace_gdata (const gdl_gglm_workspace * w)
{
	return w->gdata;	
}

const gdl_fview_wrapper *
gdl_gglm_workspace_tdata (const gdl_gglm_workspace * w)
{
	return w->tdata;	
}

const gdl_fview_wrapper *
gdl_gglm_workspace_odata (const gdl_gglm_workspace * w)
{
	return w->odata;	
}

const gdl_gmatrix_type *
gdl_gglm_workspace_gcode (const gdl_gglm_workspace * w)
{
	return w->gcode;	
}

static int
gdl_gglm_perform_test (gdl_gglm_workspace * w, gdl_gglm_model * model, gdl_gglm_parameter * params)
{
	size_t g, ng;
	const gdl_gglm_model_result * result;
	
	ng = gdl_gview_wrapper_locus_size (w->gdata);
			
	for (g = 0; g < ng; g++)
	{
		gdl_gglm_model_add_eligible (model, w->gdata, g);
		
		result = gdl_gglm_model_eval (model);
		
		if (w->logger)
		{
			gdl_gglm_model_fprintf (w->logger, model);
		}
		
		if (result->pval <= params->pheap)
		{
			gdl_list_push_back (w->heap, gdl_gglm_static_model_alloc (model), 0);
		}
		
		gdl_gglm_model_discard_eligible (model);
	}
}

static int
gdl_gglm_perform_forward (gdl_gglm_workspace * w, gdl_gglm_model * model, gdl_gglm_parameter * params)
{
	size_t g, ng, ni = 0;
	const gdl_gglm_model_result * result;
	
	ng = gdl_gview_wrapper_locus_size (w->gdata);
			
	for (g = 0; g < ng; g++)
	{
		gdl_gglm_model_add_eligible (model, w->gdata, g);
		
		result = gdl_gglm_model_eval (model);
		
		if (result->pval < params->fbpin)
		{
			if (w->logger)
			{
				fprintf (w->logger, "--\n");
				fprintf (w->logger, "  FORWARD: ADD LOCUS %d\n", g+1);
				fprintf (w->logger, "--\n");
				//gdl_gglm_model_fprintf (w->logger, model);
			}
			gdl_gglm_model_push_eligible (model);
			ni++;
		}
		else
		{
			gdl_gglm_model_discard_eligible (model);
		}
		
	}
	
	return ni;
}

static int
gdl_gglm_perform_backward (gdl_gglm_workspace * w, gdl_gglm_model * model, gdl_gglm_parameter * params)
{
	size_t g, ng, nr=0;
	const gdl_gglm_model_result * result;
	
	ng = gdl_gglm_model_size (model);
			
	for (g = 0; g < ng; g++)
	{
		gdl_gglm_model_move_to_eligible (model, ng-1-g);
		
		result = gdl_gglm_model_eval (model);
		
		if (result->pval < params->fbpout)
		{
			gdl_gglm_model_push_back (model, ng-1-g);
			if (w->logger)
			{
				fprintf (w->logger, "--\n");
				fprintf (w->logger, "  BACKWARD: KEEP LOCUS %d\n", ng-g);
				fprintf (w->logger, "--\n");
				//gdl_gglm_model_fprintf (w->logger, model);
			}
		}
		else
		{
			gdl_gglm_model_remove (model, ng-1-g);
			if (w->logger)
			{
				fprintf (w->logger, "--\n");
				fprintf (w->logger, "  BACKWARD: REMOVE LOCUS %d\n", ng-g);
				fprintf (w->logger, "--\n");
				//gdl_gglm_model_fprintf (w->logger, model);
			}
			nr++;
		}
	}
	
	return nr;
}

static int
gdl_gglm_perform_forward_backward (gdl_gglm_workspace * w, gdl_gglm_model * model, gdl_gglm_parameter * params)
{
	size_t iter;
	const gdl_gglm_model_result * result;
	
	iter=0;
	do
	{
		iter++;
		if (!gdl_gglm_perform_forward (w, model, params))
		{
			break;	
		}
		if (!gdl_gglm_perform_backward (w, model, params))
		{
			break;	
		}
	}
	while (iter < params->fbiter);
	
	result = gdl_gglm_model_eval (model);
	
	if (w->logger)
	{
		fprintf (w->logger, "--\n");
		fprintf (w->logger, " FORWARD-BACKWARD: final model (iter=%d)\n", iter);
		fprintf (w->logger, "--\n");
		gdl_gglm_model_fprintf (w->logger, model);
	}
}

int
gdl_gglm_perform (gdl_gglm_workspace * w, gdl_gglm_parameter * params)
{
	if (!params)
	{
		GDL_ERROR_VAL ("No parameter defined", GDL_EINVAL, GDL_EINVAL);		
	}
	else
	{
		size_t t, nt;
		gdl_gglm_model * model;
		
		w->gcode = params->gcode;
		
		nt = gdl_fview_wrapper_factor_size (w->tdata);
		
		// log parameter values...
		
		for (t = 0; t < nt; t++)
		{
			if (w->logger)
			{
				gdl_factor * factor = gdl_fview_wrapper_get_factor (w->tdata, t);
				fprintf (w->logger, "##\n");
				fprintf (w->logger, "# TRAIT [ %d ] [ %s ]\n", t+1, gdl_entity_get_name (factor));
				fprintf (w->logger, "##\n");
			}
			
			model = gdl_gglm_model_alloc (w->tdata, w->odata, w->gdata, params->gcode, t);
			
			if (params->mode == gdl_gglm_test)
			{
				gdl_gglm_perform_test (w, model, params);
			}
			else if (params->mode == gdl_gglm_forward_backward)
			{
				if (w->logger)
				{
					fprintf (w->logger, "--\n");
					fprintf (w->logger, "   Start Forward-Backward\n");
					fprintf (w->logger, "--\n");
				}
				gdl_gglm_perform_forward_backward (w, model, params);	
			}
			
			gdl_gglm_model_free (model);
		}
	}
}

gdl_list *
gdl_gglm_workspace_heap (const gdl_gglm_workspace * w)
{
	return w->heap;	
}

FILE * 
gdl_gglm_workspace_set_logger (gdl_gglm_workspace * w, FILE * stream)
{
	FILE * old = w->logger;
	
	w->logger = stream;
	
	return old;	
}

enum _gdl_gglm_mode
{
	GDL_GGLM_TEST,
	GDL_GGLM_FORWARD_BACKWARD
};	

static const gdl_gglm_mode _gdl_gglm_test = GDL_GGLM_TEST;
static const gdl_gglm_mode _gdl_gglm_forward_backward = GDL_GGLM_FORWARD_BACKWARD;

const gdl_gglm_mode * gdl_gglm_test = &_gdl_gglm_test;
const gdl_gglm_mode * gdl_gglm_forward_backward = &_gdl_gglm_forward_backward;
