/* 
 * gglm/model.c
 * 
 * Copyright (C) 2006 Jean-Baptiste Veyrieras
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <stdlib.h>
#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_statistics.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_fview_wrapper.h>
#include <gdl/gdl_gmatrix.h>
#include <gdl/gdl_fmatrix.h>
#include <gdl/gdl_multireg.h>
#include <gdl/gdl_gglm.h>

struct _gdl_gglm_model
{
	size_t    size;
	size_t    N;
	size_t    O;
	size_t    M0;
	size_t    M1;
	gdl_mask  * trait;
	gdl_mask  * factor;
	gdl_mask  * locus0;
	gdl_mask  * locus1;
	gdl_vector * y;
	gdl_matrix * X0;
	gdl_matrix * X1;
	gdl_gmatrix             * gdb0;
	gdl_gmatrix             * gdb1;
	gdl_fmatrix             * odb;
	gdl_fmatrix             * tdb;
	const gdl_gview_wrapper * gdata;
	const gdl_fview_wrapper * tdata;
	const gdl_fview_wrapper * odata;
	const gdl_gmatrix_type  * gcode;
	gdl_multireg_linear_workspace * wreg;
	gdl_boolean locus_update;
	gdl_boolean * exclude;
	size_t      * move;
	gdl_gglm_model_result * result;
};

static gdl_gglm_model_result *
gdl_gglm_model_result_alloc (const gdl_multireg_linear_workspace * w)
{
	size_t i;
	const gdl_vector * c;
	const gdl_matrix * cov;
	gdl_gglm_model_result * r;
	
	r = GDL_MALLOC (gdl_gglm_model_result, 1);
	
	r->df    = gdl_multireg_linear_df (w);
	r->df1   = gdl_multireg_linear_eval_df (w);
	r->rsq   = gdl_multireg_linear_rsquare (w);
	r->rsq1  = gdl_multireg_linear_eval_rsquare (w);
	r->fstat = gdl_multireg_linear_eval_fstat (w);
	r->pval  = gdl_multireg_linear_eval_pvalue (w);
	
	c = gdl_multireg_linear_coefficients (w);
	r->c  = gdl_vector_alloc (c->size);
	gdl_vector_memcpy (r->c, c);
	
	r->sd = gdl_vector_alloc (c->size);
	cov = gdl_multireg_linear_covariance (w);
	
	for (i = 0; i < c->size; i++)
	{
		gdl_vector_set (r->sd, i, sqrt (gdl_matrix_get (cov, i, i)));	
	}
	
	r->N = r->df + r->c->size;
	r->M = r->c->size;
	
	return r;
}

void
gdl_gglm_model_result_free (gdl_gglm_model_result * r)
{
	if (r)
	{
		gdl_vector_free (r->c);
		gdl_vector_free (r->sd);
		GDL_FREE (r);
	}
}
/*
 * This assumes that tdata, odata and gdata share the same accession list
 * in the same order of course...
 */
static void
_gdl_gglm_model_init_mask (gdl_gglm_model * m, const gdl_fview_wrapper * tdata, const gdl_fview_wrapper * odata, size_t t)
{
	gdl_mask * tmp;
	gdl_entity_mask * accession;
	
	tmp = gdl_mask_alloc ();
	
	gdl_mask_add_idx (tmp, GDL_FACTOR, t);
	m->trait = gdl_fview_wrapper_mask_not_informative (tdata, tmp, GDL_ACCESSION);
	
	accession = gdl_mask_get_clone (m->trait, GDL_ACCESSION);
	
	if (odata)
	{
		m->factor = gdl_mask_alloc ();
		gdl_mask_set (m->factor, GDL_ACCESSION, accession, 0);
	}
	
	m->locus0 = gdl_mask_alloc ();
	gdl_mask_set (m->locus0, GDL_ACCESSION, accession, 1);
	
	m->locus1 = gdl_mask_alloc ();
	gdl_mask_set (m->locus1, GDL_ACCESSION, accession, 0);
	
	gdl_mask_free (tmp);
}

static void
_gdl_gglm_model_init_matrix (gdl_gglm_model * m, const gdl_fview_wrapper * tdata, const gdl_fview_wrapper * odata)
{
	m->tdb  = gdl_fmatrix_wrapper_mask_alloc (tdata, m->trait, gdl_false);
	m->size = gdl_fmatrix_accession_size (m->tdb);
	if (odata)
	{
		m->odb = gdl_fmatrix_wrapper_mask_alloc (odata, m->factor, gdl_false);
		m->O   = gdl_fmatrix_column_size (m->odb);
	}
	m->y = gdl_vector_alloc (m->size);
	m->M0 = m->M1 = 0;	
}

gdl_gglm_model *
gdl_gglm_model_alloc (const gdl_fview_wrapper * tdata,
                      const gdl_fview_wrapper * odata,
                      const gdl_gview_wrapper * gdata,
                      const gdl_gmatrix_type  * gcode,
                      size_t t)
{
	
	gdl_gglm_model * m;
	
	m = GDL_CALLOC (gdl_gglm_model, 1);
	
	_gdl_gglm_model_init_mask (m, tdata, odata, t);
	
	_gdl_gglm_model_init_matrix (m, tdata, odata);
	
	m->gcode = gcode;
	m->tdata = tdata;
	m->odata = odata;
	m->gdata = gdata;
	
	return m;
}

void
gdl_gglm_model_free (gdl_gglm_model * m)
{
	if (m)
	{
		gdl_mask_free (m->trait);
		gdl_mask_free (m->factor);
	   gdl_mask_free (m->locus0);
		gdl_mask_free (m->locus1);
	   gdl_vector_free (m->y);
	   gdl_matrix_free (m->X0);
	   gdl_matrix_free (m->X1);
	   gdl_gmatrix_free (m->gdb0);
	   gdl_gmatrix_free (m->gdb1);
	   gdl_fmatrix_free (m->tdb);
	   gdl_fmatrix_free (m->odb);
	   gdl_multireg_linear_workspace_free (m->wreg);
	   gdl_gglm_model_result_free (m->result);
	   GDL_FREE (m->exclude);
	   GDL_FREE (m->move);
	   GDL_FREE (m);
	}	
}

size_t
gdl_gglm_model_size (const gdl_gglm_model * m)
{
	return gdl_mask_size (m->locus0, GDL_LOCUS);
}

int
gdl_gglm_model_add_eligible (gdl_gglm_model * m, const gdl_gview_wrapper * gdata, size_t g)
{
	return gdl_mask_add (m->locus1, gdl_gview_wrapper_get_locus (gdata, g));
}

int
gdl_gglm_model_discard_eligible (gdl_gglm_model * m)
{
	gdl_mask_reset (m->locus1, GDL_LOCUS);
}

int
gdl_gglm_model_push_eligible (gdl_gglm_model * m)
{
	size_t i, ne, ni;
	gdl_boolean * exclude;
	size_t      * move;
	
	ne = gdl_mask_size (m->locus1, GDL_LOCUS);
	ni = gdl_mask_size (m->locus0, GDL_LOCUS);
	exclude = GDL_CALLOC (gdl_boolean, ni+ne);
	move    = GDL_MALLOC (gdl_boolean, ni+ne);
	if (ni) 
	{
		memcpy (exclude, m->exclude, sizeof (gdl_boolean)*ni);
		memcpy (move, m->move, sizeof (size_t *)*ni);
		GDL_FREE (m->exclude);
		GDL_FREE (m->move);
	}
	m->exclude = exclude;
	m->move    = move;
	for (i = 0; i < ne; i++)
	{
		//printf ("ADD %d %d\n", i, gdl_mask_get_idx (m->locus1, GDL_LOCUS, i));
		gdl_mask_add_idx (m->locus0, GDL_LOCUS, gdl_mask_get_idx (m->locus1, GDL_LOCUS, i));
	}
	gdl_mask_reset (m->locus1, GDL_LOCUS);
	m->locus_update = gdl_true;
}

int
gdl_gglm_model_move_to_eligible (gdl_gglm_model * m, size_t g)
{
	m->exclude[g]    = gdl_true;
	m->move[g]       = gdl_mask_add_idx (m->locus1, GDL_LOCUS, gdl_mask_get_idx (m->locus0, GDL_LOCUS, g));
	//m->locus_update  = gdl_true;
}

int
gdl_gglm_model_push_back (gdl_gglm_model * m, size_t g)
{
	m->exclude[g]    = gdl_false;
	//m->locus_update  = gdl_true;
	gdl_mask_remove (m->locus1, GDL_LOCUS, m->move[g]);
}

int
gdl_gglm_model_remove (gdl_gglm_model * m, size_t g)
{
	size_t i, ne = gdl_mask_size (m->locus0, GDL_LOCUS);
	for (i = g; i < ne - 1; i++)
	{
		m->exclude[i] = m->exclude[i+1];
		m->move[i]    = m->move[i+1];
	}
	gdl_mask_remove (m->locus0, GDL_LOCUS, g);
	m->locus_update = gdl_true;
}

static gdl_gglm_model_result *
gdl_gglm_model_multreg (gdl_gglm_model * model)
{
	const size_t N = model->N;
	const size_t M = model->O + model->M0 + model->M1 + 1;
	gdl_vector * y;
	gdl_matrix * X0, * X1;
	gdl_multireg_linear_workspace * wreg;
	
	if (model->N < model->size)
	{
		gdl_matrix_view VX1, VX0;
		gdl_vector_view vy;
		
		vy  = gdl_vector_subvector (model->y, 0, model->N);
		y   = &(vy.vector);
		
		if (model->M1)
		{
	   	VX0 = gdl_matrix_submatrix (model->X0, 0, 0, model->N, model->O + model->M0 + 1);
			X0  = &(VX0.matrix);
			VX1 = gdl_matrix_submatrix (model->X1, 0, 0, model->N, model->M1);
			X1  = &(VX1.matrix);	
		}
		else
		{
			VX0 = gdl_matrix_submatrix (model->X0, 0, 0, model->N, model->O + 1);
			X0  = &(VX0.matrix);
			VX1 = gdl_matrix_submatrix (model->X1, 0, 0, model->N, model->M0);
			X1  = &(VX1.matrix);
		}
	}
	else
	{
		X0 = model->X0;
		X1 = model->X1;
		y  = model->y;
	}
	
	if (model->wreg==0)
	{
		model->wreg = gdl_multireg_linear_workspace_alloc (N, M);
	}
	else if (   N > gdl_multireg_linear_workspace_size1 (model->wreg)
	         || M > gdl_multireg_linear_workspace_size2 (model->wreg))
	{
		gdl_multireg_linear_workspace_free (model->wreg);
		model->wreg = gdl_multireg_linear_workspace_alloc (N, M);
	}
	
	if (model->M1 || model->M0)
	{
		gdl_multireg_linear_eval (model->wreg, X0, X1, y);
	}
	else
	{
		gdl_multireg_linear_perform (model->wreg, X0, y);	
	}
	
	return gdl_gglm_model_result_alloc (model->wreg);
}

static int
gdl_gglm_model_system_locus (gdl_gglm_model * model, size_t i, size_t ii, size_t * jj, size_t mode)
{
	gdl_gmatrix * data;
	gdl_matrix  * X;
	gdl_boolean * exclude;
	
	switch (mode)
	{
		case 0 :
			data    = model->gdb0;
			X       = (model->M1) ? model->X0 : model->X1;
			exclude = model->exclude;
			break;
		case 1 :
			data    = model->gdb1;
			X       = model->X1;
			exclude = NULL;
			break;
	}
	
	if (data)
	{
		size_t j, k, nc;
		gdl_gvalues * x;
		gdl_gvalue  * v;
		
		nc = gdl_gmatrix_locus_size (data);
		
		for (j = 0; j < nc; j++)
		{
			if (exclude && exclude[j])
			{
				continue;
			}
			if (gdl_gmatrix_locus_is_missing (data, i, j))
			{
				return GDL_CONTINUE;
			}
			x = gdl_gmatrix_locus_get (data, i, j);
			for (k = 0; k < x->size; k++)
			{
				v = x->values[k];
				gdl_matrix_set (X, ii, *jj + v->idx, v->value);
			}
			gdl_gvalues_free (x);
			(*jj) += gdl_gmatrix_locus_column_size (data, j);
		}
		
		return GDL_SUCCESS;
	}
	else
	{
		return GDL_SUCCESS;	
	}
}

static int
gdl_gglm_model_system_factor (gdl_gglm_model * model, size_t i, size_t ii, size_t * jj)
{
	if (model->odb)
	{
		size_t j, k, nc;
		gdl_gvalues * x;
		gdl_gvalue  * v;
		
		nc = gdl_fmatrix_factor_size (model->odb);
		
		for (j = 0; j < nc; j++)
		{
			if (gdl_fmatrix_factor_is_missing (model->odb, i, j))
			{
				return GDL_CONTINUE;
			}
			x = gdl_fmatrix_factor_get (model->odb, i, j);
			for (k = 0; k < x->size; k++)
			{
				v = x->values[k];
				gdl_matrix_set (model->X0, ii, *jj + v->idx, v->value);
			}
			gdl_gvalues_free (x);
			(*jj) += gdl_fmatrix_factor_column_size (model->odb, j);
		}
		
		return GDL_SUCCESS;
	}
	else
	{
		return GDL_SUCCESS;
	}
}

int
gdl_gglm_model_system (gdl_gglm_model * model)
{
	size_t i, ii, j, jj, M0 = model->M0;
	
	gdl_matrix_free (model->X0);
	gdl_matrix_free (model->X1);
	model->X1 = NULL;
	
	for (i = 0; i < gdl_mask_size (model->locus0, GDL_LOCUS); i++)
	{
		if (model->exclude[i])
		{
			M0 -= gdl_gmatrix_locus_column_size (model->gdb0, i);
		}	
	}
	
	model->M0 = M0;
	
	if (model->M1)
	{
		model->X0 = gdl_matrix_alloc (model->size, model->O + model->M0 + 1);
		model->X1 = gdl_matrix_alloc (model->size, model->M1);
	}
	else
	{
		model->X0 = gdl_matrix_alloc (model->size, model->O + 1);
		if (model->M0)
		{
			model->X1 = gdl_matrix_alloc (model->size, model->M0);
		}
	}
	
	for (ii = i = 0; i < model->size; i++)
	{
		gdl_vector_set (model->y, ii, gdl_fmatrix_get (model->tdb, i, 0));
		gdl_matrix_set (model->X0, ii, 0, 1.0); // intercept
		j = 1;
		if (model->O && gdl_gglm_model_system_factor (model, i, ii, &j)==GDL_CONTINUE)
		{
			continue;
		}
		j = (model->M1) ? j : 0;
		if (model->M0 && gdl_gglm_model_system_locus (model, i, ii, &j, 0)==GDL_CONTINUE)
		{
			continue;
		}
		j = 0;
		if (model->M1 && gdl_gglm_model_system_locus (model, i, ii, &j, 1)==GDL_CONTINUE)
		{
			continue;
		}
		ii++;
	}
	model->N = ii;
}

static int
gdl_gglm_model_create_gdb0 (gdl_gglm_model * m)
{
	if (m->locus_update && gdl_mask_size (m->locus0, GDL_LOCUS))
	{	
		gdl_gmatrix_free (m->gdb0);
		m->gdb0 = gdl_gmatrix_wrapper_mask_alloc (m->gcode, m->gdata, m->locus0, gdl_false);
		m->M0   = gdl_gmatrix_column_size (m->gdb0);
	}
	else if (gdl_mask_size (m->locus0, GDL_LOCUS)==0)
	{
		m->M0 = 0;
	}
}

static int
gdl_gglm_model_create_gdb1 (gdl_gglm_model * m)
{
	if (gdl_mask_size (m->locus1, GDL_LOCUS))
	{
		gdl_gmatrix_free (m->gdb1);
		m->gdb1 = gdl_gmatrix_wrapper_mask_alloc (m->gcode, m->gdata, m->locus1, gdl_false);
		m->M1   = gdl_gmatrix_column_size (m->gdb1);		
	}
	else
	{
		m->M1 = 0;	
	}
}

const gdl_gglm_model_result *
gdl_gglm_model_eval (gdl_gglm_model * m)
{
	gdl_gglm_model_create_gdb0 (m);
	
	gdl_gglm_model_create_gdb1 (m);
	
	gdl_gglm_model_system (m);
	
	gdl_gglm_model_result_free (m->result);
	
	m->result = gdl_gglm_model_multreg (m);
	
	return m->result;
}

gdl_gglm_static_model *
gdl_gglm_static_model_alloc (const gdl_gglm_model * m)
{
	size_t i, j;
	gdl_gglm_static_model * s;
	
	s = GDL_CALLOC (gdl_gglm_static_model, 1);
	
	s->N = m->result->N;
	s->M = m->result->M;
	if (m->odata)
	{
		s->no = gdl_fview_wrapper_factor_size (m->odata);//gdl_mask_size (m->factor, GDL_FACTOR);
	}
	s->nl = gdl_mask_size (m->locus0, GDL_LOCUS) + gdl_mask_size (m->locus1, GDL_LOCUS);
	s->trait = gdl_mask_get_idx (m->trait, GDL_FACTOR, 0);
	if (s->no)
	{
		s->other = GDL_MALLOC (size_t, s->no);
		for (i = 0; i < s->no; i++)
		{
			s->other[i] = i;//gdl_mask_get_idx (m->factor, GDL_FACTOR, i);	
		}
	}
	if (s->nl)
	{
		s->locus = GDL_MALLOC (size_t, s->nl);
		for (i = 0, j = 0; j < gdl_mask_size (m->locus0, GDL_LOCUS); j++,  i++)
		{
			s->locus[i] = gdl_mask_get_idx (m->locus0, GDL_LOCUS, j);	
		}
		for (j = 0; j < gdl_mask_size (m->locus1, GDL_LOCUS); j++,  i++)
		{
			s->locus[i] = gdl_mask_get_idx (m->locus1, GDL_LOCUS, j);	
		}
	}
	s->df    = m->result->df;
	s->df1   = m->result->df1;
	s->rsq   = m->result->rsq;
	s->rsq1  = m->result->rsq1;
	s->fstat = m->result->fstat;
	s->pval  = m->result->pval;
	s->coeff = gdl_vector_alloc (s->M);
	gdl_vector_memcpy (s->coeff, m->result->c);
	s->sd = gdl_vector_alloc (s->M);
	gdl_vector_memcpy (s->sd, m->result->sd);
		
	return s;
}

void
gdl_gglm_static_model_free (gdl_gglm_static_model * s)
{
	if (s)
	{
		GDL_FREE (s->other);
		GDL_FREE (s->locus);
		gdl_vector_free (s->coeff);
		gdl_vector_free (s->sd);
		GDL_FREE (s);
	}	
}

gdl_gglm_static_model *
gdl_gglm_static_model_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		gdl_gglm_static_model * s;
		
		s = GDL_CALLOC (gdl_gglm_static_model, 1);
		
		status = fread (&(s->N), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(s->M), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(s->no), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(s->nl), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(s->trait), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		if (s->no)
		{
			s->other = GDL_MALLOC (size_t, s->no);
			status   = fread (s->other, sizeof (size_t), s->no, stream);
			GDL_FREAD_STATUS (status, s->no);
		}
		if (s->nl)
		{
			s->locus = GDL_MALLOC (size_t, s->nl);
			status   = fread (s->locus, sizeof (size_t), s->nl, stream);
			GDL_FREAD_STATUS (status, s->nl);
		}
		status = fread (&(s->df), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(s->df1), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(s->rsq), sizeof (double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(s->rsq1), sizeof (double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(s->fstat), sizeof (double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(s->pval), sizeof (double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		s->coeff = gdl_vector_fread (stream);
		GDL_FREAD_STATUS (s->coeff!=0, 1);
		s->sd = gdl_vector_fread (stream);
		GDL_FREAD_STATUS (s->sd!=0, 1);
		
		return s;
	}
	
	return NULL;
}

int
gdl_gglm_static_model_fwrite (FILE * stream, const gdl_gglm_static_model * s)
{
	if (stream && s)
	{
		int status;
		
		status = fwrite (&(s->N), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(s->M), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(s->no), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(s->nl), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(s->trait), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		if (s->no)
		{
			status = fwrite (s->other, sizeof (size_t), s->no, stream);
			GDL_FWRITE_STATUS (status, s->no);
		}
		if (s->nl)
		{
			status = fwrite (s->locus, sizeof (size_t), s->nl, stream);
			GDL_FWRITE_STATUS (status, s->nl);
		}
		status = fwrite (&(s->df), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(s->df1), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(s->rsq), sizeof (double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(s->rsq1), sizeof (double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(s->fstat), sizeof (double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(s->pval), sizeof (double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = gdl_vector_fwrite (stream, s->coeff);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_vector_fwrite (stream, s->sd);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}

static gdl_boolean
gdl_gglm_model_factor_fprintf (FILE * stream, const gdl_gglm_model * model, gdl_boolean add)
{
	if (model->odata)
	{
		size_t i, j;
		gdl_factor * factor;
		
		for (i = 0; i < gdl_fview_wrapper_factor_size (model->odata); i++)
		{
			if ((!i && add) || i) fprintf (stream, " +"); 
			factor = gdl_fview_wrapper_get_factor (model->odata, i);
			fprintf (stream, " [ %s", gdl_entity_get_name (factor));
			if (gdl_factor_get_type (factor) != gdl_factor_continuous)
			{
				fprintf (stream, ":");
				for (j = 0; j < gdl_factor_size (factor) - 1; j++)
				{
					if (j) fprintf (stream, " +");
					fprintf (stream, " %s", gdl_entity_get_name (gdl_factor_get(factor, j)));
				}
			}
			fprintf (stream, " ]");
		}
		return gdl_true;
	}
	return add;
}

static gdl_boolean
gdl_gglm_model_locus_fprintf (FILE * stream, const gdl_gglm_model * model, size_t mode, gdl_boolean add)
{
	if (model->gdata)
	{
		size_t i, j;
		gdl_mask    * mask;
		gdl_locus   * locus;
		gdl_boolean * exclude;
		
		switch (mode)
		{
			case 0 :
			   mask    = model->locus0;
			   exclude = model->exclude; 
				break;
			case 1 :
			   mask = model->locus1;
			   exclude = 0;
				break;
		}
		
		for (i = 0; i < gdl_mask_size (mask, GDL_LOCUS); i++)
		{
			if (exclude && exclude[i])
			{
				continue;
			}
			if ((!i && add) || i) fprintf (stream, " +"); 
			locus = gdl_gview_wrapper_get_locus (model->gdata, gdl_mask_get_idx (mask, GDL_LOCUS, i));
			fprintf (stream, " [ %s", gdl_entity_get_name (locus));
			if (model->gcode == gdl_gmatrix_allele)
			{
				fprintf (stream, ":");
				for (j = 0; j < gdl_locus_allele (locus) - 1; j++)
				{
					if (j) fprintf (stream, " +");
					fprintf (stream, " %s", gdl_entity_get_name (gdl_locus_get_allele(locus, j)));
				}
			}
			else if (model->gcode == gdl_gmatrix_genotype)
			{
				fprintf (stream, ":");
				for (j = 0; j < gdl_locus_genotype (locus) - 1; j++)
				{
					if (j) fprintf (stream, " +");
					fprintf (stream, " %s", gdl_entity_get_name (gdl_locus_get_genotype(locus, j)));
				}
			}
			fprintf (stream, " ]");
		}
		
		return (gdl_mask_size (mask, GDL_LOCUS)) ? gdl_true : add;
	}
	
	return add;
}

#define PRINT_COEF(a, b)(fprintf (stream, "\t%.3f\t(+/- %.3f)\n", (a), (b)))

static int
gdl_gglm_model_factor_result_fprintf (FILE * stream, const gdl_gglm_model * model, const gdl_gglm_model_result * result, size_t * ii)
{
	if (model->odata)
	{
		size_t i, j;
		gdl_factor * factor;
		
		for (i = 0; i < gdl_fview_wrapper_factor_size (model->odata); i++)
		{
			factor = gdl_fview_wrapper_get_factor (model->odata, i);
			if (gdl_factor_get_type (factor) == gdl_factor_continuous)
			{
				fprintf (stream, "   %s\t-", gdl_entity_get_name (factor));
				PRINT_COEF(gdl_vector_get (result->c, *ii), gdl_vector_get (result->sd, *ii));
				(*ii)++; 
			}
			else
			{
				for (j = 0; j < gdl_factor_size (factor) - 1; j++)
				{
					fprintf (stream, "   %s\t%s", gdl_entity_get_name (factor), gdl_entity_get_name (gdl_factor_get(factor, j)));
					PRINT_COEF (gdl_vector_get (result->c, *ii), gdl_vector_get (result->sd, *ii));
					(*ii)++;
				}
			}
		}
	}
}

static int
gdl_gglm_model_locus_result_fprintf (FILE * stream, const gdl_gglm_model * model, const gdl_gglm_model_result * result, size_t * ii, size_t mode)
{
	if (model->gdata)
	{
		size_t i, j;
		gdl_mask  * mask;
		gdl_locus * locus;
		gdl_boolean * exclude;
		
		switch (mode)
		{
			case 0 :
			   mask    = model->locus0;
			   exclude = model->exclude;
				break;
			case 1 :
			   mask    = model->locus1;
			   exclude = 0;
				break;
		}
		
		for (i = 0; i < gdl_mask_size (mask, GDL_LOCUS); i++)
		{
			if (exclude && exclude[i])
			{
				continue;
			}
			locus = gdl_gview_wrapper_get_locus (model->gdata, gdl_mask_get_idx (mask, GDL_LOCUS, i));
			if (model->gcode == gdl_gmatrix_allele)
			{
				for (j = 0; j < gdl_locus_allele (locus) - 1; j++)
				{
					fprintf (stream, "   %s\t%s", gdl_entity_get_name (locus), gdl_entity_get_name (gdl_locus_get_allele (locus, j)));
					PRINT_COEF (gdl_vector_get (result->c, *ii), gdl_vector_get (result->sd, *ii));
					(*ii)++;
				}
			}
			else if (model->gcode == gdl_gmatrix_genotype)
			{
				for (j = 0; j < gdl_locus_genotype (locus) - 1; j++)
				{
					fprintf (stream, "   %s\t%s", gdl_entity_get_name (locus), gdl_entity_get_name (gdl_locus_get_genotype (locus, j)));
					PRINT_COEF (gdl_vector_get (result->c, *ii), gdl_vector_get (result->sd, *ii));
					(*ii)++;
				}
			}
		}
	}	
}

int
gdl_gglm_model_fprintf (FILE * stream, const gdl_gglm_model * model)
{
	if (stream && model)
	{
		size_t i, ii;
		double x;
		gdl_factor * factor;
		const gdl_gglm_model_result * result = model->result;
		
//		fprintf (stream, "--\n");
//		fprintf (stream, "GGLM Model\n");
		fprintf (stream, "--\n");
		factor = gdl_fview_wrapper_get_factor (model->tdata, gdl_mask_get_idx (model->trait, GDL_FACTOR, 0));
		fprintf (stream,"  [ %s ] ~", gdl_entity_get_name (factor));
		gdl_gglm_model_locus_fprintf (stream, model, 1, 
          gdl_gglm_model_locus_fprintf (stream, model, 0,
              gdl_gglm_model_factor_fprintf (stream, model, gdl_false)));
		fprintf (stream, "\n--\n");
		fprintf (stream, "   Number of individuals      %d\n", result->N);
		fprintf (stream, "   Number of variables        %d\n", result->M);
		fprintf (stream, "   Degrees of freedom (H1)    %d\n", result->df);
		fprintf (stream, "   Degrees of freedom (H1:H0) %d\n", result->df1);
		fprintf (stream, "   R-square (H1)              %1.3f\n", result->rsq);
		fprintf (stream, "   R-square (H1:H0)           %1.3f\n", result->rsq1);
		fprintf (stream, "   F-stat (H1:H0)             %.2f\n", result->fstat);
		fprintf (stream, "   P-value (H1:H0)            %e ", result->pval);
		i = 0;
		x = result->pval;
		while (x <= 0.05)
		{
			if (!i)
			{
				fprintf (stream, "(");
			}
			i++;
			x *= 100;
			fprintf (stream, "*");
			if (i==3 || x > 0.05) 
			{
				fprintf (stream, ")");
				break;
			}
		}
		fprintf (stream, "\n");
		fprintf (stream, "--\n");
		fprintf (stream, "   Intercept\t-");
		PRINT_COEF(gdl_vector_get (result->c, 0), gdl_vector_get (result->sd, 0));
		ii = 1;
		gdl_gglm_model_factor_result_fprintf (stream, model, result, &ii);
		gdl_gglm_model_locus_result_fprintf (stream, model, result, &ii, 0);
		gdl_gglm_model_locus_result_fprintf (stream, model, result, &ii, 1);
		
		return GDL_SUCCESS;
	}

	return GDL_EINVAL;	
}

#undef PRINT_COEF
