/*  
 *  gmap/extract.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:41 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_sort.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_gmap.h>

gdl_gmap *
gdl_gmap_extract (const gdl_gmap * gmap, 
                    gdl_locus_ptr loci[],
                    gdl_boolean ismap[],
                    size_t nl,
                    size_t owner)
{
	size_t ig, ic, il, im, ng, nc, * idx, all = 0;
	int * pos;
	gdl_gmap * egmap;
	
	idx = GDL_CALLOC (size_t, nl);
	pos = GDL_CALLOC (int, nl);
	
	ng = gdl_gmap_size (gmap);
	
	egmap = gdl_gmap_alloc ();
	
	for (ig = 0; ig < ng; ig++)
	{
		gdl_genome * egenome;
		gdl_genome * genome 
		   = gdl_gmap_get (gmap, ig);
		
		egenome 
			= gdl_genome_new (genome->name);
			
		nc = gdl_genome_size (genome);
		
		for (ic = 0; ic < nc; ic++)
		{
			gdl_chromosome * echrom;
			gdl_chromosome * chrom
			   = gdl_genome_get (genome, ic);
			   
			echrom = gdl_chromosome_new (chrom->name);
			
			for (im = il = 0; il < nl; il++)
			{
				
				if (!ismap[il])
				{
					int ip
						= gdl_chromosome_search (chrom, loci[il]);
					
					if (ip>=0)
					{
						ismap[il] = gdl_true;
						idx[im] = il;
						pos[im] = ip;
						im++;
						all++;
					}					
				}
			}
			
			if (im)
			{
				// sort locus index according to the positions
				
				size_t * pidx = GDL_CALLOC (size_t, im);
				
				gdl_sort_int_index (pidx, pos, 1, im);
				
				gdl_chromosome_push (echrom, loci[idx[pidx[0]]], owner);
				
				for (il = 1; il < im; il++)
				{
					const gdl_gdistance * d = 
						gdl_chromosome_search_distance (chrom, loci[idx[pidx[il-1]]], loci[idx[pidx[il]]]);
					
					gdl_chromosome_add (echrom, loci[idx[pidx[il]]], d, owner);
				}
				
				gdl_genome_add (egenome, echrom);
				
				GDL_FREE (pidx);
			}
			else
			{
				gdl_chromosome_free (echrom);
			}
			if (all == nl)
				break;
		}
		
		if (gdl_genome_size (egenome))
		{
			gdl_gmap_add (egmap, egenome);
		}
		else
		{
			gdl_genome_free (egenome);
		}
	}
	
	if (!gdl_gmap_size (egmap))
	{
		gdl_gmap_free (egmap);
		egmap = NULL;
	}
	
	GDL_FREE (idx);
	GDL_FREE (pos);
	
	return egmap;	
}
