/*  
 * 	gmap/test.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:41 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_util.h>
#include <gdl/gdl_test.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_gmap.h>
#include <gdl/gdl_gmap_reader.h>

static const int NGENO  = 2;
static const int NCHROM = 2;
static const int NLOCUS = 5;

static gdl_chromosome *
test_chromosome (const char * name)
{
	size_t i;
	gdl_chromosome * c 
		= gdl_chromosome_new (name);
	for (i = 0; i < NLOCUS; i++)
	{
		gdl_string * name = gdl_string_sprintf ("locus_%d", i);
		gdl_locus * l = gdl_locus_new (name);
		gdl_gdistance d;
		
		d.value = 0.1;
		d.type  = gdl_gdistance_morgan;
		
		if (i)
		{
			gdl_chromosome_add (c, l, &d, 1);
		}
		else
		{
			gdl_chromosome_push (c, l, 1);
		}
		
		gdl_string_free (name);
	}
	
	return c;
}

static gdl_genome *
test_genome (const char * name)
{
	size_t i;
	gdl_genome * g = gdl_genome_new (name);
	
	for (i = 0; i < NCHROM; i++)
	{
		gdl_string * name  = gdl_string_sprintf ("chromosome_%d", i);
		
		gdl_chromosome * c = test_chromosome (name);
		
		size_t j = gdl_genome_add (g, c);
		
		gdl_string_free (name);
	}
	
	return g;
}

static gdl_gmap *
test_gmap ()
{
	size_t i;
	gdl_gmap * gmap = gdl_gmap_alloc ();
	
	for (i = 0; i < NGENO; i++)
	{
		gdl_string * name  = gdl_string_sprintf ("genome_%d", i);
		
		gdl_genome * genome = test_genome (name);
		
		size_t j = gdl_gmap_add (gmap, genome);
		
		gdl_string_free (name);
	}
	
	return gmap;
}

static void
test_alloc (void)
{
	gdl_gmap * gmap = test_gmap ();
	gdl_test (gmap == NULL, "gdl_gmap_alloc returns valid pointer");
	gdl_gmap_free (gmap);
}

static void
test_fwrite (void)
{
	FILE * stream;
	
	gdl_gmap * g = test_gmap ();
		
	stream = gdl_fileopen ("test.dat", "w");
		
	gdl_test (gdl_gmap_fwrite (stream, g) != GDL_SUCCESS, "gdl_gmap_fwrite returns ok");
		
	gdl_fileclose ("test.dat", stream);
		
	gdl_gmap_free (g);
}

static void
test_fread (void)
{
	FILE * stream;
	gdl_gmap * g;
	
	stream = gdl_fileopen ("test.dat", "r");
		
	g = gdl_gmap_fread (stream);
		
	gdl_test (g == NULL, "gdl_gmap_fread returns ok");
		
	gdl_fileclose ("test.dat", stream);
		
	gdl_gmap_free (g);
}

static void
test_reader (void)
{
	gdl_gmap_reader * reader;
	
	reader = gdl_gmap_reader_alloc (gdl_gmap_reader_absolute);
	
	gdl_gmap_reader_open (reader, "examples/short.txt");
	
	gdl_gmap_reader_parse (reader);
	
	gdl_gmap_reader_close (reader);
	
	gdl_gmap_reader_free (reader);
	
	reader = gdl_gmap_reader_alloc (gdl_gmap_reader_relative);
	
	gdl_gmap_reader_open (reader, "examples/relative.txt");
	
	if (gdl_gmap_reader_parse (reader) == NULL)
	{
		fprintf (stderr, "[ ERROR ] %s\n", reader->error);	
	}
	
	gdl_gmap_reader_close (reader);
	
	gdl_gmap_reader_free (reader);
}

static void
test_extract (void)
{
	size_t nl;
	
	gdl_gmap_reader * reader;
	
	gdl_gmap * refmap, * shortmap, * emap;
	
	gdl_locus ** loci;
	
	gdl_boolean * ismap;
	
	reader = gdl_gmap_reader_alloc (gdl_gmap_reader_absolute);
	
	gdl_gmap_reader_open (reader, "examples/short.txt");
	
	shortmap = gdl_gmap_reader_parse (reader);
	
	gdl_gmap_reader_close (reader);
	
	gdl_gmap_reader_free (reader);
	
	reader = gdl_gmap_reader_alloc (gdl_gmap_reader_absolute);
	
	gdl_gmap_reader_open (reader, "examples/refmap.txt");
	
	refmap = gdl_gmap_reader_parse (reader);
	
	gdl_gmap_reader_close (reader);
	
	gdl_gmap_reader_free (reader);
	
	loci = gdl_gmap_loci (shortmap, &nl);
	
	ismap = GDL_CALLOC (gdl_boolean, nl);
	
	emap = gdl_gmap_extract (refmap, 
                      loci,
                      ismap,
                      nl,
                      0);
	
	GDL_FREE (loci);	
	GDL_FREE (ismap);
	
	gdl_genome_fprintf (stdout, gdl_gmap_get (emap, 0));
	
	gdl_gmap_free (refmap);
	gdl_gmap_free (shortmap);
	gdl_gmap_free (emap);
}

int
main (void)
{
//	test_alloc ();
//	test_fwrite ();
//	test_fread ();
//test_reader ();
test_extract ();
	exit (gdl_test_summary());
}
