/*  
 *  gmatrix/allele.c
 *  
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:53 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_hview.h>
#include <gdl/gdl_gmatrix.h>

static void
_gdl_gmatrix_locus_push (gdl_gmatrix * gm, size_t i, size_t j, const gdl_gvalues * x, gdl_boolean last_allele)
{
	size_t k, jj;
	double y;
	
	if (x)
	{
		for (k = 0; k < x->size; k++)
		{
			if ((x->values[k]->idx < gm->nc[j]) || last_allele)
			{
				jj = gm->tnc[j] + x->values[k]->idx;
				y  = gdl_matrix_get (gm->data, i, jj);
				y += x->values[k]->value;
				gdl_matrix_set (gm->data, i, jj, y);
			}
		}
	}
	else
	{
		gdl_gmatrix_locus_set_missing (gm, i, j);
	}
} 

static int
gdl_gmatrix_allele_alloc (gdl_gmatrix * gm, const gdl_gview * gview, const gdl_mask * mask, const gdl_clustering * clustering, gdl_boolean last_allele)
{
	size_t i, j, k, l, nr, nl, np, nn;
	gdl_locus * locus;
	double t;
	gdl_gvalues_get * gbuf;
	const gdl_gvalues   * gv;
	
	np = gdl_gview_ploidy (gview);
	nr = nn = GDL_GVIEW_ACCESSION_SIZE (gview, mask);
	if (clustering)
	{
		nr = gdl_clustering_nclust (clustering);
	}
	nl = GDL_GVIEW_LOCUS_SIZE (gview, mask);
	
	gm->size1 = nr;
	gm->nl    = nl;
	gm->rw    = gdl_vector_alloc (nr);
	gm->nc    = GDL_MALLOC (size_t, nl);
	gm->tnc   = GDL_MALLOC (size_t, nl);
	gm->tnc[0] = 0;
	
	for (i = 0; i < nl; i++)
	{
		locus     = GDL_GVIEW_GET_LOCUS (gview, mask, i);
		gm->nc[i] = gdl_locus_allele (locus);
		if ((gm->nc[i]==1 && !last_allele) || gm->nc[i]==0)
		{
			GDL_ERROR_VAL (gdl_string_sprintf ("Locus %s is not polymorphe", gdl_entity_get_name (locus)),
			               GDL_FAILURE,
			               GDL_FAILURE);
		}
		if (!last_allele)
		{
			(gm->nc[i])--;
		}
		if (i)
		{
			gm->tnc[i] = gm->tnc[i-1] + gm->nc[i-1];
		}
	}
	
	gm->size2 = gm->tnc[i-1] + gm->nc[i-1];
	
	gm->data  = gdl_matrix_calloc (gm->size1, gm->size2);
	
	gbuf = GDL_GVIEW_GET_NEW (gview, mask);
	
	for (i = 0; i < nr; i++)
	{
		t = 1.0;
		if (clustering)
		{
			t = (double) gdl_clustering_clust_size (clustering, i);
		}
		t /= (double) nn;
		
		gdl_vector_set (gm->rw, i, t);
		
		for (j = 0; j < nl; j++)
		{
			for (k = 0; k < np; k++)
			{
				if (clustering)
				{
					GDL_GVIEW_GET_ALLELE (gview, mask, gdl_clustering_clust_idx (clustering, i), j, k, gbuf);
				}
				else
				{
					GDL_GVIEW_GET_ALLELE (gview, mask, i, j, k, gbuf);
				}
				_gdl_gmatrix_locus_push (gm, i, j, gdl_gvalues_get_gvalues (gbuf), last_allele);
			}
		}
	}
	
	gdl_gvalues_get_free (gbuf);
	
	return GDL_SUCCESS;
}

static int
gdl_gmatrix_allele_wrapper_alloc (gdl_gmatrix * gm, const gdl_gview_wrapper * gwrap, gdl_boolean accession_cluster, gdl_boolean last_allele)
{
	size_t i, j, k, l, nr, nl, np, nn;
	gdl_locus * locus;
	double t;
	gdl_gvalues_get * gbuf;
	const gdl_gvalues   * gv;
	
	np = gdl_gview_wrapper_ploidy (gwrap);
	nr = nn = gdl_gview_wrapper_accession_size (gwrap);
	if (accession_cluster)
	{
		nr = gdl_gview_wrapper_accession_size_c (gwrap);
	}
	nl = gdl_gview_wrapper_locus_size (gwrap);
	
	gm->size1 = nr;
	gm->nl  = nl;
	gm->rw  = gdl_vector_alloc (nr);
	gm->nc  = GDL_MALLOC (size_t, nl);
	gm->tnc = GDL_MALLOC (size_t, nl);
	gm->tnc[0] = 0;
	
	for (i = 0; i < nl; i++)
	{
		locus     = gdl_gview_wrapper_get_locus (gwrap, i);
		gm->nc[i] = gdl_locus_allele (locus);
		if ((gm->nc[i]==1 && !last_allele) || gm->nc[i]==0)
		{
			GDL_ERROR_VAL (gdl_string_sprintf ("Locus %s is not polymorphe", gdl_entity_get_name (locus)),
			               GDL_FAILURE,
			               GDL_FAILURE);
		}
		if (!last_allele)
		{
			(gm->nc[i])--;
		}
		if (i)
		{
			gm->tnc[i] = gm->tnc[i-1] + gm->nc[i-1];
		}
	}
	
	gm->size2 = gm->tnc[i-1] + gm->nc[i-1];
	
	gm->data  = gdl_matrix_calloc (gm->size1, gm->size2);
	
	gbuf = gdl_gview_wrapper_get_new (gwrap);
	
	for (i = 0; i < nr; i++)
	{
		t = 1.0;
		if (accession_cluster)
		{
			t = (double) gdl_gview_wrapper_accession_mult_c (gwrap, i);
		}
		t /= (double) nn;
		
		gdl_vector_set (gm->rw, i, t);
		
		for (j = 0; j < nl; j++)
		{
			for (k = 0; k < np; k++)
			{
				if (accession_cluster)
				{
					gdl_gview_wrapper_get_allele_c (gwrap, i, j, k, gbuf);
				}
				else
				{
					gdl_gview_wrapper_get_allele (gwrap, i, j, k, gbuf);	
				}
				
				_gdl_gmatrix_locus_push (gm, i, j, gdl_gvalues_get_gvalues (gbuf), last_allele);
			}
		}
	}
	
	gdl_gvalues_get_free (gbuf);
	
	return GDL_SUCCESS;
}

static int
gdl_gmatrix_allele_wrapper_mask_alloc (gdl_gmatrix * gm, const gdl_gview_wrapper * gwrap, const gdl_mask * mask, gdl_boolean last_allele)
{
	size_t i, j, k, l, nr, nl, np, nn;
	gdl_locus * locus;
	double t;
	gdl_gvalues_get * gbuf;
	const gdl_gvalues   * gv;
	
	np = gdl_gview_wrapper_ploidy (gwrap);
	nr = nn = gdl_mask_size (mask, GDL_ACCESSION);
	if (!nn)
	{
		nr = nn = gdl_gview_wrapper_accession_size (gwrap);	
	}
	nl = gdl_mask_size (mask, GDL_LOCUS);
	if (!nl)
	{
		nl = gdl_gview_wrapper_locus_size (gwrap);
	}
	
	gm->size1 = nr;
	gm->nl  = nl;
	gm->rw  = gdl_vector_alloc (nr);
	gm->nc  = GDL_MALLOC (size_t, nl);
	gm->tnc = GDL_MALLOC (size_t, nl);
	gm->tnc[0] = 0;
	
	for (i = 0; i < nl; i++)
	{
		locus     = gdl_gview_wrapper_get_locus (gwrap, gdl_mask_get_idx (mask, GDL_LOCUS, i));
		gm->nc[i] = gdl_locus_allele (locus);
		if ((gm->nc[i]==1 && !last_allele) || gm->nc[i]==0)
		{
			GDL_ERROR_VAL (gdl_string_sprintf ("Locus %s is not polymorphe", gdl_entity_get_name (locus)),
			               GDL_FAILURE,
			               GDL_FAILURE);
		}
		if (!last_allele)
		{
			(gm->nc[i])--;
		}
		if (i)
		{
			gm->tnc[i] = gm->tnc[i-1] + gm->nc[i-1];
		}
	}
	
	gm->size2 = gm->tnc[i-1] + gm->nc[i-1];
	
	gm->data  = gdl_matrix_calloc (gm->size1, gm->size2);
	
	gbuf = gdl_gview_wrapper_get_new (gwrap);
	
	for (i = 0; i < nr; i++)
	{
		gdl_vector_set (gm->rw, i, 1.0/(double)nr);
		for (j = 0; j < nl; j++)
		{
			for (k = 0; k < np; k++)
			{
				gdl_gview_wrapper_get_allele (gwrap,
				                              gdl_mask_get_idx (mask, GDL_ACCESSION, i),
				                              gdl_mask_get_idx (mask, GDL_LOCUS, j),
				                              k, gbuf);
				
				_gdl_gmatrix_locus_push (gm, i, j, gdl_gvalues_get_gvalues (gbuf), last_allele);
			}
		}
	}
	
	gdl_gvalues_get_free (gbuf);
	
	return GDL_SUCCESS;
}

//static int
//gdl_gmatrix_allele_halloc (gdl_gmatrix * gm, const gdl_hview * hview)
//{
//	size_t i, j, k, l, nr, nl, np;
//	gdl_locus * locus;
//	gdl_gvalues_get * gbuf;
//	const gdl_gvalues   * gv;
//	
//	np = gdl_hview_ploidy (hview);
//	nr = gdl_hview_haplotype_size (hview);
//	nl = gdl_hview_locus_size (hview);
//	
//	gm->size1 = nr;
//	gm->nl  = nl;
//	gm->rw  = gdl_vector_alloc (nr);
//	gm->nc  = GDL_MALLOC (size_t, nl);
//	gm->tnc = GDL_MALLOC (size_t, nl);
//	gm->tnc[0] = 0;
//	
//	for (i = 0; i < nl; i++)
//	{
//		locus     = gdl_hview_get_locus (hview, i);
//		gm->nc[i] = gdl_locus_allele (locus)-1;
//		if (!gm->nc[i])
//		{
//			GDL_ERROR_VAL (gdl_string_sprintf ("Locus %s is not polymorphe", gdl_entity_get_name (locus)),
//			               GDL_FAILURE,
//			               GDL_FAILURE);
//		}
//		if (i)
//		{
//			gm->tnc[i] = gm->tnc[i-1] + gm->nc[i-1];
//		}
//	}
//	
//	gm->size2 = gm->tnc[i-1] + gm->nc[i-1];
//	gm->data = gdl_matrix_calloc (gm->size1, gm->size2);
//	
//	for (i = 0; i < nr; i++)
//	{
//		gdl_haplotype * haplo = gdl_hview_get_haplotype (hview, i);
//		for (j = 0; j < nl; j++)
//		{
//			gdl_allele * allele = gdl_hview_get_hallele (hview, haplo, j);
//			gdl_matrix_set (gm->data, i, gm->tnc[j] + allele->idx, 1.0);
//		}
//		gdl_vector_set (gm->rw, i, gdl_hview_get_haplotype_freq (hview, i));
//	}
//	
//	return GDL_SUCCESS;	
//}

static const gdl_gmatrix_type _gdl_gmatrix_allele =
{
	"gdl_gmatrix_allele",
	"allele",
	&gdl_gmatrix_allele_alloc,
	&gdl_gmatrix_allele_wrapper_alloc,
	&gdl_gmatrix_allele_wrapper_mask_alloc
};

const gdl_gmatrix_type * gdl_gmatrix_allele = &_gdl_gmatrix_allele;
