/*  
 * 	gmatrix/gmatrix.c
 *  
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:53 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_hview.h>
#include <gdl/gdl_glabels.h>
#include <gdl/gdl_gmatrix.h>

gdl_gmatrix *
gdl_gmatrix_alloc (const gdl_gmatrix_type * T, const gdl_gview * gview, const gdl_mask * mask, const gdl_clustering * clustering, gdl_boolean last_allele)
{
	gdl_gmatrix * gm;
	
	gm = GDL_CALLOC (gdl_gmatrix, 1);
	
	(T->alloc)(gm, gview, mask, clustering, last_allele);
   
   gm->gview             = gview;
   gm->mask              = mask;
   gm->clust             = clustering;
   gm->accession_cluster = (clustering) ? gdl_true : gdl_false;
	gm->last_allele       = last_allele;
	
	return gm;
}

gdl_gmatrix *
gdl_gmatrix_wrapper_alloc (const gdl_gmatrix_type * T, const gdl_gview_wrapper * gwrap, gdl_boolean accession_cluster, gdl_boolean last_allele)
{
	gdl_gmatrix * gm;
	
	gm = GDL_CALLOC (gdl_gmatrix, 1);
	
	(T->wrapper_alloc)(gm, gwrap, accession_cluster, last_allele);
	
	gm->gwrap             = gwrap;
	gm->accession_cluster = accession_cluster;
	gm->last_allele       = last_allele;
	
	return gm;
}

gdl_gmatrix *
gdl_gmatrix_wrapper_mask_alloc (const gdl_gmatrix_type * T, const gdl_gview_wrapper * gwrap, const gdl_mask * mask, gdl_boolean last_allele)
{
	gdl_gmatrix * gm;
	
	gm = GDL_CALLOC (gdl_gmatrix, 1);
	
	(T->wrapper_mask_alloc)(gm, gwrap, mask, last_allele);
	
	gm->gwrap             = gwrap;
   gm->mask              = mask;
	
	gm->accession_cluster = gdl_false;
	gm->last_allele       = last_allele;
	
	return gm;	
}

void
gdl_gmatrix_free (gdl_gmatrix * m)
{
	if (m)
	{
		GDL_FREE (m->rw);
		GDL_FREE (m->nc);
		GDL_FREE (m->tnc);
		gdl_matrix_free (m->data);
		GDL_FREE (m);
	}	
}

size_t
gdl_gmatrix_row_size (const gdl_gmatrix * m)
{
	return m->size1;	
}

size_t
gdl_gmatrix_column_size (const gdl_gmatrix * m)
{
	return m->size2;
}

size_t
gdl_gmatrix_accession_size (const gdl_gmatrix * m)
{
	return m->size1;	
}

size_t
gdl_gmatrix_locus_size (const gdl_gmatrix * m)
{
	return m->nl;	
}

size_t
gdl_gmatrix_locus_column_size (const gdl_gmatrix * m, size_t i)
{
	return m->nc[i];	
}

size_t
gdl_gmatrix_locus_first_column (const gdl_gmatrix * m, size_t i)
{
	return m->tnc[i];
}

double
gdl_gmatrix_get (const gdl_gmatrix * m, size_t i, size_t j)
{
	return gdl_matrix_get (m->data, i, j);	
}

void
gdl_gmatrix_set (gdl_gmatrix * m, size_t i, size_t j, double x)
{
	gdl_matrix_set (m->data, i, j, x);
}

void
gdl_gmatrix_locus_set (gdl_gmatrix * m, size_t i, size_t j, const gdl_gvalues * x)
{
	size_t k, jj;
	
	if (x)
	{
		for (k = 0; k < x->size; k++)
		{
			if (x->values[k]->idx < m->nc[j])
			{
				jj = m->tnc[j] + x->values[k]->idx;
				gdl_matrix_set (m->data, i, jj, x->values[k]->value);
			}
		}
	}
	else
	{
		gdl_gmatrix_locus_set_missing (m, i, j);	
	}	
}

gdl_boolean
gdl_gmatrix_locus_is_missing (const gdl_gmatrix * m, size_t i, size_t j)
{
	return (gdl_isnan (gdl_matrix_get (m->data, i, m->tnc[j]))) ? gdl_true : gdl_false;
}

gdl_gvalues *
gdl_gmatrix_locus_get (const gdl_gmatrix * m, size_t i, size_t j)
{
	if (!gdl_gmatrix_locus_is_missing (m, i, j))
	{
		size_t k, n;
		gdl_gvalues * glx;
		
		for (n = 0, k = m->tnc[j]; k < m->tnc[j] + m->nc[j]; k++)
		{
			if (gdl_matrix_get (m->data, i, k))
			{
				n++;	
			}
		}
		
		glx = gdl_gvalues_alloc (n);
		
		for (n = 0, k = m->tnc[j]; k < m->tnc[j] + m->nc[j]; k++)
		{
			if (gdl_matrix_get (m->data, i, k))
			{
				glx->values[n]        = gdl_gvalue_alloc ();
				glx->values[n]->idx   = k-m->tnc[j];
				glx->values[n]->value = gdl_matrix_get (m->data, i, k);
				n++;
			}
		}
		
		return glx;
	}
	
	return NULL;
}

void
gdl_gmatrix_locus_set_missing (gdl_gmatrix * m, size_t i, size_t j)
{
	size_t k;
	
	for (k = m->tnc[j]; k < m->tnc[j] + m->nc[j]; k++)
	{
		gdl_matrix_set (m->data, i, k, GDL_NAN);
	}
}

void
gdl_gmatrix_locus_set_all (gdl_gmatrix * m, size_t i, size_t j, double x)
{
	size_t k;
	
	for (k = m->tnc[j]; k < m->tnc[j] + m->nc[j]; k++)
	{
		gdl_matrix_set (m->data, i, k, x);
	}
}

double
gdl_gmatrix_locus_column_get (const gdl_gmatrix * m, size_t i, size_t j, size_t a)
{
	return gdl_matrix_get (m->data, i, m->tnc[j]+a);	
}

void
gdl_gmatrix_locus_column_set (gdl_gmatrix * m, size_t i, size_t j, size_t a, double x)
{
	gdl_matrix_set (m->data, i, m->tnc[j]+a, x);
}

double
gdl_gmatrix_get_row_weight (const gdl_gmatrix * m, size_t i)
{
	return gdl_vector_get (m->rw, i);	
}

const gdl_vector *
gdl_gmatrix_get_row_weights (const gdl_gmatrix * m)
{
	return m->rw;
}

const gdl_matrix *
gdl_gmatrix_get_matrix (const gdl_gmatrix * m)
{
	return m->data;	
}

gdl_glabels *
gdl_gmatrix_get_labels (const gdl_gmatrix * m)
{
	if (m->gview)
	{
		return gdl_glabels_alloc (m->gview, m->mask, m->clust, m->last_allele);	
	}
	else if (m->gwrap && !m->mask)
	{
		return gdl_glabels_wrapper_alloc (m->gwrap, m->accession_cluster, m->last_allele);
	}
	else if (m->gwrap && m->mask)
	{
		return gdl_glabels_wrapper_mask_alloc (m->gwrap, m->mask, m->last_allele);
	}
}

gdl_mask *
gdl_gmatrix_get_mask (const gdl_gmatrix * m)
{
	if (m->gview && !m->clust)
	{
		return gdl_mask_clone (m->mask);
	}
	else if (m->gview && m->clust)
	{
		return gdl_mask_clustering (m->mask, GDL_ACCESSION, m->clust);	
	}
	else if (m->gwrap && !m->mask)
	{
		if (m->accession_cluster)
		{
			return gdl_mask_clustering (gdl_gview_wrapper_gmask (m->gwrap), GDL_ACCESSION, gdl_gview_wrapper_clustering (m->gwrap));
		}
		else
		{
			return gdl_mask_clone (gdl_gview_wrapper_gmask (m->gwrap));
		}
	}
	else if (m->gwrap && m->mask)
	{
		return gdl_mask_clone (m->mask);
	}
	
	return NULL;
}

int
gdl_gmatrix_fprintf (FILE * stream, const gdl_gmatrix * m)
{
	size_t i, j;
	const size_t L = gdl_gmatrix_locus_size (m);
	
	gdl_glabels * labels = gdl_gmatrix_get_labels (m);
	
	for (i = 0; i < L; i++)
	{
		const size_t A = gdl_gmatrix_locus_column_size (m, i);
		for (j = 0; j < A; j++)
		{
			fprintf (stream, " %s:%s", gdl_glabels_locus (labels, i), gdl_glabels_allele (labels, i, j));
		}
	}
	fprintf (stream,"\n");
	
	for (i = 0; i < m->size1; i++)
	{
		fprintf (stream, "\"%s\"", gdl_glabels_accession (labels, i));
		
		for (j = 0; j < m->size2; j++)
		{
			if (!gdl_isnan (gdl_matrix_get (m->data, i, j)))
			{
				fprintf (stream, " %1.4f", gdl_matrix_get (m->data, i, j));
			}
			else
			{
				fprintf (stream, " ?");
			}
		}
		fprintf (stream, "\n");
	}
	
	gdl_glabels_free (labels);
	
	return GDL_SUCCESS;
}

const
gdl_gmatrix_type * gdl_gmatrix_type_fread (FILE * stream)
{
	if (stream)
	{
		gdl_string * name;
		
		name = gdl_string_fread (stream);
		GDL_FREAD_STATUS (name!=0, 1);
		
		if (!strcmp (name, gdl_gmatrix_allele->name))
		{
			return gdl_gmatrix_allele;
		}
		else if (!strcmp (name, gdl_gmatrix_genotype->name))
		{
			return gdl_gmatrix_genotype;
		}
		
		return NULL;
	}
	
	return NULL;
}

int
gdl_gmatrix_type_fwrite (FILE * stream, const gdl_gmatrix_type * T)
{
	if (stream && T)
	{
		return gdl_string_fwrite (stream, T->name);
	}
	
	return GDL_EINVAL;	
}
