/*  
 * 	gpart/gpart.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:41 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gpartition.h>

enum _gdl_gligation_type
{
	_gdl_gligation_progressive,
	_gdl_gligation_hierarchical
};

struct _gdl_gpartition
{
	size_t np;
	size_t _np;
	const gdl_gview * g;
	const gdl_mask * m;
	gdl_gpartition_state ** part;
};

gdl_gpartition *
gdl_gpartition_alloc (const gdl_gview * g, const gdl_mask * m)
{
	gdl_gpartition * gp;
	
	gp = GDL_CALLOC (gdl_gpartition, 1);
	
	gp->g = g;
	gp->m = m;
	
	return gp;
}

gdl_gpartition *
gdl_gpartition_new (gdl_gpartition_state ** part, size_t np)
{
	gdl_gpartition * g;
	
	g = GDL_CALLOC (gdl_gpartition, 1);
	
	g->part = part;
	g->_np  = g->np = np;
	
	return g;
}

void
gdl_gpartition_free (gdl_gpartition * g, gdl_gpartition_state_free fre)
{
	if (g)
	{
		if (fre)
		{
			size_t i;
			for (i = 0; i < g->np; i++)
			{
				if (g->part[i])
				{
					(fre)(g->part[i]->data);
					gdl_mask_free (g->part[i]->mask);
				}
			}
		}
		GDL_FREE (g->part);
		GDL_FREE (g);
	}	
}

int
gdl_gpartition_init (gdl_gpartition * g, gdl_gpartition_fonction f, void * extra)
{
	g->part = (f)(g->g, g->m, &(g->np), extra);
	g->_np  = g->np;
}

size_t
gdl_gpartition_size (const gdl_gpartition * g)
{
	return g->np;
}

size_t
gdl_gpartition_eligible_size (const gdl_gpartition * g, const gdl_gligation_type * T)
{
	switch (*T)
	{
		case _gdl_gligation_progressive :
			return (g->_np == g->np) ? g->np : 1;
		case _gdl_gligation_hierarchical :
			return g->np;
	}	
}

gdl_gpartition_state *
gdl_gpartition_get (const gdl_gpartition * g, size_t i)
{
	return g->part[i];
}


static int
_gdl_gpartition_ligation_progressive (gdl_gpartition * g, gdl_gligation_fonction f, gdl_gpartition_state_free free, gdl_gligation_update update, void * extra)
{
	if (g->np > 1)
	{
		
		size_t i;
		
		(f)(g->part[0], g->part[1], gdl_gligation_progressive, extra);
		
		if (free)
		{
			(free)(g->part[1]->data);
			gdl_mask_free (g->part[1]->mask);
		}
		
		for (i = 2; i < g->np; i++)
		{
			g->part[i-1] = g->part[i];
			if (update)
			{
				(update)(g->part[i-1], gdl_gligation_progressive, extra);
			}
		}
		g->part[i-1] = NULL;
		
		(g->np)--;
		
		return (g->np > 1) ? GDL_CONTINUE : GDL_SUCCESS;
	}
	
	return GDL_SUCCESS;
}

static int
_gdl_gpartition_ligation_hierarchical (gdl_gpartition * g, gdl_gligation_fonction f, gdl_gpartition_state_free free, gdl_gligation_update update, void * extra)
{
	if (g->np > 1)
	{
		size_t i, j, np;
		
		np = g->np;
		
		for (i = 1; i < g->np; i+=2)
		{
			(f)(g->part[i-1], g->part[i], gdl_gligation_hierarchical, extra);
			if (free)
			{
				(free)(g->part[i]->data);
				gdl_mask_free (g->part[i]->mask);
			}
			g->part[i] = NULL;
			np--;	
		}
		
		for (j = 1, i = 2; i < g->np; i+=2, j++)
		{
			g->part[i-j] = g->part[i];
			g->part[i]   = NULL;
		}
		
		if (GDL_IS_ODD (g->np) && update)
		{
			(update)(g->part[np-1], gdl_gligation_hierarchical, extra);
		}
				
		g->np = np;	
		
		return (g->np > 1) ? GDL_CONTINUE : GDL_SUCCESS;
	}
	
	return GDL_SUCCESS;
}

int
gdl_gpartition_ligation (gdl_gpartition * g, const gdl_gligation_type * T, gdl_gligation_fonction f, gdl_gpartition_state_free free, gdl_gligation_update update, void * extra)
{
	switch (*T)
	{
		case _gdl_gligation_progressive :
			return _gdl_gpartition_ligation_progressive (g, f, free, update, extra);
		case _gdl_gligation_hierarchical :
			return _gdl_gpartition_ligation_hierarchical (g, f, free, update, extra);
	}
}

static const gdl_gligation_type _progressive = _gdl_gligation_progressive;
static const gdl_gligation_type _hierarchical = _gdl_gligation_hierarchical;

const gdl_gligation_type * gdl_gligation_progressive = &_progressive;
const gdl_gligation_type * gdl_gligation_hierarchical = &_hierarchical;
