/*  
 *  gpca/gpca.c
 *  
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:42 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_gmatrix.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_hview.h>
#include <gdl/gdl_pca.h>
#include <gdl/gdl_ppca.h>
#include <gdl/gdl_gpca.h>

gdl_gpca_workspace *
gdl_gpca_workspace_alloc (const gdl_gpca_workspace_type * T, const gdl_gview_wrapper * gwrap)
{
	gdl_gpca_workspace * g;
	
	g = GDL_CALLOC (gdl_gpca_workspace, 1);
	
	g->gmatrix = (T->gmatrix)(gwrap);
	
	g->pca  = (T->pca)();
	
	g->ppca = (T->ppca)();
	
	g->type = T;
	
	return g;
}

void
gdl_gpca_workspace_free (gdl_gpca_workspace * w)
{
	if (w)
	{
		gdl_gmatrix_free (w->gmatrix);
		gdl_pca_workspace_free (w->pca);
		GDL_FREE (w);
	}		
}

int
gdl_gpca_workspace_perform (gdl_gpca_workspace * w, size_t naxe)
{
	int status;
	gdl_matrix * X;
	gdl_vector * wr;
	
	w->_naxe = naxe;
	
	X  = (w->type->matrix)(w->gmatrix);
	
	wr = (w->type->weights)(w->gmatrix);
	
	if (!naxe)
	{
		w->algo = 0;
		
		if (X->size1 >= X->size2)
		{
			w->_naxe = (naxe) ? naxe : X->size2;
			
			if (wr)
			{
				status = gdl_pca_workspace_wperform (w->pca, X, wr);
				gdl_vector_free (wr);
			}
			else
			{
				status = gdl_pca_workspace_perform (w->pca, X);
			}
		}
		else
		{
			gdl_matrix * tX;
			
			w->_naxe = (naxe) ? naxe : X->size1;
			
			tX = gdl_matrix_alloc (X->size2, X->size1);
			gdl_matrix_transpose_memcpy (tX, X);
			gdl_matrix_free (X);
			X = tX;
			
			if (wr)
			{
				status = gdl_pca_workspace_wperform_transpose (w->pca, X, wr);
				gdl_vector_free (wr);
			}
			else
			{
				status = gdl_pca_workspace_perform_transpose (w->pca, X);
			}
			
		}
	}
	else
	{
		w->_naxe = GDL_MIN (naxe, GDL_MIN (X->size2, X->size1));
		
		w->algo = 1;
			
		if (wr)
		{
			status = gdl_ppca_workspace_wperform (w->ppca, X, wr, w->_naxe, 1.e-3, 50);
			gdl_vector_free (wr);
		}
		else
		{
			status = gdl_ppca_workspace_perform (w->ppca, X, w->_naxe, 1.e-3, 50);
		}
		
	}
	
	gdl_matrix_free (X);
	
	return status;
}

const gdl_gpca_workspace_type *
gdl_gpca_workspace_get_type (const gdl_gpca_workspace * r)
{
	return r->type;	
}

const gdl_gmatrix *
gdl_gpca_workspace_data (const gdl_gpca_workspace * w)
{
	return w->gmatrix;
}

const gdl_matrix *
gdl_gpca_workspace_projection (const gdl_gpca_workspace * w)
{
	if (w->algo)
	{
		return gdl_ppca_workspace_projection (w->ppca);
	}
	else
	{
		return gdl_pca_workspace_projection (w->pca);
	}
}

const gdl_matrix *
gdl_gpca_workspace_rotation (const gdl_gpca_workspace * w)
{
	if (w->algo)
	{
		return gdl_ppca_workspace_rotation (w->ppca);
	}
	else
	{
		return gdl_pca_workspace_rotation (w->pca);
	}
}

const gdl_vector *
gdl_gpca_workspace_weights (const gdl_gpca_workspace * w)
{	
	if (w->algo)
	{
		return gdl_ppca_workspace_weights (w->ppca);
	}
	else
	{
		return gdl_pca_workspace_weights (w->pca);
	}
}

const gdl_vector *
gdl_gpca_workspace_tracy_widom (const gdl_gpca_workspace * w)
{	
	if (w->algo)
	{
		return gdl_ppca_workspace_tracy_widom (w->ppca);
	}
	else
	{
		return gdl_pca_workspace_tracy_widom (w->pca);
	}
}

size_t
gdl_gpca_workspace_scree_threshold (const gdl_gpca_workspace * w, double threshold)
{
	return gdl_pca_workspace_scree_threshold (w->pca, threshold);
}
