/*  
 *  gpca/icov.c
 *  
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:42 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_blas.h>
#include <gdl/gdl_cblas.h>
#include <gdl/gdl_gmatrix.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_hview.h>
#include <gdl/gdl_statistics.h>
#include <gdl/gdl_pca.h>
#include <gdl/gdl_gpca.h>

static gdl_gmatrix *
gdl_gpca_accession_allele_euclidean_distance_gmatrix (const gdl_gview_wrapper * gwrap)
{
	return gdl_gmatrix_wrapper_alloc (gdl_gmatrix_allele, gwrap, gdl_true, gdl_true);
}

//static gdl_gmatrix *
//gdl_gpca_accession_allele_euclidean_distance_hmatrix (const gdl_hview * hview)
//{
//	return gdl_gmatrix_halloc (gdl_gmatrix_allele, hview);
//}

static gdl_pca_workspace *
gdl_gpca_accession_allele_euclidean_distance_pca (void)
{
	return gdl_pca_workspace_alloc (gdl_pca_standard);
}

static gdl_matrix *
gdl_gpca_accession_allele_euclidean_distance_matrix (const gdl_gmatrix * g)
{
	size_t i, j, k;
	double x1, x2;
	const gdl_matrix * X  = gdl_gmatrix_get_matrix (g);
	gdl_matrix       * D  = gdl_matrix_calloc (X->size1, X->size1);
	gdl_vector       * d  = gdl_vector_alloc (X->size2);
	gdl_vector       * m  = gdl_vector_alloc (X->size2);
	
	// Compute the mean of the columns
	for (i = 0; i < X->size2; i++)
	{
		gdl_vector_const_view ci = gdl_matrix_const_column (X, i);
		x1 = gdl_stats_mean (ci.vector.data, ci.vector.stride, ci.vector.size);
		gdl_vector_set (m, i, x1);
	}
	// Compute the distances
	for (i = 0; i < X->size1; i++)
	{
		gdl_vector_const_view ri = gdl_matrix_const_row (X, i);
		for (j = i+1; j < X->size1; j++)
		{
			 gdl_vector_const_view rj = gdl_matrix_const_row (X, j);
			 for (k = 0; k < X->size2; k++)
			 {
			 	 x1 = gdl_vector_get (&(ri.vector), k);
			 	 x2 = gdl_vector_get (&(rj.vector), k);
			 	 if (gdl_isnan (x1))
			 	 {
			 	 	x1 = gdl_vector_get (m, k);
			 	 }
			 	 if (gdl_isnan (x2))
			 	 {
			 	 	x2 = gdl_vector_get (m, k); 
			 	 }
			 	 gdl_vector_set (d, k, x1-x2);
			 } 
			 gdl_matrix_set (D, i, j, gdl_blas_dnrm2 (d));
			 gdl_matrix_set (D, j, i, gdl_matrix_get (D, i, j));
		}
	}
	
	gdl_vector_free (d);
	
	return D;
}

static gdl_vector *
gdl_gpca_accession_allele_euclidean_distance_weights (const gdl_gmatrix * g)
{
	return NULL;
}

static const gdl_gpca_workspace_type _gdl_gpca_accession_allele_euclidean_distance =
{
	"gdl_gpca_accession_allele_euclidean_distance",
	"iaeuc",
	&gdl_gpca_accession_allele_euclidean_distance_gmatrix,
//	&gdl_gpca_accession_allele_euclidean_distance_hmatrix,
	&gdl_gpca_accession_allele_euclidean_distance_pca,
	&gdl_gpca_accession_allele_euclidean_distance_matrix,
	&gdl_gpca_accession_allele_euclidean_distance_weights
};

const gdl_gpca_workspace_type * gdl_gpca_accession_allele_euclidean_distance = &_gdl_gpca_accession_allele_euclidean_distance;
