/*  
 *  gpca/result.c
 *  
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:42 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_odb.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_hview.h>
#include <gdl/gdl_glabels.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_pca.h>
#include <gdl/gdl_gpca.h>
#include <gdl/gdl_gpca_result.h>

struct _gdl_gpca_result
{
	const gdl_gpca_workspace_type * type;
	gdl_glabels * labels;
	size_t naxe;
	gdl_matrix * U;
	gdl_vector * S;
	gdl_vector * TW;
	gdl_matrix * E;
};

gdl_gpca_result *
gdl_gpca_result_alloc (const gdl_gpca_workspace * w)
{
	const gdl_matrix * U, * E;
	const gdl_vector * S, * TW;
	gdl_gpca_result * r;
	
	r = GDL_MALLOC (gdl_gpca_result, 1);
	
	r->type = gdl_gpca_workspace_get_type (w);
	
	U  = gdl_gpca_workspace_projection (w);
	E  = gdl_gpca_workspace_rotation (w);
	S  = gdl_gpca_workspace_weights (w);
	TW = gdl_gpca_workspace_tracy_widom (w);
	
	r->U  = gdl_matrix_alloc (U->size1, U->size2);
	r->E  = gdl_matrix_alloc (E->size1, E->size2);
	r->S  = gdl_vector_alloc (S->size);
	r->TW = gdl_vector_calloc (S->size);
	
	gdl_matrix_memcpy (r->U, U);
	gdl_matrix_memcpy (r->E, E);
	gdl_vector_memcpy (r->S, S);
	//gdl_vector_memcpy (r->TW, TW);
		
	r->naxe = r->U->size2;
	
	r->labels = gdl_gmatrix_get_labels (w->gmatrix);
	
	return r;	
}

void
gdl_gpca_result_free (gdl_gpca_result * r)
{
	if (r)
	{
		gdl_matrix_free (r->U);
		gdl_matrix_free (r->E);
		gdl_vector_free (r->S);
		gdl_vector_free (r->TW);
		GDL_FREE (r);
	}	
}

size_t
gdl_gpca_result_naxe (const gdl_gpca_result * r)
{
	return r->naxe;
}

gdl_matrix *
gdl_gpca_result_projection (const gdl_gpca_result * r)
{
	return r->U;
}

gdl_matrix *
gdl_gpca_result_rotation (const gdl_gpca_result * r)
{
	return r->E;	
}

gdl_vector *
gdl_gpca_result_weights (const gdl_gpca_result * r)
{
	return r->S;
}

gdl_vector *
gdl_gpca_result_tracy_widom (const gdl_gpca_result * r)
{
	return r->TW;
}

const gdl_gpca_workspace_type *
gdl_gpca_result_get_type (const gdl_gpca_result * r)
{
	return r->type;	
}

const gdl_glabels *
gdl_gpca_result_get_labels (const gdl_gpca_result * r)
{
	return r->labels;	
}

const gdl_gpca_workspace_type *
gdl_gpca_workspace_type_fread (FILE * stream)
{
	gdl_string * name = gdl_string_fread (stream);
	if (name)
	{
		const gdl_gpca_workspace_type_registry * R;
		
		R = gdl_gpca_workspace_type_registry_get ();
		
		return gdl_gpca_workspace_type_registry_search (R, name);
	}
	
	return NULL;	
}

int
gdl_gpca_workspace_type_fwrite (FILE * stream, const gdl_gpca_workspace_type * T)
{
	return gdl_string_fwrite (stream, T->acronym);
}

gdl_gpca_result *
gdl_gpca_result_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		gdl_gpca_result * r;
		
		r = GDL_MALLOC (gdl_gpca_result, 1);
		
		r->type = gdl_gpca_workspace_type_fread (stream);
		GDL_FREAD_STATUS (r->type!=0, 1);
		r->U = gdl_matrix_fread (stream);
		GDL_FREAD_STATUS (r->U!=0, 1);
		r->E = gdl_matrix_fread (stream);
		GDL_FREAD_STATUS (r->E!=0, 1);
		r->S = gdl_vector_fread (stream);
		GDL_FREAD_STATUS (r->S!=0, 1);
		r->TW = gdl_vector_fread (stream);
		GDL_FREAD_STATUS (r->TW!=0, 1);
		r->labels = gdl_glabels_fread (stream);
		GDL_FWRITE_STATUS (r->labels!=0, 1);
		
		r->naxe = r->U->size2;
		
		return r;
	}
	return NULL;	
}

int
gdl_gpca_result_fwrite (FILE * stream, const gdl_gpca_result * r)
{
	if (stream && r)
	{
		int status;
		
		status = gdl_gpca_workspace_type_fwrite (stream, r->type);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_matrix_fwrite (stream, r->U);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_matrix_fwrite (stream, r->E);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_vector_fwrite (stream, r->S);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_vector_fwrite (stream, r->TW);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_glabels_fwrite (stream, r->labels);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
				
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}

gdl_string *
gdl_gpca_result_get_row_label (const gdl_gpca_result * r, size_t i)
{
	if (r->type    == gdl_gpca_locus_allele_covariance
	    || r->type == gdl_gpca_locus_allele_correlation
	    || r->type == gdl_gpca_accession_allele_euclidean_distance)
	{
	   return gdl_string_clone (gdl_glabels_accession (r->labels, i));
	}
	else if (r->type    == gdl_gpca_accession_allele_covariance
	         || r->type == gdl_gpca_accession_allele_correlation)
	{
	   size_t l, a;
	   gdl_glabels_allele_cumul_index (r->labels, i, &l, &a);
	   return gdl_string_sprintf ("%s %s", gdl_glabels_locus (r->labels, l), gdl_glabels_allele (r->labels, l, a));
	}
	
	return NULL;
}

gdl_string *
gdl_gpca_result_get_column_label (const gdl_gpca_result * r, size_t i)
{
	if (r->type == gdl_gpca_locus_allele_covariance
	    || r->type == gdl_gpca_locus_allele_correlation)
	{
		size_t l, a;
	   gdl_glabels_allele_cumul_index (r->labels, i, &l, &a);
	   return gdl_string_sprintf ("%s %s", gdl_glabels_locus (r->labels, l), gdl_glabels_allele (r->labels, l, a));
	}
	else if (r->type    == gdl_gpca_accession_allele_covariance
	         || r->type == gdl_gpca_accession_allele_correlation
	         || r->type == gdl_gpca_accession_allele_euclidean_distance)
	{
	   return gdl_string_clone (gdl_glabels_accession (r->labels, i));
	}
}

int
gdl_gpca_result_fprintf (FILE * stream, const gdl_gpca_result * r)
{
	if (stream && r)
	{
		size_t i, j;
		
		fprintf (stream, "Parameters:\n");
		fprintf (stream, "\tNumber of rows %d\n", r->U->size1);
		fprintf (stream, "\tNumber of columns %d\n", r->E->size1);
		fprintf (stream, "\tNumber of PC %d\n", r->naxe);
		fprintf (stream, "\n--------------------------------------------\n");
		fprintf (stream, "Weights of the principal components\n\n");
		for (i = 0; i < r->S->size; i++)
		{
			fprintf (stream, "\tPCA%d", i+1);
		}
		fprintf (stream, "\n");
		for (i = 0; i < r->S->size; i++)
		{
			fprintf (stream, "\t%g", gdl_vector_get (r->S, i));
		}
		fprintf (stream, "\n--------------------------------------------\n");
		fprintf (stream, "Tracy-Widom statistics of the principal components\n\n");
		for (i = 0; i < r->TW->size; i++)
		{
			fprintf (stream, "\tPCA%d", i+1);
		}
		fprintf (stream, "\n");
		for (i = 0; i < r->TW->size; i++)
		{
			fprintf (stream, "\t%g", gdl_vector_get (r->TW, i));
		}
		fprintf (stream, "\n--------------------------------------------\n");
		fprintf (stream, "Projection on the principal components\n\n");
		for (i = 0; i < r->U->size2; i++)
		{
			fprintf (stream, "\tPCA%d", i+1);
		}
		fprintf (stream, "\n");
		for (i = 0; i < r->U->size1; i++)
		{
			// check the type to get the label
			gdl_string * label = gdl_gpca_result_get_row_label (r, i);
			fprintf (stream, "%s", label);
			gdl_string_free (label);
			for (j = 0; j < r->U->size2; j++)
			{
				fprintf (stream, "\t%g", gdl_matrix_get (r->U, i, j));
			}
			fprintf (stream, "\n");
		}
		fprintf (stream, "\n--------------------------------------------\n");
		fprintf (stream, "Principal components coordinates\n\n");
		for (i = 0; i < r->E->size2; i++)
		{
			fprintf (stream, "\tPCA%d", i+1);
		}
		fprintf (stream, "\n");
		for (i = 0; i < r->E->size1; i++)
		{
			// check the type to get the label
			gdl_string * label = gdl_gpca_result_get_column_label (r, i);
			fprintf (stream, "%s", label);
			gdl_string_free (label);
			for (j = 0; j < r->E->size2; j++)
			{
				fprintf (stream, "\t%g", gdl_matrix_get (r->E, i, j));
			}
			fprintf (stream, "\n");
		}
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}
