/*  
 * 	gpoint/data.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:53 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_util.h>
#include <gdl/gdl_gpoint.h>

gdl_gdatapoint * 
gdl_gdatapoint_alloc (const gdl_gpoint_type * type, size_t size)
{
	return gdl_gregistry_entry_alloc (type, size);
}

void
gdl_gdatapoint_free (gdl_gdatapoint * g)
{
	gdl_gregistry_entry_free (g);
}

struct _gdl_gdatapoints
{
	size_t p;
	size_t n;
	size_t l;
	size_t na;
	gdl_gpoint     ** gpoints;
	gdl_gregistry  * registry;
};

#define INDEX(g,i,j)((i)*(g)->l+(j))

gdl_gdatapoints *
gdl_gdatapoints_alloc (size_t p, size_t n, size_t l)
{
	gdl_gdatapoints * g;
	
	g = GDL_CALLOC (gdl_gdatapoints, 1);
	
	if (g == 0)
	{
		GDL_ERROR_VAL ("Unable to allocate memory in gdl_gdatapoints_alloc",
		               GDL_ENOMEM,
		               0);
	}
	
	g->p  = p;
	g->n  = n;
	g->l  = l;
	g->na = n*l;
	
	g->gpoints = GDL_CALLOC (gdl_gpoint *, g->n*g->l);
	
	if (g->gpoints == 0)
	{
		GDL_FREE (g);
		GDL_ERROR_VAL ("Unable to allocate memory in gdl_gdatapoints_alloc",
		               GDL_ENOMEM,
		               0);	
	}
	
	g->registry = gdl_gregistry_alloc (p);
	
	if (g->registry == 0)
	{
		GDL_FREE (g->gpoints);
		GDL_FREE (g);
		GDL_ERROR_VAL ("Unable to allocate memory in gdl_gdatapoints_alloc",
		               GDL_ENOMEM,
		               0);		
	}
	
	return g;
}

void
gdl_gdatapoints_free (gdl_gdatapoints * g)
{
	if (g)
	{
		GDL_FREE (g->gpoints);
		gdl_gregistry_free (g->registry);
	}
}

int
gdl_gdatapoints_set (gdl_gdatapoints * g, 
                         size_t i,
                         size_t j,
                         gdl_gdatapoint * x)
{
	/* This can be improved by using a sort of 
	 * garbage collector for the gpoint pointers
	 * which are still in the registry but no longer
	 * in used in the matrix...
	 */
	size_t k = INDEX(g,i,j);
	gdl_gpoint * p = g->gpoints[k];
	
	if (x)
	{
		g->gpoints[k] = gdl_gregistry_add (g->registry, x);
	}
	else
	{
		g->gpoints[k] = NULL;	
	}
	
	if (g->gpoints[k] != NULL && p == NULL)
	{
		g->na--;
	}
	else if (g->gpoints[k] == NULL && p != NULL)
	{
		g->na++;
	}
	
	return (0);
}

gdl_gdatapoint *
gdl_gdatapoints_get (gdl_gdatapoints * g, size_t i, size_t j)
{
	size_t k = INDEX(g,i,j);
	
	if (g->gpoints[k])
	{
		return 
			gdl_gregistry_get (g->registry, g->gpoints[k]);
	}
	else
	{
		return NULL;
	}
}

gdl_boolean
gdl_gdatapoints_is_homozygous (gdl_gdatapoints * g, size_t i, size_t j)
{
	size_t k = INDEX(g,i,j);
	
	if (g->gpoints[k])
	{
		return 
			gdl_gregistry_is_homozygous (g->registry, g->gpoints[k]);
	}
	else
	{
		return gdl_false;
	}	
}

gdl_boolean
gdl_gdatapoints_has_missing (gdl_gdatapoints * g, size_t i, size_t j)
{
	if (g->gpoints[INDEX(g,i,j)])
	{
		return gdl_gregistry_has_missing (g->registry, g->gpoints[INDEX(g,i,j)]);
	}
	else
	{
		return gdl_true;
	}
}

gdl_boolean
gdl_gdatapoints_is_missing (gdl_gdatapoints * g, size_t i, size_t j, size_t k)
{
	if (g->gpoints[INDEX(g,i,j)])
	{
		return gdl_gregistry_is_missing (g->registry, g->gpoints[INDEX(g,i,j)], k);
	}
	else
	{
		return gdl_true;
	}
}

static int
_gdl_gvalues_compare (gdl_gvalues ** g1, gdl_gvalues ** g2, size_t size)
{
	size_t i, j, n;
	gdl_boolean * ok;
	
	ok = GDL_CALLOC (gdl_boolean, size);
	n  = 0;
	
	for (i = 0; i < size; i++)
	{
		for (j = 0; j < size; j++)
		{
			if (ok[j])
			{
				continue;
			}
			if (!gdl_gvalues_compare (g1[i], g2[j]))
			{
				ok[j] = gdl_true;
				n++;
				break;
			}					
		}
		if (n != i + 1)
		{
			GDL_FREE (ok);
			return (-1);
		}
	}
	
	GDL_FREE (ok);
	
	return (0);	
}

int
gdl_gdatapoints_compare (const gdl_gdatapoints * g,
                           size_t i,
                           size_t j,
                           size_t ii,
                           size_t jj)
{
	gdl_gpoint * g1 = g->gpoints[INDEX(g,i,j)];
	gdl_gpoint * g2 = g->gpoints[INDEX(g,ii,jj)];
	
	if (g1 == g2)
	{
		return 0;
	}
	else if (g1 && g2)
	{
		int status;
		size_t i, size;
		const gdl_gpoint_type * type;
	   	gdl_gdatapoint * gp;
	   	gdl_gvalues ** v1;
	   	
	   	gp = gdl_gregistry_get (g->registry, g1);
	   	
	   	size = gp->size;
	   	type = gp->type;
	   	v1   = GDL_MALLOC (gdl_gvalues *, size);
	   	
	   	for (i =  0; i < size; i++)
	   	{
	   		v1[i] = gp->values[i];
	   	}
	   	
	   	gp = gdl_gregistry_get (g->registry, g2);
	   	
	   	if (gp->size != size)
	   	{
	   		return (-1);	
	   	}
	   	if (gp->type != type)
	   	{
	   		return (-1);
	   	}
	   	else
	   	{
	   		_gdl_gvalues_compare (v1, gp->values, size);
	   	}
	   	
	   	GDL_FREE (v1);
	   	
	   	return status;
	}
	else
	{
		return (-1);	
	}
}

int
gdl_gdatapoint_compare (const gdl_gdatapoint * g1, const gdl_gdatapoint * g2)
{
	if (g1 && g2)
	{
		if (g1->type != g2->type)
		{
			return (-1);
		}
		if (g1->size != g2->size)
		{
			return (-1);
		}
		else
		{
			return _gdl_gvalues_compare (g1->values, g2->values, g1->size);
		}
	}
	return (-1);
}

gdl_gdatapoints *
gdl_gdatapoints_fread (FILE * stream)
{
	if (stream == 0)
		return NULL;
	else
	{
		size_t i, j, p, n, l, status;
		gdl_gdatapoints * g;
		
		status = fread (&p, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&n, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&l, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		g = gdl_gdatapoints_alloc (p, n, l);
		
		gdl_gregistry_free (g->registry);
		
		g->registry = gdl_gregistry_fread (stream);
		GDL_FREAD_STATUS (g->registry != NULL, 1);
		
		status = fread (&g->na, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		for (i = 0; i < g->n*g->l - g->na; i++)
		{
			status = fread (&j, sizeof (size_t), 1, stream);
			GDL_FREAD_STATUS (status, 1);
			g->gpoints[j] = gdl_gpoint_fread (stream);
			GDL_FREAD_STATUS (g->gpoints[j] != NULL, 1);
			gdl_gregistry_add_gpoint (g->registry, &g->gpoints[j]);
		}
		
		return g;
	}	
}

int
gdl_gdatapoints_fwrite (FILE * stream, const gdl_gdatapoints * g)
{
	if (stream == 0 || g == 0)
		return GDL_EINVAL;
	else
	{
		size_t i, j, k, status;
		
		status = fwrite (&g->p, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&g->n, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&g->l, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		status = gdl_gregistry_fwrite (stream, g->registry);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		status = fwrite (&g->na, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < g->n; i++)
		{
			for (j = 0; j < g->l; j++)
			{
				k = INDEX(g,i,j);
				
				if (g->gpoints[k])
				{
					status = fwrite (&k, sizeof (size_t), 1, stream);
					GDL_FWRITE_STATUS (status, 1);
					status = gdl_gpoint_fwrite (stream, g->gpoints[k]);
					GDL_FWRITE_STATUS (status, GDL_SUCCESS);
				}
			}
		}
	}
	return GDL_SUCCESS;
}

#undef INDEX

