/*  
 * 	gpoint/greg.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:53 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_util.h>
#include <gdl/gdl_list.h>
#include <gdl/gdl_gpoint.h>

struct _gdl_gregistry
{
	size_t ploidy;
	size_t nhc;
	gdl_list * ggpoints;
	gdl_list * hgpoints;
	size_t   * hconfig;
	gdl_gvalues_registry * gvalues;
	gdl_gvalues_registry * hvalues;
	gdl_gregistry_entry  * gentry;
	gdl_gregistry_entry  * hentry;	
};

#define GET_HCONFIG(gr, i, j)((int)(gr)->hconfig[(gr)->ploidy*(i)+(j)]-1)
#define SET_HCONFIG(gr, i, j, k)((gr)->hconfig[(gr)->ploidy*(i)+(j)]=(size_t)((k)+1))

static int
_gresitry_grow_hconfig (gdl_gregistry * gr, size_t n)
{
	size_t * new = GDL_MALLOC (size_t, n*gr->ploidy);
	memcpy (new, gr->hconfig, sizeof(size_t)*gr->nhc*gr->ploidy);
	GDL_FREE (gr->hconfig);
	gr->hconfig = new;
	gr->nhc = n;
	return (0);
}

static int
_gregistry_compare (gdl_gvalues_registry * g,
                    size_t idx,
                    gdl_gvalues * v)
{                                   
	gdl_gvalues * gv = gdl_gvalues_registry_get (g, idx);
	return gdl_gvalues_compare (gv, v);
}

static int
_gregistry_compare_phased (gdl_gregistry * gr,
                                   size_t idx,
                                   gdl_gvalues ** v)
{                                   
	size_t i;
	
	for (i = 0; i < gr->ploidy; i++)
	{
		int j = GET_HCONFIG(gr,idx,i);
		
		if (j == -1 && v[i] != NULL)
			return (1);
		if ( j != -1 && v[i] == NULL)
			return (1);
		if ( j == -1 && v[i] == NULL)
			continue;
		
		if (_gregistry_compare (gr->hvalues, (size_t)j, v[i]))
		{
			return (1);
		}
	}
	
	return (0);
}

static int
_gregistry_compare_unphased (gdl_gregistry * gr,
                                   size_t idx,
                                   gdl_gvalues ** v)
{                                   
	size_t i, j, f = 0;
	size_t * flag  = GDL_CALLOC (size_t, gr->ploidy);
	
	for (i = 0; i < gr->ploidy; i++)
	{
		for (j = 0; j < gr->ploidy; j++)
		{
			if (!flag[j])
			{
				int k = GET_HCONFIG(gr,idx,j);
				
				if ( k == -1 && v[i] != NULL)
					continue;
				if ( k != -1 && v[i] == NULL)
					continue;
				if ( k == -1 && v[i] == NULL)
				{
					flag[j]=1;
					f++;
					break;
				}
				if (!_gregistry_compare (gr->hvalues,(size_t)k, v[i]))
				{
					flag[j]=1;
					f++;
					break;
				}
			}
		}
		if (f != i+1)
		{
			GDL_FREE (flag);
			return (1);
		}
	}
	GDL_FREE (flag);
	return (0);
}

static gdl_gpoint *
_gregistry_lookup_hconfig (gdl_gregistry * gr,
                           gdl_gvalues ** v,
                           size_t is_phased)
{
	gdl_gpoint * g;
	gdl_list_itr * itr;
	
	if (!gdl_list_empty (gr->hgpoints))
	{
	
		itr = gdl_list_iterator_front (gr->hgpoints);
		
		do {
			
			g = (gdl_gpoint *) gdl_list_iterator_value (itr);
			
			if (is_phased)
			{
				if (!_gregistry_compare_phased (gr, g->idx, v))
				{
					gdl_list_iterator_free (itr);
					return g;
				}			
			}
			else
			{
				if (!_gregistry_compare_unphased (gr, g->idx, v))
				{
					gdl_list_iterator_free (itr);
					return g;
				}
			}
		}
		while (gdl_list_iterator_next (itr));
		
		gdl_list_iterator_free (itr);
		
	}
	
	return NULL;
} 

static gdl_gpoint *
_gregistry_lookup_gconfig (gdl_gregistry * gr,
                           gdl_gvalues * v)
{
	gdl_gpoint * g;
	gdl_list_itr * itr;
	
	if (!gdl_list_empty (gr->ggpoints))
	{
	
	itr = gdl_list_iterator_front (gr->ggpoints);
	
	do {
		
		g = (gdl_gpoint *) gdl_list_iterator_value (itr);
		
		if (!_gregistry_compare (gr->gvalues, g->idx, v))
		{
			gdl_list_iterator_free (itr);
			return g;
		}
		else
		{
			continue;
		}
	}
	while (gdl_list_iterator_next (itr));
	
	gdl_list_iterator_free (itr);
	
	}
	
	return NULL;
} 

static gdl_gpoint * 
_gresitry_add_hgpoint (gdl_gregistry * gr, size_t idx)
{
	gdl_gpoint * new = gdl_gpoint_alloc (gdl_gpoint_haplo);
	new->idx = idx;
	gdl_list_push_back (gr->hgpoints, new, 1);
	return new;
}

static gdl_gpoint * 
_gresitry_add_ggpoint (gdl_gregistry * gr, size_t idx)
{
	gdl_gpoint * new = gdl_gpoint_alloc (gdl_gpoint_geno);
	new->idx = idx;
	gdl_list_push_back (gr->ggpoints, new, 1);
	return new;
}

static gdl_gpoint *
_gregistry_add_hconfig (gdl_gregistry * gr,
                        gdl_gvalues ** v)
{
	size_t i;
	
	_gresitry_grow_hconfig (gr, gr->nhc+1);
	
	for (i = 0; i < gr->ploidy; i++)
	{
		if (v[i] == NULL)
		{
			SET_HCONFIG(gr,gr->nhc-1,i,-1);
		}
		else
		{
			size_t idx = gdl_gvalues_registry_add (gr->hvalues, v[i]);
			SET_HCONFIG(gr, gr->nhc-1,i,idx);
		}
	}
	
	return _gresitry_add_hgpoint (gr, gr->nhc-1);
} 

static gdl_gpoint *
_gregistry_add_gconfig (gdl_gregistry * gr,
                        gdl_gvalues * v)
{
	int idx = gdl_gvalues_registry_add (gr->gvalues, v);
	return _gresitry_add_ggpoint (gr, idx);
}                        

static gdl_gpoint *
gdl_gregistry_add_haplo (gdl_gregistry * gr,
                         gdl_gregistry_entry * ge)
{
	gdl_gpoint * g = _gregistry_lookup_hconfig (gr, ge->values, ge->is_phased);
	if (g == NULL)
	{
		g = _gregistry_add_hconfig (gr, ge->values);
	}
	return g;
}

static gdl_gpoint *
gdl_gregistry_add_geno (gdl_gregistry * gr,
                        gdl_gregistry_entry * ge)
{
	gdl_gpoint * g = _gregistry_lookup_gconfig (gr, ge->values[0]);
	
	if (g == NULL)
	{
		g = _gregistry_add_gconfig (gr, ge->values[0]);
	}
	
	return g;
}

/*====================================================================
 *     
 *                          PUBLIC
 * 
 * ===================================================================
 */

gdl_gregistry *
gdl_gregistry_alloc (size_t ploidy)
{
	gdl_gregistry * gr;
	
	gr = GDL_MALLOC (gdl_gregistry, 1);
	
	if (gr == 0)
	{
		GDL_ERROR_VAL ("Unable to allocate memory in gdl_gregistry_alloc",
		               GDL_ENOMEM,
		               0);	
	}
	
	gr->ploidy = ploidy;
	gr->nhc    = 0;
	gr->hconfig  = GDL_MALLOC (size_t, 1);
	gr->hgpoints = gdl_list_alloc (gdl_gpoint_interface);
	gr->ggpoints = gdl_list_alloc (gdl_gpoint_interface);
	gr->gvalues  = gdl_gvalues_registry_alloc ();
	if (gr->gvalues == 0)
	{
		GDL_FREE (gr);
		GDL_ERROR_VAL ("Unable to allocate memory in gdl_gregistry_alloc",
		               GDL_ENOMEM,
		               0);	
	}
	gr->hvalues  = gdl_gvalues_registry_alloc ();
	if (gr->gvalues == 0)
	{
		gdl_gvalues_registry_free (gr->gvalues);
		GDL_FREE (gr);
		GDL_ERROR_VAL ("Unable to allocate memory in gdl_gregistry_alloc",
		               GDL_ENOMEM,
		               0);	
	}
	
	gr->gentry = gdl_gregistry_entry_alloc (gdl_gpoint_geno, 1);
	
	if (gr->gentry == 0)
	{
		gdl_gvalues_registry_free (gr->gvalues);
		gdl_gvalues_registry_free (gr->hvalues);
		GDL_FREE (gr);
		GDL_ERROR_VAL ("Unable to allocate memory in gdl_gregistry_alloc",
		               GDL_ENOMEM,
		               0);
	}
	
	gr->hentry = gdl_gregistry_entry_alloc (gdl_gpoint_haplo, ploidy);
	
	if (gr->hentry == 0)
	{
		
		gdl_gvalues_registry_free (gr->gvalues);
		gdl_gvalues_registry_free (gr->hvalues);
		gdl_gregistry_entry_free (gr->gentry);
		GDL_FREE (gr);
		GDL_ERROR_VAL ("Unable to allocate memory in gdl_gregistry_alloc",
		               GDL_ENOMEM,
		               0);
	}
	
	return gr;
}

void
gdl_gregistry_free (gdl_gregistry * gr)
{
	if (gr)
	{
		size_t i;
		GDL_FREE (gr->hconfig);
		gdl_list_free (gr->ggpoints);
		gdl_list_free (gr->hgpoints);
		gdl_gvalues_registry_free (gr->gvalues);
		gdl_gvalues_registry_free (gr->hvalues);
		for ( i = 0; i < gr->hentry->size; i++)
		{
			gr->hentry->values[i] = NULL;
		}
		gdl_gregistry_entry_free (gr->hentry);
		for ( i = 0; i < gr->gentry->size; i++)
		{
			gr->gentry->values[i] = NULL;
		}
		gdl_gregistry_entry_free (gr->gentry);
		GDL_FREE (gr);
	}
}



gdl_gpoint *
gdl_gregistry_add (gdl_gregistry * gr,
                        gdl_gregistry_entry * ge)
{
	if (ge->type == gdl_gpoint_haplo)
	{
		return gdl_gregistry_add_haplo (gr, ge);
	}	
	else if (ge->type == gdl_gpoint_geno)
	{
		return gdl_gregistry_add_geno (gr, ge);
	}
}

int
gdl_gregistry_add_gpoint (gdl_gregistry * gr, gdl_gpoint ** gp)
{
	gdl_gpoint * g;
	gdl_list * list;
	
	if ((*gp)->type == gdl_gpoint_haplo)
	{
		list = gr->hgpoints;
	}
	else if ((*gp)->type == gdl_gpoint_geno)
	{
		list = gr->ggpoints;	
	}
		
	g = (gdl_gpoint *) gdl_list_search (list, *gp);
	
	if (g == NULL)
	{
		gdl_list_push_back (list, *gp, 1);
	}
	else
	{
		gdl_gpoint_free (*gp);
		*gp = g;
	}
}

gdl_gregistry_entry *
gdl_gregistry_get (gdl_gregistry * gr, gdl_gpoint * gp)
{
	if (gp->type == gdl_gpoint_haplo)
	{
		size_t i;
		for (i = 0; i < gr->ploidy; i++)
		{
			int k = GET_HCONFIG(gr, gp->idx, i);
			if ( k == -1)
			{
				gr->hentry->values[i] = NULL;
			}
			else
			{
				gr->hentry->values[i] = gdl_gvalues_registry_get (gr->hvalues, (size_t)k);				
			}
		}
		return gr->hentry;
	}
	else if (gp->type == gdl_gpoint_geno)
	{
		gr->gentry->values[0] = gdl_gvalues_registry_get (gr->gvalues, gp->idx);
		return gr->gentry;
	}
}

gdl_boolean
gdl_gregistry_has_missing (gdl_gregistry * gr, gdl_gpoint * gp)
{
	if (gp->type == gdl_gpoint_haplo)
	{
		size_t i;
		int k;
		for (i = 0; i < gr->ploidy; i++)
		{
			k = GET_HCONFIG(gr, gp->idx, i);
			if (k == -1)
				return gdl_true;	
		}
		return gdl_false;
	}
	else if (gp->type == gdl_gpoint_geno)
	{
		return gdl_false;
	}
}

gdl_boolean
gdl_gregistry_is_missing (gdl_gregistry * gr, gdl_gpoint * gp, size_t p)
{
	if (p >= gr->ploidy)
	{
		return gdl_true;	
	}
	if (gp->type == gdl_gpoint_haplo)
	{
		int k = GET_HCONFIG(gr, gp->idx, p);
		if (k == -1)
		{
			return gdl_true;	
		}
		return gdl_false;
	}
	else if (gp->type == gdl_gpoint_geno)
	{
		return gdl_false;
	}
}

gdl_boolean
gdl_gregistry_is_homozygous (gdl_gregistry * gr, gdl_gpoint * gp)
{
	gdl_gvalues * x;
	
	if (gp->type == gdl_gpoint_haplo)
	{
		size_t i;
		int k, kold;
		for (i = 0; i < gr->ploidy; i++)
		{
			k = GET_HCONFIG(gr, gp->idx, i);
			if (k == -1 || i && k != kold)
				return gdl_false;
			x = gdl_gvalues_registry_get (gr->hvalues, (size_t)k);
			if (x->size > 1)
				return gdl_false;
			kold = k;
		}
		return gdl_true;
	}
	else if (gp->type == gdl_gpoint_geno)
	{
		return gdl_false;
	}
}

gdl_gregistry *
gdl_gregistry_fread (FILE * stream)
{
	if (stream == 0)
		return NULL;
	else
	{
		size_t i, p, status;
		gdl_gregistry * g;
		
		status = fread (&p, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		g = gdl_gregistry_alloc (p);
		
		status = fread (&g->nhc, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		if (g->nhc)
		{
			_gresitry_grow_hconfig (g, g->nhc);
			status = fread (g->hconfig, sizeof (size_t), g->nhc*g->ploidy, stream);
			GDL_FREAD_STATUS (status, g->nhc*g->ploidy);			
		}
		g->gvalues = gdl_gvalues_registry_fread (stream);
		GDL_FREAD_STATUS (g->gvalues != NULL, 1);
		g->hvalues = gdl_gvalues_registry_fread (stream);
		GDL_FREAD_STATUS (g->hvalues != NULL, 1);
		
		return g;
	}
}

int
gdl_gregistry_fwrite (FILE * stream, const gdl_gregistry * g)
{
	if (stream == 0 || g == 0)
		return GDL_EINVAL;
	else
	{
		size_t status;
		
		status = fwrite (&g->ploidy, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&g->nhc, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (g->hconfig, sizeof (size_t), g->nhc*g->ploidy, stream);
		GDL_FWRITE_STATUS (status, g->nhc*g->ploidy);
		status = gdl_gvalues_registry_fwrite (stream, g->gvalues);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_gvalues_registry_fwrite (stream, g->hvalues);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
	}
	return GDL_SUCCESS;
}

#undef GET_HCONFIG
#undef SET_HCONFIG
