/*  
 * 	gpoint/point.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:53 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this programnt_allo; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_gpoint.h>

enum _gdl_gpoint_type
{
	gpoint_haplo,
	gpoint_geno,
	gpoint_missing
};

gdl_gpoint * 
gdl_gpoint_alloc (const gdl_gpoint_type * type)
{
	gdl_gpoint * g = GDL_MALLOC (gdl_gpoint, 1);
	
	if (g)
	{
		g->type = type;
	}
	
	return g;
}

void
gdl_gpoint_free (gdl_gpoint * g)
{
	if (g)
	{
		g->type = NULL;
		GDL_FREE (g);
	}	
}

gdl_gpoint * 
gdl_gpoint_clone (const gdl_gpoint * g)
{
	if (g)
	{
		gdl_gpoint * c = gdl_gpoint_alloc (g->type);
		c->idx = g->idx;
		return c;
	}	
	return NULL;
}

int 
gdl_gpoint_compare (const gdl_gpoint * g1, const gdl_gpoint * g2)
{
	if (g1 != 0 && g2 != 0)
	{
		if (g1->type != g2->type)
			return (-1);
		if (g1->idx != g2->idx)
			return (-1);
		return (0);
	}	
	return (-1);
}

gdl_gpoint * 
gdl_gpoint_fread (FILE * stream)
{
	if (stream == 0)
		return NULL;
	else
	{
		int status;
		char type;
		gdl_gpoint * g;
		
		status = fread (&type, sizeof (char), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		switch (type)
		{
			case 'h':
				g = gdl_gpoint_alloc (gdl_gpoint_haplo);
				break;
			case 'g':
				g = gdl_gpoint_alloc (gdl_gpoint_geno);
				break;			
		}
		status = fread (&g->idx, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		return g;
	}
}

int
gdl_gpoint_fwrite (FILE * stream, const gdl_gpoint * g)
{
	if (stream == 0 || g == 0)
		return GDL_EINVAL;
	else
	{
		int status;
		char * type;
		
		if (g->type == gdl_gpoint_haplo)
		{
			type = "h";
		}
		else if (g->type == gdl_gpoint_geno)
		{
			type = "g";
		}
		status = fwrite (type, sizeof (char), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&g->idx, sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		return GDL_SUCCESS;
	}
}

static void
_gdl_gpoint_free (void * vg)
{
	gdl_gpoint * g = (gdl_gpoint *) vg;
	gdl_gpoint_free (g);
}

static void *
_gdl_gpoint_clone (const void * vg)
{
	gdl_gpoint * g = (gdl_gpoint *) vg;
	return gdl_gpoint_clone (g);
}

static int
_gdl_gpoint_compare (const void * vg1, const void * vg2)
{
	gdl_gpoint * g1 = (gdl_gpoint *) vg1;
	gdl_gpoint * g2 = (gdl_gpoint *) vg2;
	return gdl_gpoint_compare (g1, g2);
}

static void *
_gdl_gpoint_fread (FILE * stream)
{
	return gdl_gpoint_fread (stream);
}

static int
_gdl_gpoint_fwrite (FILE * stream, const void * vg)
{
	gdl_gpoint * g = (gdl_gpoint *) vg;
	return gdl_gpoint_fwrite (stream, g);
}

static const gdl_gpoint_type hgpoint = gpoint_haplo;
static const gdl_gpoint_type ggpoint = gpoint_geno;
static const gdl_gpoint_type mgpoint = gpoint_missing;
static const gdl_data_interface _gdl_gpoint_interface =
{
	&_gdl_gpoint_free,
	&_gdl_gpoint_clone,
	&_gdl_gpoint_compare,
	&_gdl_gpoint_fread,
	&_gdl_gpoint_fwrite
};

const gdl_gpoint_type * gdl_gpoint_haplo    = &hgpoint;
const gdl_gpoint_type * gdl_gpoint_geno     = &ggpoint;
const gdl_gpoint_type * gdl_gpoint_missing  = &mgpoint;
const gdl_data_interface * gdl_gpoint_interface = &_gdl_gpoint_interface;

