/*  
 * 	gpoint/value.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:53 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_gpoint.h>

gdl_gvalue *
gdl_gvalue_alloc (void)
{
	gdl_gvalue * g;
	
	g = GDL_MALLOC (gdl_gvalue, 1);
	
	return g;	
}

void
gdl_gvalue_free (gdl_gvalue * g)
{
	if (g)
		GDL_FREE (g);	
}

int
gdl_gvalue_compare (const gdl_gvalue * g1, const gdl_gvalue * g2)
{
	if (g1 == 0 || g2 == 0)
	{
		return (-1);
	}
	
	return (g1->value != g2->value || g1->idx != g2->idx);
}

gdl_gvalue *
gdl_gvalue_clone (const gdl_gvalue * g)
{
	if (g)
	{
		gdl_gvalue * c;
		
		c = gdl_gvalue_alloc ();
		
		c->idx = g->idx,
		c->value = g->value;
		
		return c;
	}
	return NULL;	
}

int
gdl_gvalue_copy (gdl_gvalue * dest, const gdl_gvalue * src)
{
	if (dest == NULL || src == NULL)
	{
		return (-1);	
	}
	{
		dest->idx   = src->idx;
		dest->value = src->value;	
		return (0);
	}
}

gdl_gvalue *
gdl_gvalue_fread (FILE * stream)
{
	if (stream == 0)
		return NULL;
	else
	{
		size_t status;
		gdl_gvalue * g = gdl_gvalue_alloc ();
		
		status = fread (&(g->idx), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(g->value), sizeof (double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		return g;	
	}
}

int
gdl_gvalue_fwrite (FILE * stream, const gdl_gvalue * g)
{
	size_t status;
	
	if (stream == 0 || g == 0)
		return GDL_EINVAL;
	
	status = fwrite (&(g->idx), sizeof (size_t), 1, stream);
	GDL_FWRITE_STATUS (status, 1);
	status = fwrite (&(g->value), sizeof (double), 1, stream);
	GDL_FWRITE_STATUS (status, 1);
	
	return GDL_SUCCESS;	
}

gdl_gvalues *
gdl_gvalues_alloc (size_t size)
{
	size_t i;
	gdl_gvalues * g;
	
	g = GDL_MALLOC (gdl_gvalues, 1);
	
	g->size = size;
	
	g->values = GDL_MALLOC (gdl_gvalue *, size);
	
	if (g->values == 0)
	{
		GDL_FREE (g);
		GDL_ERROR_VAL ("Enable to allocate memory in gdl_gvalues_alloc",
		               GDL_ENOMEM,
		               0);
	}
	for (i = 0; i < size; i++)
	{
		g->values[i] = NULL;
	}
	
	return g;	
}

void
gdl_gvalues_free (gdl_gvalues * g)
{
	if (g)
	{
	   if (g->values)
	   {
	   		size_t i;
	   		for (i = 0; i < g->size; i++) 
	   		{
	   			gdl_gvalue_free (g->values[i]);
	   		}
	   		GDL_FREE (g->values);
	   }
	   GDL_FREE (g);
	}	
}

gdl_gvalues *
gdl_gvalues_clone (const gdl_gvalues * g)
{
	if (g)
	{
		size_t i;
		gdl_gvalues * c;
		
		c = gdl_gvalues_alloc (g->size);
		
		for (i = 0; i < g->size; i++)
		{
			c->values[i] = gdl_gvalue_clone (g->values[i]);
		}
		
		return c;
	}
	return NULL;	
}

int
gdl_gvalues_copy (gdl_gvalues * dest, const gdl_gvalues * src)
{
	if (dest == NULL || src == NULL)
	{
		return (-1);	
	}
	else if (dest->size < src->size)
	{
		return (-1);
	}
	else
	{
		size_t i;
		
		dest->size = src->size;
		
		for (i = 0; i < src->size; i++)
		{
			gdl_gvalue_copy(dest->values[i], src->values[i]);
		}
		
		return (0);
	}
}

int
gdl_gvalues_compare (const gdl_gvalues * g1, const gdl_gvalues * g2)
{
	if (g1 == 0 || g2 == 0)
	{
		return (-1);
	}	
	if (g1->size != g2->size)
	{
		return (-1);
	}	
	else
	{
		size_t i, j, n;
		gdl_boolean * ok;
		
		ok = GDL_CALLOC (gdl_boolean, g1->size);
		n  = 0;
		
		for (i = 0; i < g1->size; i++)
		{
			for (j = 0; j < g2->size; j++)
			{
				if (ok[j])
				{
				   continue;
				}
				if (!gdl_gvalue_compare (g1->values[i], g2->values[j]))
				{
					ok[j] = gdl_true;
					n++;
					break;
				}
			}
			if (n != i+1)
			{
				GDL_FREE (ok);
				return (-1);	
			}
		}
		
		GDL_FREE (ok);
		
		return (0);
	}
}

const gdl_gvalue *
gdl_gvalues_get_max (const gdl_gvalues * g)
{
	if (!g) return 0;
	
	size_t i, maxi;
	double max = GDL_NEGINF;
	gdl_gvalue * x;
	
	for (maxi = i = 0; i < g->size; i++)
	{
		if (g->values[i]->value > max)
		{
			max  = g->values[i]->value;
			maxi = i;	
		}
	}
	
	return g->values[maxi];
}

gdl_gvalues *
gdl_gvalues_fread (FILE * stream)
{
	if (stream == 0)
	{
		return NULL;
	}
	else
	{
		size_t i, size;
		int status;
		
		gdl_gvalues * g;
		
		status = fread (&size, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		g = gdl_gvalues_alloc (size);
		
		for (i = 0; i < size; i++)
		{
			g->values[i] = gdl_gvalue_fread (stream);
			GDL_FREAD_STATUS (g->values[i] != 0, 1);
		}
		
		return g;	
	}
}

int
gdl_gvalues_fwrite (FILE * stream, const gdl_gvalues * g)
{
	if (stream == 0 || g == 0)
	{
		return GDL_EINVAL;
	}
	else
	{
		size_t i, status;
		
		status = fwrite (&(g->size), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < g->size; i++)
		{
			status = gdl_gvalue_fwrite (stream, g->values[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;	
	}	
}

static void
_gdl_gvalues_free (void * vg)
{
	gdl_gvalues * g = (gdl_gvalues *) vg;
	gdl_gvalues_free (g);	
}

static void *
_gdl_gvalues_clone (const void * vg)
{
	gdl_gvalues * g = (gdl_gvalues *) vg;
	return gdl_gvalues_clone (g);	
}

static int
_gdl_gvalues_compare (const void * vg1, const void * vg2)
{
	gdl_gvalues * g1 = (gdl_gvalues *) vg1;
	gdl_gvalues * g2 = (gdl_gvalues *) vg2;
	return gdl_gvalues_compare (g1, g2);
}

static void *
_gdl_gvalues_fread (FILE * stream)
{
	return gdl_gvalues_fread (stream);
}

static int
_gdl_gvalues_fwrite (FILE * stream, const void * vg)
{
	gdl_gvalues * g = (gdl_gvalues *) vg;
	return gdl_gvalues_fwrite (stream, g);
}

static const gdl_data_interface _gdl_gvalues_interface =
{
   &_gdl_gvalues_free,
   &_gdl_gvalues_clone,
   &_gdl_gvalues_compare,
   &_gdl_gvalues_fread,
   &_gdl_gvalues_fwrite		
};

const gdl_data_interface * gdl_gvalues_interface = &_gdl_gvalues_interface;
