/*  
 *  graphics/transform2D.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:52 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <math.h> 

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_graphics_transform.h>
/**
 * 
 */
gdl_affine_transform_ptr
gdl_affine_transform_alloc()
{
  gdl_affine_transform_ptr  gptr = NULL;

  gptr = GDL_MALLOC (gdl_affine_transform, 1);

  gptr->type         = GDL_GRAPHICS_IDENTITY;
  gptr->matrix	      = GDL_MATRIX_ALLOC (double, 3, 3);
  gptr->matrix[0][0] = 1;
  gptr->matrix[1][1] = 1;
  gptr->matrix[2][2] = 1;
  gptr->next         = NULL;
  gptr->prev         = NULL;
  
  return gptr;
}
/**
 * 
 */
void
gdl_affine_transform_free(gdl_affine_transform_ptr ptr)
{
	if (ptr != NULL) {
		if (ptr->matrix != NULL) {
			GDL_MATRIX_FREE (ptr->matrix, 3);
		}
		free(ptr);
	}
}
/**
 * 
 */
void
gdl_affine_transform_new(gdl_affine_transform_ptr *grp)
{
	if (*grp == NULL)
		*grp = gdl_affine_transform_alloc();
	else {
		(*grp)->next = gdl_affine_transform_alloc();
		(*grp)->next->prev = *grp;
		*grp = (*grp)->next;
	}	
}
/**
 * 
 */
gdl_affine_transform_ptr gdl_affine_transform_first(gdl_affine_transform_ptr gptr)
{
	if (gptr != NULL)
		while(gptr->prev != NULL)
			gptr = gptr->prev;
	return gptr;	
}
/**
 * 
 */
gdl_affine_transform_ptr gdl_affine_transform_last(gdl_affine_transform_ptr gptr)
{
	if (gptr != NULL)
		while(gptr->next != NULL)
			gptr = gptr->next;
	return gptr;	
}
/**
 * 
 */
void
gdl_affine_transform_clear(gdl_affine_transform_ptr gptr)
{
  gdl_affine_transform_ptr gtptr;
  gtptr = gptr;
  
  if ( gtptr != NULL ) {
  	
  	while ((gtptr = gdl_affine_transform_delete(gtptr)) != NULL);
  	
  }
  
  gptr = NULL;	
}
/**
 * 
 */
gdl_affine_transform_ptr
gdl_affine_transform_delete(gdl_affine_transform_ptr gptr)
{
  
  gdl_affine_transform_ptr lgptr, rgptr ;
  if ( gptr == NULL )
    return(NULL);
  lgptr = gdl_affine_transform_move_to_front(gptr);
  rgptr = lgptr->next;
  if (rgptr != NULL )
     rgptr->prev = NULL; 
     
  gdl_affine_transform_free(lgptr);
  lgptr=NULL;
  
  return(rgptr);
}
/**
 * 
 * 
 */
gdl_affine_transform_ptr
gdl_affine_transform_move_to_front(gdl_affine_transform_ptr gptr)
{
  gdl_affine_transform_ptr lgptr;
  if (gptr->prev == NULL )/* If at the front, leave it. */
    return(gptr);
  lgptr=gptr;
  while (lgptr->prev != NULL )  /* find first node */
    lgptr = lgptr->prev;
  if (gptr->prev != NULL )
    gptr->prev->next = gptr->next;
  if (gptr->next != NULL )
    gptr->next->prev = gptr->prev;
  lgptr->prev = gptr;
  gptr->next = lgptr;
  gptr->prev = NULL;
  return(gptr);
}
/**
 * 
 */
gdl_affine_transform_ptr
gdl_affine_transform_clone(gdl_affine_transform_ptr ptr)
{
	int i,j;
	gdl_affine_transform_ptr tptr = NULL;;
	gdl_affine_transform_ptr gptr = NULL;;
	
	for(tptr=ptr;tptr!=NULL;tptr=tptr->next) {
		gdl_affine_transform_new(&gptr);
		gptr->type = tptr->type;
		for(i=0;i<3;i++)
			for(j=0;j<3;j++)
				gptr->matrix[i][j]=tptr->matrix[i][j];
	}
	
	return gdl_affine_transform_first(gptr);
}
/**
 * he matrix representing the returned transform is:
 *  [   cos(theta)    -sin(theta)    0   ]
 *  [   sin(theta)     cos(theta)    0   ]
 *  [       0              0         1   ]
 * 
 */
gdl_affine_transform_ptr
gdl_affine_transform_rotation(double theta)
{
	gdl_affine_transform_ptr t = gdl_affine_transform_alloc();
	t->type                    = GDL_GRAPHICS_ROTATION;
	t->matrix[0][0]            = cos(theta);
	t->matrix[0][1]            = -sin(theta);
	t->matrix[1][0]            = sin(theta);
	t->matrix[1][1]            = cos(theta);
	return t;
}
/**
 * The matrix representing the returned transform is:

                [   cos(theta)    -sin(theta)    x-x*cos+y*sin  ]
                [   sin(theta)     cos(theta)    y-x*sin-y*cos  ]
                [       0              0               1        ]
*/
gdl_affine_transform_ptr
gdl_affine_transform_anchor_rotation(double x, double y, double theta)
{
	gdl_affine_transform_ptr t = gdl_affine_transform_rotation(theta);
	t->matrix[0][2] = x - x*cos(theta)+y*sin(theta);
	t->matrix[1][2] = y - x*sin(theta)+y*cos(theta);
	return t;
}
/**
 * the matrix representing the returned transform is:
 * 
 * [   1    0    tx  ]
   [   0    1    ty  ]
   [   0    0    1   ]
 */
gdl_affine_transform_ptr
gdl_affine_transform_translation(double tx, double ty)
{
	gdl_affine_transform_ptr t = gdl_affine_transform_alloc();
	t->type                    = GDL_GRAPHICS_TRANSLATION;
	t->matrix[0][2]            = tx;
	t->matrix[1][2]            = ty;
	return t;
}
/**
 * the matrix representing the returned transform is:
 * 
 * [   sx    0    0  ]
   [   0    sx    0  ]
   [   0    0     1  ]
 */
gdl_affine_transform_ptr
gdl_affine_transform_scale(double sx, double sy)
{
	gdl_affine_transform_ptr t = gdl_affine_transform_alloc();
	t->type                    = GDL_GRAPHICS_SCALE;
	t->matrix[0][0]            = sx;
	t->matrix[1][1]            = sy;
	return t;
}
/**
 * [ x']   [  m00  m01  m02  ] [ x ]   [ m00x + m01y + m02 ]
        [ y'] = [  m10  m11  m12  ] [ y ] = [ m10x + m11y + m12 ]
        [ 1 ]   [   0    0    1   ] [ 1 ]   [         1         ]
*/ 
void gdl_affine_transform_apply(gdl_affine_transform_ptr transform, double x, double y, double *tx, double *ty)
{
	double xx,yy;
	gdl_affine_transform_ptr tptr;
	
	*tx = x;
	*ty = y;
	
	for(tptr=transform;tptr!=NULL;tptr=tptr->prev) {
		if (tptr->type == GDL_GRAPHICS_IDENTITY) continue;
		*tx = tptr->matrix[0][0]*(*tx)+tptr->matrix[0][1]*(*ty)+tptr->matrix[0][2];
		*ty = tptr->matrix[1][0]*(*tx)+tptr->matrix[1][1]*(*ty)+tptr->matrix[1][2];
	}
}

/**
 * 
 * 
 */
gdl_affine_transform_ptr gdl_affine_transform_concat(gdl_affine_transform_ptr t1,
                                                     gdl_affine_transform_ptr t2)
{
	gdl_affine_transform_ptr tptr;
	
	if (t1 == NULL) return t2;
	if (t2 == NULL) return t1;
	
	tptr = gdl_affine_transform_last(t1);
	tptr->next = t2;
	t2->prev   = tptr;
	
	return t1;
}
