/*  
 * 	gview/cell.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:43 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <string.h>
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_gpoint.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_reader.h>

#include "unit.c"

struct _gdl_gview_reader_cells 
{
	gdl_hashtable * hash;
};

static void
_gdl_gview_reader_cell_free (void * vr)
{
	gdl_gview_reader_cell * r = 	(gdl_gview_reader_cell *) vr;
	gdl_gview_reader_cell_free (r);
}

static const gdl_data_interface _gdl_gview_reader_cell_interface =
{
	&_gdl_gview_reader_cell_free,
	NULL,
	NULL,
	NULL,
	NULL
};

static gdl_gdatapoint *
_gdl_gview_reader_cell_hpoint (gdl_gview_reader_cell * cell)
{
	size_t i, j, na, tna = 0;
	gdl_gdatapoint * g;
	
	g = gdl_gdatapoint_alloc (gdl_gpoint_haplo, cell->size);
	
	g->is_phased = cell->is_phased;
	
	for (i = 0; i < cell->size; i++)
	{
		if (!gdl_list_empty (cell->units[i]))
		{
			gdl_list_itr * itr 
				= gdl_list_iterator_front (cell->units[i]);
			
			gdl_gvalues * gvs = g->values[i]
				= gdl_gvalues_alloc (gdl_list_size (cell->units[i]));
			
			j  = 0;
			na = 0;
			
			do
			{
				gdl_gview_reader_gpoint * unit 
				       = (gdl_gview_reader_gpoint *) gdl_list_iterator_value (itr);
				
				gdl_gvalue * gv = NULL;
				
				if (unit->alleles[0])
				{
				    gv        = gdl_gvalue_alloc();
					gv->idx   = unit->idx;
					gv->value = unit->value;
				}
				else
				{
					na++;					
				}
				gvs->values[j] = gv;
				
				j++;
			}
			while (gdl_list_iterator_next (itr));
			
			gdl_list_iterator_free (itr);
			
			if (na == gdl_list_size (cell->units[i]))
			{
				gdl_gvalues_free (g->values[i]);
				g->values[i] = NULL;
				tna++;
			}
			
		}
	
	}
	
	if (tna == cell->size)
	{
		gdl_gdatapoint_free (g);
		g = NULL;	
	}
	
	return g;
}

static gdl_gdatapoint *
_gdl_gview_reader_cell_gpoint (gdl_gview_reader_cell * cell)
{
	size_t i;
	gdl_gdatapoint * g;
	
	g = gdl_gdatapoint_alloc (gdl_gpoint_geno, 1);
	
	if (!gdl_list_empty (cell->units[0]))
	{
		gdl_list_itr * itr 
			= gdl_list_iterator_front (cell->units[0]);
		
		gdl_gvalues * gvs = g->values[0]
			= gdl_gvalues_alloc (gdl_list_size (cell->units[0]));
		
		i = 0;
		
		do
		{
			gdl_gview_reader_gpoint * unit 
			       = (gdl_gview_reader_gpoint *) gdl_list_iterator_value (itr);
			
			gdl_gvalue * gv;
			
		    gv        = gdl_gvalue_alloc();
			gv->idx   = unit->idx;
			gv->value = unit->value;
			
			gvs->values[i] = gv;
			
			i++;
		}
		while (gdl_list_iterator_next (itr));
		
		gdl_list_iterator_free (itr);
		
	}
	
	return g;
}

static gdl_gdatapoint *
_gdl_gview_reader_cell_gdatapoint (gdl_gview_reader_cell * cell)
{
	if (cell == NULL)
	{
		return NULL;
	}	
	else if (cell->type == gdl_gpoint_haplo)
	{
		return _gdl_gview_reader_cell_hpoint (cell);
	}
	else if (cell->type == gdl_gpoint_geno)
	{
		return _gdl_gview_reader_cell_gpoint  (cell);
	}
	else
	{
		return NULL;
	}	
}

gdl_gview_reader_cell *
gdl_gview_reader_cell_alloc (const gdl_gpoint_type * type, size_t size)
{
	size_t i;
	gdl_gview_reader_cell * r;
	
	r = GDL_MALLOC (gdl_gview_reader_cell, 1);
	
	if (r == 0)
	{
		GDL_ERROR_VAL ("Unable to allocate memory in gdl_gview_reader_cell_alloc",
		               GDL_ENOMEM,
		               0);	
	}
	
	r->point     = NULL;
	r->size      = size;
	r->type      = type;
	r->units     = GDL_MALLOC (gdl_list *, size);
	if (r->units == 0)
	{
		GDL_FREE (r);
		GDL_ERROR_VAL ("Unable to allocate memory in gdl_gview_reader_cell_alloc",
		               GDL_ENOMEM,
		               0);		
	}
	for (i = 0; i < size; i++)
	{  
		r->units[i] = gdl_list_alloc (gdl_gview_reader_gpoint_interface);
		
		if (r->units[i] == 0)
		{
			size_t j;
			for (j = 0; j < i; j++)
			{
				gdl_list_free (r->units[j]);
			}
			GDL_FREE (r);
			GDL_ERROR_VAL ("Unable to allocate memory in gdl_gview_reader_cell_alloc",
		               GDL_ENOMEM,
		               0);		
		}
	}
	return r;
}

gdl_gview_reader_cell *
gdl_gview_reader_parse_cell (gdl_gview_reader * reader, const gdl_string * cell)
{
	if (reader == 0 || cell == 0)
	{
		return NULL;
	}
	else if (gdl_string_is_space (cell))
	{
		return NULL;
	}
	else
	{
		gdl_gview_reader_cell * r = (reader->type->cell)(reader, cell);
		return r;
	}
}

void
gdl_gview_reader_cell_free (gdl_gview_reader_cell * r)
{
	if (r)
	{
		if (r->units)
		{
			size_t i;
			
			for (i = 0; i < r->size; i++)
			{
				gdl_list_free (r->units[i]);
			}
			GDL_FREE (r->units);
		}
		
		gdl_gdatapoint_free (r->point);
		
		GDL_FREE (r);
	}
}

gdl_gdatapoint *
gdl_gview_reader_cell_gdatapoint (gdl_gview_reader_cell * cell)
{
	if (cell->point == NULL)
	{
		cell->point = _gdl_gview_reader_cell_gdatapoint (cell);
	}
	return cell->point;
}


gdl_gview_reader_cells *
gdl_gview_reader_cells_alloc ()
{
	gdl_gview_reader_cells * b;
	
	b = GDL_MALLOC (gdl_gview_reader_cells, 1);
	
	b->hash = gdl_hashtable_alloc (&_gdl_gview_reader_cell_interface, 0);
	
	return b;
}

void
gdl_gview_reader_cells_free (gdl_gview_reader_cells * b)
{
	if (b)
	{
		gdl_hashtable_free (b->hash);
		GDL_FREE (b);
	}
}

gdl_gview_reader_cell * 
gdl_gview_reader_cells_lookup (gdl_gview_reader_cells * b,
                                const char * data)
{
	return 	(gdl_gview_reader_cell *) gdl_hashtable_lookup (b->hash, data);
}                                

size_t
gdl_gview_reader_cells_add (gdl_gview_reader_cells * b,
                             const char * data,
                             gdl_gview_reader_cell * r)
{
	return gdl_hashtable_add (b->hash, data, r, 1);
}                        

