/*  
 * 	gview/gdl_gview.h
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:43 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#ifndef __GDL_GVIEW_H__
#define __GDL_GVIEW_H__

#include <gdl/gdl_common.h>
#include <gdl/gdl_meta.h>
#include <gdl/gdl_gblock.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_gmap.h>
#include <gdl/gdl_gpoint.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_clustering.h>
#include <gdl/gdl_mask.h>

__BEGIN_DECLS

typedef struct
{
	gdl_string * name;
	size_t       size;
	int (*alloc) (void * data);
	void (*free) (void * data);
	int (*init) (void * data, size_t na, size_t nl, size_t p);
	gdl_locus * (*add_locus) (void * data, const gdl_string * name);
	gdl_accession * (*add_accession) (void * data, const gdl_string * name);
	size_t (*ploidy) (const void * data);
	size_t (*locus_size) (const void * data);
	size_t (*accession_size) (const void * data);
	gdl_locus * (*get_locus) (const void * data, size_t i);
	gdl_accession * (*get_accession) (const void * data, size_t i);
	gdl_locus * (*search_locus) (const void * data, const gdl_string * name);
	gdl_accession * (*search_accession) (const void * data, const gdl_string * name);
	gdl_gvalues_get * (*get_new)(const void * data);
	int (*set_gdatapoint) (void * data, const gdl_accession * a, const gdl_locus * l, gdl_gdatapoint * gd);
	int (*genotype) (const void * data, const gdl_accession * a, const gdl_locus * l, gdl_gvalues_get * gb);
	int (*allele) (const void * data, const gdl_accession * a, const gdl_locus * l, size_t p, gdl_gvalues_get * gb);
	gdl_boolean (*has_missing) (const void * data, const gdl_accession * a, const gdl_locus * l);
	gdl_boolean (*is_missing) (const void * data, const gdl_accession * a, const gdl_locus * l, size_t p);
	gdl_boolean (*is_homozygous) (const void * data, const gdl_accession * a, const gdl_locus * l);
	int (*template) (const void * data, const gdl_accession * a, const gdl_locus * l, gdl_gvalues_get * gb);
	int (*accession_compare) (const void * data, const gdl_accession * a1, const gdl_accession * a2, const gdl_locus_mask * m);
	int (*locus_compare) (const void * data, const gdl_locus * l1, const gdl_locus * l2, const gdl_accession_mask * m);
	int (*genotype_compare) (const void * data, const gdl_accession * a1, const gdl_locus * l1, const gdl_accession * a2, const gdl_locus * l2);
	int (*set_gdatapoint_f) (void * data, size_t a, size_t l, gdl_gdatapoint * gd);
	int (*genotype_f) (const void * data, size_t a, size_t l, gdl_gvalues_get * gb);
	int (*allele_f) (const void * data, size_t a, size_t l, size_t p, gdl_gvalues_get * gb);
	gdl_boolean (*has_missing_f) (const void * data, size_t a, size_t l);
	gdl_boolean (*is_missing_f) (const void * data, size_t a, size_t l, size_t p);
	gdl_boolean (*is_homozygous_f) (const void * data, size_t a, size_t l);
	int (*template_f) (const void * data, size_t a, const gdl_locus * l, gdl_gvalues_get * gb);
	int (*accession_compare_f) (const void * data, size_t a1, size_t a2, const gdl_locus_mask * m);
	int (*locus_compare_f) (const void * data, size_t l1, size_t l2, const gdl_accession_mask * m);
	int (*genotype_compare_f) (const void * data, size_t a1, size_t l1, size_t a2, size_t l2);
	void * (*fread)(FILE * stream);
	int (*fwrite)(FILE * stream, void * data);
	int (*add) (void * dest, const gdl_string * name, void * src, size_t owner);
} gdl_gview_type;

typedef struct
{
	const gdl_gview_type * type;
	void * data;
	gdl_meta  * meta;
} gdl_gview;

gdl_gview * gdl_gview_alloc (const gdl_gview_type * T);
void gdl_gview_free (gdl_gview * v);

int gdl_gview_init (gdl_gview * g, size_t na, size_t nl, size_t p);

gdl_gview * gdl_gview_fread (FILE * stream);
int gdl_gview_fwrite (FILE * stream, gdl_gview * g);

size_t gdl_gview_ploidy (const gdl_gview * g);
size_t gdl_gview_locus_size (const gdl_gview * g);
size_t gdl_gview_accession_size (const gdl_gview * g);

gdl_locus     * gdl_gview_add_locus (gdl_gview * g, const char * name);
gdl_accession * gdl_gview_add_accession (gdl_gview * g, const char * name);

gdl_locus     * gdl_gview_get_locus (const gdl_gview * g, size_t idx);
gdl_accession * gdl_gview_get_accession (const gdl_gview * g, size_t idx);
gdl_locus     * gdl_gview_search_locus (const gdl_gview * g, const char * name);
gdl_accession * gdl_gview_search_accession (const gdl_gview * g, const char * name);

gdl_gvalues_get * gdl_gview_get_new (const gdl_gview * g);

int gdl_gview_set_gdatapoint (gdl_gview * g, const gdl_accession * va, const gdl_locus * vl, gdl_gdatapoint * gd);

int gdl_gview_genotype (const gdl_gview * g, const gdl_accession * va, const gdl_locus * vl, gdl_gvalues_get * gb);
int gdl_gview_allele (const gdl_gview * g, const gdl_accession * va, const gdl_locus * vl, size_t p, gdl_gvalues_get * gb);
gdl_boolean gdl_gview_has_missing (const gdl_gview * g, const gdl_accession * va, const gdl_locus * vl);
gdl_boolean gdl_gview_is_missing (const gdl_gview * g, const gdl_accession * va, const gdl_locus * vl, size_t p);
gdl_boolean gdl_gview_is_homozygous (const gdl_gview * g, const gdl_accession * va, const gdl_locus * vl);
int gdl_gview_genotype_template (const gdl_gview * g, const gdl_accession * va, const gdl_locus * vl, gdl_gvalues_get * gb);
int gdl_gview_accession_compare (const gdl_gview * g, const gdl_accession * a1, const gdl_accession * a2, const gdl_locus_mask * m);
int gdl_gview_locus_compare (const gdl_gview * g, const gdl_locus * l1, const gdl_locus * l2, const gdl_accession_mask * m);
int gdl_gview_genotype_compare (const gdl_gview * g, const gdl_accession * a1, const gdl_locus * l1, const gdl_accession * a2, const gdl_locus * l2);

int gdl_gview_set_gdatapoint_f (gdl_gview * g, size_t a, size_t l, gdl_gdatapoint * gd);

int gdl_gview_genotype_f (const gdl_gview * g, size_t i, size_t j, gdl_gvalues_get * gb);
int gdl_gview_allele_f (const gdl_gview * g, size_t i, size_t j, size_t k, gdl_gvalues_get * gb);
gdl_boolean gdl_gview_has_missing_f (const gdl_gview * g, size_t i, size_t j);
gdl_boolean gdl_gview_is_missing_f (const gdl_gview * g, size_t i, size_t j, size_t k);
gdl_boolean gdl_gview_is_homozygous_f (const gdl_gview * g, size_t i, size_t j);
int gdl_gview_genotype_template_f (const gdl_gview * g, size_t i, const gdl_locus * l, gdl_gvalues_get * gb);
int gdl_gview_accession_compare_f (const gdl_gview * g, size_t i, size_t j, const gdl_locus_mask * m);
int gdl_gview_locus_compare_f (const gdl_gview * g, size_t i, size_t j, const gdl_accession_mask * m);
int gdl_gview_genotype_compare_f (const gdl_gview * g, size_t i, size_t ii, size_t j, size_t jj);

gdl_vector * gdl_gview_genotype_freq (const gdl_gview * g, const gdl_locus * locus, gdl_gvalues_get * gb);
gdl_vector * gdl_gview_allele_freq (const gdl_gview * g, const gdl_locus * locus, gdl_gvalues_get * gb);
gdl_vector * gdl_gview_genotype_freq_f (const gdl_gview * g, size_t i, gdl_gvalues_get * gb);
gdl_vector * gdl_gview_allele_freq_f (const gdl_gview * g, size_t i, gdl_gvalues_get * gb);

size_t gdl_gview_accession_missing_size (const gdl_gview * g, const gdl_accession * a);
size_t gdl_gview_locus_missing_size (const gdl_gview * g, const gdl_locus * l);
size_t gdl_gview_gpoint_missing_size (const gdl_gview * g, const gdl_accession * a, const gdl_locus * l);

size_t gdl_gview_accession_missing_size_f (const gdl_gview * g, size_t a);
size_t gdl_gview_locus_missing_size_f (const gdl_gview * g, size_t l);
size_t gdl_gview_gpoint_missing_size_f (const gdl_gview * g, size_t a, size_t l);


gdl_clustering * gdl_gview_accession_clustering (const gdl_gview * g, const gdl_mask * m);
gdl_gmap * gdl_gview_extract_gmap (const gdl_gview * g, const gdl_mask * m, const gdl_gmap * map);
gdl_gmap * gdl_gview_extract_gmap_clone (const gdl_gview * g, const gdl_mask * m, const gdl_gmap * map);

int gdl_gview_add (gdl_gview * g, const gdl_string * name, gdl_gview * s, size_t owner);
const gdl_gview * gdl_gview_virtual_get (const gdl_gview * g, const gdl_string * name);

GDL_VAR const gdl_data_interface * gdl_gview_interface;

GDL_VAR const gdl_gview_type * gdl_gview_standard;
GDL_VAR const gdl_gview_type * gdl_gview_virtual;

__END_DECLS

#endif /* __GDL_GVIEW_H__ */
