/*  
 *  gview/gview.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:43 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_meta.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_gblock.h>
#include <gdl/gdl_gmap.h>
#include <gdl/gdl_gpoint.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_clustering.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview.h>

static const gdl_gview_type *
gdl_gview_type_fread (FILE * stream)
{
	const gdl_gview_type * T;
	gdl_string * st;
	
	st = gdl_string_fread (stream);
	
	if (!strcmp (st, gdl_gview_standard->name))
	{
		T = gdl_gview_standard;
	}
	else if (!strcmp (st, gdl_gview_virtual->name))
	{
		T = gdl_gview_virtual;
	}
	
	gdl_string_free (st);
	
	return T;	
}

static int
gdl_gview_type_fwrite (FILE * stream, const gdl_gview_type * T)
{
	return gdl_string_fwrite (stream, T->name);
}

gdl_gview *
gdl_gview_alloc (const gdl_gview_type * T)
{
	gdl_gview * g = GDL_MALLOC (gdl_gview, 1);
	
	g->type = T;
	
	g->data = gdl_malloc (T->size);
	
	if (g->data)
	{
		(T->alloc)(g->data);
	}
	else
	{
		GDL_FREE (g);
		g = NULL;
	}
		
	return g;
}

void
gdl_gview_free (gdl_gview * g)
{
	if (g)
	{
		(g->type->free)(g->data);
		GDL_FREE (g->data);
		GDL_FREE (g);
	}	
}

int
gdl_gview_init (gdl_gview * g, size_t na, size_t nl, size_t p)
{
	return (g->type->init)(g->data, na, nl, p);
}

gdl_gview *
gdl_gview_fread (FILE * stream)
{
	const gdl_gview_type * T = gdl_gview_type_fread (stream);
	gdl_gview * g = gdl_gview_alloc (T);
	(g->type->free)(g->data);
	GDL_FREE (g->data);
	g->data = (g->type->fread)(stream);
	return g;
}

int
gdl_gview_fwrite (FILE * stream, gdl_gview * g)
{
	gdl_gview_type_fwrite (stream, g->type);
	return (g->type->fwrite)(stream, g->data);
}

size_t
gdl_gview_ploidy (const gdl_gview * g)
{
	return (g->type->ploidy)(g->data);
}

size_t
gdl_gview_locus_size (const gdl_gview * g)
{
	return (g->type->locus_size)(g->data);
}

size_t
gdl_gview_accession_size (const gdl_gview * g)
{
	return (g->type->accession_size)(g->data);	
}

gdl_locus *
gdl_gview_add_locus (gdl_gview * g, const gdl_string * name)
{
	return (g->type->add_locus) (g->data, name);
}

gdl_accession *
gdl_gview_add_accession (gdl_gview * g, const gdl_string * name)
{
	return (g->type->add_accession) (g->data, name);
}

gdl_locus *
gdl_gview_get_locus (const gdl_gview * g, size_t i)
{
	return (g->type->get_locus)(g->data, i);
}

gdl_accession *
gdl_gview_get_accession (const gdl_gview * g, size_t i)
{
	return (g->type->get_accession)(g->data, i);	
}

gdl_locus *
gdl_gview_search_locus (const gdl_gview * g, const gdl_string * name)
{
	return (g->type->search_locus)(g->data, name);
}

gdl_accession *
gdl_gview_search_accession (const gdl_gview * g, const gdl_string * name)
{
	return (g->type->search_accession)(g->data, name);
}

gdl_gvalues_get *
gdl_gview_get_new (const gdl_gview * g)
{
	return (g->type->get_new)(g->data);
}

int
gdl_gview_set_gdatapoint (gdl_gview * g, const gdl_accession * va, const gdl_locus * vl, gdl_gdatapoint * gd)
{
	return (g->type->set_gdatapoint)(g->data, va, vl, gd);
}

int
gdl_gview_genotype (const gdl_gview * g, const gdl_accession * a, const gdl_locus * l, gdl_gvalues_get * gb)
{
	return (g->type->genotype)(g->data, a, l, gb);
}

int
gdl_gview_allele (const gdl_gview * g, const gdl_accession * a, const gdl_locus * l, size_t p, gdl_gvalues_get * gb)
{
	return (g->type->allele)(g->data, a, l, p, gb);
}

int
gdl_gview_genotype_template (const gdl_gview * g, const gdl_accession * a, const gdl_locus * l, gdl_gvalues_get * gb)
{
	return (g->type->template)(g->data, a, l, gb);
}

gdl_boolean
gdl_gview_has_missing (const gdl_gview * g, const gdl_accession * a, const gdl_locus * l)
{
	return (g->type->has_missing)(g->data, a, l);	
}

gdl_boolean
gdl_gview_is_missing (const gdl_gview * g, const gdl_accession * a, const gdl_locus * l, size_t p)
{
	return (g->type->is_missing)(g->data, a, l, p);
}

gdl_boolean
gdl_gview_is_homozygous (const gdl_gview * g, const gdl_accession * a, const gdl_locus * l)
{
	return (g->type->is_homozygous)(g->data, a, l);
}

int
gdl_gview_accession_compare (const gdl_gview * g, const gdl_accession * a1, const gdl_accession * a2, const gdl_locus_mask * m)
{
	return (g->type->accession_compare)(g->data, a1, a2, m);	
}

int
gdl_gview_locus_compare (const gdl_gview * g, const gdl_locus * l1, const gdl_locus * l2, const gdl_accession_mask * m)
{
	return (g->type->locus_compare)(g->data, l1, l2, m);		
}

int
gdl_gview_genotype_compare (const gdl_gview * g, const gdl_accession * a1, const gdl_locus * l1, const gdl_accession * a2, const gdl_locus * l2)
{
	return (g->type->genotype_compare)(g->data, a1, l1, a2, l2);
}

size_t
gdl_gview_accession_missing_size (const gdl_gview * g, const gdl_accession * a)
{
	return gdl_gview_accession_missing_size_f (g, a->idx);	
}

size_t
gdl_gview_locus_missing_size (const gdl_gview * g, const gdl_locus * l)
{
	return gdl_gview_locus_missing_size_f (g, l->idx);
}

size_t
gdl_gview_gpoint_missing_size (const gdl_gview * g, const gdl_accession * a, const gdl_locus * l)
{
	return gdl_gview_gpoint_missing_size_f (g, a->idx, l->idx);
}

int
gdl_gview_set_gdatapoint_f (gdl_gview * g, size_t a, size_t l, gdl_gdatapoint * gd)
{
	return (g->type->set_gdatapoint_f)(g->data, a, l, gd);
}

int
gdl_gview_genotype_f (const gdl_gview * g, size_t i, size_t j, gdl_gvalues_get * gb)
{
	return (g->type->genotype_f)(g->data, i, j, gb);
}

int
gdl_gview_allele_f (const gdl_gview * g, size_t i, size_t j, size_t k, gdl_gvalues_get * gb)
{
	return (g->type->allele_f)(g->data, i, j, k, gb);
}

int
gdl_gview_genotype_template_f (const gdl_gview * g, size_t i, const gdl_locus * l, gdl_gvalues_get * gb)
{
	return (g->type->template_f)(g->data, i, l, gb);
}

gdl_boolean
gdl_gview_has_missing_f (const gdl_gview * g, size_t i, size_t j)
{
	return (g->type->has_missing_f)(g->data, i, j);
}

gdl_boolean
gdl_gview_is_missing_f (const gdl_gview * g, size_t i, size_t j, size_t p)
{
	return (g->type->is_missing_f)(g->data, i, j, p);
}

gdl_boolean
gdl_gview_is_homozygous_f (const gdl_gview * g, size_t i, size_t j)
{
	return (g->type->is_homozygous_f)(g->data, i, j);
}

int
gdl_gview_accession_compare_f (const gdl_gview * g, size_t i, size_t j, const gdl_locus_mask * m)
{
	return (g->type->accession_compare_f)(g->data, i, j, m);
}

int
gdl_gview_locus_compare_f (const gdl_gview * g, size_t i, size_t j, const gdl_accession_mask * m)
{
	return (g->type->locus_compare_f)(g->data, i, j, m);
}

int
gdl_gview_genotype_compare_f (const gdl_gview * g, size_t i, size_t ii, size_t j, size_t jj)
{
	return (g->type->genotype_compare_f)(g->data, i, ii, j, jj);
}

size_t
gdl_gview_accession_missing_size_f (const gdl_gview * g, size_t a)
{
	size_t i, nl, nm=0;
	
	nl = gdl_gview_locus_size (g);
	
	for (i = 0; i < nl; i++)
	{
		nm += gdl_gview_gpoint_missing_size_f (g, a, i);
	}
	
	return nm;	
}

size_t
gdl_gview_locus_missing_size_f (const gdl_gview * g, size_t l)
{
	size_t i, na, nm=0;
	
	na = gdl_gview_accession_size (g);
	
	for (i = 0; i < na; i++)
	{
		nm += gdl_gview_gpoint_missing_size_f (g, i, l);
	}
	
	return nm;	
}

size_t
gdl_gview_gpoint_missing_size_f (const gdl_gview * g, size_t a, size_t l)
{
	size_t i, np, nm=0;	
	
	np = gdl_gview_ploidy (g);
	
	for (i = 0; i < np; i++)
	{
		if (gdl_gview_is_missing_f (g, a, l, i))
		{
			nm++;
		}
	}
	
	return nm;
}

int
gdl_gview_add (gdl_gview * g, const gdl_string * name, gdl_gview * s, size_t owner)
{
	if (g->type->add)
	{
		return (g->type->add)(g->data, name, s, owner);
	}
	else
	{
		GDL_ERROR_VAL ("Cannot add a gview : fonction not available",
		               GDL_EINVAL,
		               -1);
	}	
}

gdl_vector *
gdl_gview_genotype_freq (const gdl_gview * g, const gdl_locus * locus, gdl_gvalues_get * gb)
{
	return gdl_gview_genotype_freq_f (g, locus->idx, gb);
}

gdl_vector *
gdl_gview_allele_freq (const gdl_gview * g, const gdl_locus * locus, gdl_gvalues_get * gb)
{
	return gdl_gview_allele_freq_f (g, locus->idx, gb);
}

gdl_vector *
gdl_gview_genotype_freq_f (const gdl_gview * g, size_t l, gdl_gvalues_get * gb)
{
	size_t i, j, k, n, obs=0;
	const gdl_gvalues * x;
	gdl_vector * f;
	
	n = gdl_gview_accession_size (g);
	
	f = gdl_vector_calloc (gdl_locus_genotype (gdl_gview_get_locus (g, l)));
	
	for (i = 0; i < n; i++)
	{
		gdl_gview_genotype_f (g, i, l, gb);
		x = gdl_gvalues_get_gvalues (gb);
		if (x)
		{
			for (k = 0; k < x->size; k++)
			{
				size_t a = x->values[k]->idx;
				gdl_vector_set (f, a, gdl_vector_get (f, a) + x->values[k]->value);
			}
			obs++;
		}	
	}
	
	gdl_vector_scale (f, 1.0/obs);
	
	return f;
}

gdl_vector *
gdl_gview_allele_freq_f (const gdl_gview * g, size_t l, gdl_gvalues_get * gb)
{
	size_t i, j, k, n, p, obs=0;
	const gdl_gvalues * x;
	gdl_vector * f;
	
	p = gdl_gview_ploidy (g);
	n = gdl_gview_accession_size (g);
	
	f = gdl_vector_calloc (gdl_locus_allele (gdl_gview_get_locus (g, l)));
	
	for (i = 0; i < n; i++)
	{
		for (j = 0; j < p; j++)
		{
			gdl_gview_allele_f (g, i, l, j, gb);
			x = gdl_gvalues_get_gvalues (gb);
			if (x)
			{
				for (k = 0; k < x->size; k++)
				{
					size_t a = x->values[k]->idx;
					gdl_vector_set (f, a, gdl_vector_get (f, a) + x->values[k]->value);
				}
				obs++;
			}
		}	
	}
	
	gdl_vector_scale (f, 1.0/obs);
	
	return f;
}

static void
_gdl_gview_free (void * vg)
{
	if (vg)
	{
		gdl_gview * g = (gdl_gview *) vg;
		gdl_gview_free (g);
	}	
}

static void *
_gdl_gview_fread (FILE * stream)
{
	return gdl_gview_fread (stream);
}

static int
_gdl_gview_fwrite (FILE * stream, const void * vg)
{
	return gdl_gview_fwrite (stream, (gdl_gview *) vg);
}

static const gdl_data_interface _interface=
{
	&_gdl_gview_free,
	NULL,
	NULL,
    &_gdl_gview_fread,
    &_gdl_gview_fwrite
};

const gdl_data_interface * gdl_gview_interface = &_interface;
