/*  
 * 	gview/reader.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:43 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_util.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_fsheet.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_reader.h>

static gdl_string *
_gdl_gview_reader_get_key (char * name, size_t idx)
{
	if (name)
		return gdl_string_clone (name);
	else
	{
		gdl_string * key = gdl_string_alloc (10);
		sprintf (key, "%d", idx);
		return key;
	}		
}

gdl_gview_reader *
gdl_gview_reader_alloc (gdl_gview_reader_type * type)
{
	gdl_gview_reader * r;
	
	r = GDL_CALLOC (gdl_gview_reader, 1);
	
	r->type = type;
		
	return r;
}

void
gdl_gview_reader_free (gdl_gview_reader * reader)
{
	if (reader)
	{
		gdl_string_free (reader->filename);
		gdl_string_free (reader->error);
		gdl_gview_reader_type_free (reader->type);
		GDL_FREE (reader);
	}
}

int
gdl_gview_reader_open (gdl_gview_reader * reader, 
                          const gdl_string * filename)
{
	if (reader && filename)
	{
		reader->stream = gdl_fileopen (filename, "r");
		if (reader->stream)
		{
			gdl_string_free (reader->filename);
			reader->filename = gdl_string_clone (filename);
			gdl_string_free (reader->error);
			reader->error = NULL;
			return GDL_SUCCESS;
		}
		else
			return GDL_EINVAL;
	}
	return GDL_EINVAL;
}

gdl_gview *
gdl_gview_reader_parse (gdl_gview_reader * reader)
{
	if (reader==0)
	{
		return NULL;
	}
	else if ((reader->type->read)(reader) == GDL_SUCCESS)
	{
		return reader->gview;
	}
	else
	{
		return NULL;
	}
}

int
gdl_gview_reader_close (gdl_gview_reader * reader)
{
	if (reader && reader->stream)
	{
		gdl_fileclose (reader->filename, reader->stream);
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}

int
gdl_gview_reader_error (gdl_gview_reader * reader, char * format, ...)
{
	if (reader == 0)
		return GDL_EINVAL;
	else
	{
		va_list ap; 
		
		va_start (ap, format);
		
		gdl_string_free (reader->error);
		
		reader->error = gdl_string_vsprintf (format, ap);
		
		va_end (ap);
	 	
	 	return GDL_SUCCESS;
	}
		
}

gdl_locus * 
gdl_gview_reader_add_locus (gdl_gview_reader * reader, char * name, size_t idx)
{
	gdl_string * key  = _gdl_gview_reader_get_key (name, idx);
	gdl_locus * locus = gdl_gview_add_locus (reader->gview, key);
	gdl_string_free (key);
	return locus;
}

gdl_accession *
gdl_gview_reader_add_accession (gdl_gview_reader * reader, char * name, size_t idx)
{
	gdl_string * key = _gdl_gview_reader_get_key (name, idx);
	gdl_accession * accession = gdl_gview_add_accession (reader->gview, key);
	gdl_string_free (key);
	return accession;
}

int
gdl_gview_reader_update_allele (gdl_locus * locus, gdl_gview_reader_cell * cell)
{
	size_t i, na = 0, addgeno = 1;
	gdl_genotype * geno = gdl_genotype_alloc ();
	
	for (i = 0; i < cell->size; i++)
	{
		size_t n = 0;
		
		gdl_list_itr * itr 
		        = gdl_list_iterator_front (cell->units[i]);
		
		do
		{
			gdl_gview_reader_gpoint * unit 
			       = (gdl_gview_reader_gpoint *) gdl_list_iterator_value (itr);
	        
	        gdl_allele * allele = NULL;
	        
	        if (unit->alleles[0])
	        {
	        	if (unit->recessives[0])
	        	{
	        		allele = gdl_locus_add_recessive (locus);
	        	}
	        	else
	        	{
		        	allele = gdl_allele_new (unit->alleles[0]);
					gdl_locus_add_allele (locus, &allele, 1);
	        	}
	        	unit->idx = allele->idx;
	        }
	        if (!n)
			{
				if (allele)
					gdl_genotype_add (geno, allele, 0);
				else 
				{
					gdl_genotype_add_missing (geno);
					na++;
				}
			}
			n++;
		} 
		while (gdl_list_iterator_next (itr));
		
		gdl_list_iterator_free (itr);
		
		addgeno &= (n==1);
	}
	
	if ((cell->size && addgeno)
	    && na != cell->size)
	{
		gdl_locus_add_genotype (locus, &geno, 1);
	}
	else
	{
		gdl_genotype_free (geno);	
	}
	
	return GDL_SUCCESS;
}

int
gdl_gview_reader_update_genotype (gdl_locus * locus, gdl_gview_reader_cell * cell)
{
	gdl_list_itr * itr = gdl_list_iterator_front (cell->units[0]);
		
	do
	{
		size_t i;
		
		gdl_gview_reader_gpoint * unit 
		       = (gdl_gview_reader_gpoint *) gdl_list_iterator_value (itr);
		
		gdl_genotype * genotype = gdl_genotype_alloc ();
		
		for (i = 0; i < unit->n; i++)
		{
			if (unit->alleles[i] == NULL)
			{
				gdl_genotype_free (genotype);
				return (-1);	
			}
			else
			{
				gdl_allele * a = gdl_allele_new (unit->alleles[i]);
				gdl_genotype_add (genotype, a, 1);
			}
		}
		
		gdl_locus_add_genotype (locus, &genotype, 1);
		
		unit->idx = genotype->idx;
		
	} 
	while (gdl_list_iterator_next (itr));
	
	gdl_list_iterator_free (itr);
	
	return GDL_SUCCESS;
}

int
gdl_gview_reader_update_locus (gdl_locus * locus, gdl_gview_reader_cell * cell)
{
	if (cell->type == gdl_gpoint_haplo)
	{
		return gdl_gview_reader_update_allele (locus, cell);
	}
	else if (cell->type == gdl_gpoint_geno)
	{
		return gdl_gview_reader_update_genotype (locus, cell);
	}
	else
	{
		return (-1);	
	}
}

int 
gdl_gview_reader_set_gdatapoint (gdl_gview_reader * reader, 
                                    gdl_accession * va,
                                    gdl_locus     * vl,
                                    gdl_gview_reader_cell * cell)
{
	gdl_gdatapoint * g = gdl_gview_reader_cell_gdatapoint (cell);
	
	if (g)
	{
		gdl_gview_set_gdatapoint (reader->gview, va, vl, g);
	}
}

