/*  
 * 	gview/standard.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:43 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_locus.h>
#include <gdl/gdl_accession.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_mask.h>

typedef struct
{
	size_t pl;
	size_t nl;
	size_t na;
	gdl_hashtable   * locus;
	gdl_hashtable   * accession;
	gdl_gdatapoints * gdata;
	gdl_locus       ** _locus;
	gdl_accession   ** _accession;	
} gdl_gview_standard_t;

static int
_gdl_gview_standard_grow_locus (gdl_gview_standard_t * v, size_t n)
{
	if (v->_locus == NULL)
	{
		v->_locus = GDL_MALLOC (gdl_locus *, n);
	}
	else
	{
		gdl_locus ** new = GDL_MALLOC (gdl_locus *, n);
		memcpy (new, v->_locus, sizeof(gdl_locus *)*v->nl);
		GDL_FREE (v->_locus);
		v->_locus = new;
	}
	v->nl = n;
	return (0);
}

static int
_gdl_gview_standard_grow_accession (gdl_gview_standard_t * v, size_t n)
{
	if (v->_accession == NULL)
	{
		v->_accession = GDL_MALLOC (gdl_accession *, n);
	}
	else
	{
		gdl_accession ** new = GDL_MALLOC (gdl_accession *, n);
		memcpy (new, v->_accession, sizeof(gdl_accession *)*v->na);
		GDL_FREE (v->_accession);
		v->_accession = new;
	}
	v->na = n;
	return (0);
}

static int
_gdl_gview_standard_genotype (const gdl_gview_standard_t * v, size_t i, const gdl_locus * vl, gdl_gvalues_get * get)
{	
	gdl_gdatapoint * p
	   = gdl_gdatapoints_get (v->gdata, i, vl->idx);
	
	if (p == NULL)
	{
		get->na = gdl_true;
		return (0);
	}
	else if (p->type == gdl_gpoint_geno)
	{
		get->na = gdl_false;
		get->x->size = p->values[0]->size;
		gdl_gvalues_copy(get->x, p->values[0]);
		return (0);
	}
	else if (p->type == gdl_gpoint_haplo && v->pl > 1)
	{
		size_t j;
		gdl_string * name;
		gdl_genotype * g = 
			gdl_genotype_alloc ();
			
		for (j = 0; j < v->pl; j++)
		{
			if (p->values[j] == NULL || p->values[j]->size != 1)
			{
				gdl_genotype_free (g);
				get->na = gdl_true;
				return (0);
			}
			
			gdl_allele * allele = gdl_locus_get_allele (vl, p->values[j]->values[0]->idx);
			
			gdl_genotype_add (g, allele, 0);
		}
		
		name = g->name;
		g->name = NULL;
		
		gdl_genotype_free (g);
		
		g = gdl_locus_search_genotype (vl, name);
		
		if (g == NULL)
		{
			gdl_string_free (name);
			get->na = gdl_true;
			return (0);
		}
		
		get->na = gdl_false;
		get->x->size = 1;
		get->x->values[0]->value = 1.0;
		get->x->values[0]->idx   = g->idx;
			
		gdl_string_free (name);
		
		return (0);
	}
	else if (p->type == gdl_gpoint_haplo && v->pl == 1)
	{
		get->na = gdl_false;
		// To allow the copy to work...
		get->x->size = p->values[0]->size;
		gdl_gvalues_copy(get->x, p->values[0]);
		return (0);
	}
}

#include "stemp.c"

int
gdl_gview_standard_alloc (void * vg)
{
	gdl_gview_standard_t * g = (gdl_gview_standard_t *) vg;
	
	g->pl = g->na = g->nl = 0;
	g->locus      = NULL;
	g->accession  = NULL;
	g->gdata      = NULL;
	g->_locus     = NULL;
	g->_accession = NULL;

	return GDL_SUCCESS;
}

void
gdl_gview_standard_free (void * vg)
{
	if (vg)
	{
		gdl_gview_standard_t * g = (gdl_gview_standard_t *) vg;
		
		gdl_hashtable_free (g->locus);
		gdl_hashtable_free (g->accession);
		gdl_gdatapoints_free (g->gdata);
		GDL_FREE (g->_locus);
		GDL_FREE (g->_accession);
	}	
}

int
gdl_gview_standard_init (void * vg, size_t na, size_t nl, size_t p)
{
	gdl_gview_standard_t * g = (gdl_gview_standard_t *)vg;
	
	g->pl         = p;
	g->locus      = gdl_hashtable_alloc (gdl_entity_interface, (nl > 0) ? nl : 0);
	g->accession  = gdl_hashtable_alloc (gdl_entity_interface, (na > 0) ? na : 0);
	g->gdata      = gdl_gdatapoints_alloc (p, na, nl);
		
	return GDL_SUCCESS;
}

size_t
gdl_gview_standard_locus_size (const void * vg)
{
	return ((gdl_gview_standard_t *)vg)->nl;
}

size_t
gdl_gview_standard_accession_size (const void * vg)
{
	return ((gdl_gview_standard_t *)vg)->na;
}

size_t
gdl_gview_standard_ploidy (const void * vg)
{
	return ((gdl_gview_standard_t *)vg)->pl;
}

gdl_locus *
gdl_gview_standard_add_locus (void * vg, const gdl_string * name)
{
	if (vg == 0 || name == 0)
		return NULL;
	else
	{
		gdl_gview_standard_t * v = (gdl_gview_standard_t *) vg;
		
		gdl_locus * vl = 
			(gdl_locus *) gdl_hashtable_lookup (v->locus, name);
		if (vl)
		{
			return vl;
		}
		else
		{
			vl      = gdl_locus_new (name);
			vl->idx = gdl_hashtable_size (v->locus);
		    gdl_hashtable_add (v->locus, name, vl, 1);
		    _gdl_gview_standard_grow_locus (v, vl->idx+1);
		    v->_locus[vl->idx] = vl;
		    return vl;
		}
	}
}

gdl_accession *
gdl_gview_standard_add_accession (void * vg, const gdl_string * name)
{
	if (vg == 0 || name == 0)
		return NULL;
	else
	{
		gdl_gview_standard_t * v = (gdl_gview_standard_t *) vg;
		
		gdl_accession * va = 
			(gdl_accession *) gdl_hashtable_lookup (v->accession, name);
		if (va)
		{
			return va;
		}
		else
		{
			va      = gdl_accession_new (name);
			va->idx = gdl_hashtable_size (v->accession);
		    gdl_hashtable_add (v->accession, name, va, 1);
		    _gdl_gview_standard_grow_accession (v, va->idx+1);
		    v->_accession[va->idx] = va;
		    return va;
		}
	}
}

gdl_locus *
gdl_gview_standard_get_locus (const void * vg, size_t idx)
{
	return ((gdl_gview_standard_t *)vg)->_locus[idx];
}

gdl_accession *
gdl_gview_standard_get_accession (const void * vg, size_t idx)
{
	return ((gdl_gview_standard_t *)vg)->_accession[idx];
}

gdl_locus *
gdl_gview_standard_search_locus (const void * vg, const char * name)
{
	return 	gdl_hashtable_lookup (((gdl_gview_standard_t *)vg)->locus, name);
}

gdl_accession *
gdl_gview_standard_search_accession (const void * vg, const char * name)
{
	return 	gdl_hashtable_lookup (((gdl_gview_standard_t *)vg)->accession, name);
}

gdl_gvalues_get *
gdl_gview_standard_get_new (const void * gv)
{
	gdl_gvalues_get * get;
	size_t i, j, max=0;
	gdl_gview_standard_t * g = (gdl_gview_standard_t *) gv;
	
	for (i = 0; i < g->nl; i++)
	{
		gdl_locus * vl = gdl_gview_standard_get_locus(g, i);
		j = GDL_MAX (gdl_locus_allele (vl), gdl_locus_genotype (vl)); 	
		if (j > max)
		{
			max = j;
		}
	}
	return gdl_gvalues_get_alloc (max);
}

int
gdl_gview_standard_set_gdatapoint (void * vg, const gdl_accession * va, const gdl_locus * vl, gdl_gdatapoint * gd)
{
	return 
	   gdl_gdatapoints_set (((gdl_gview_standard_t *)vg)->gdata, va->idx, vl->idx, gd);
}

int
gdl_gview_standard_genotype (const void * gv, const gdl_accession * va, const gdl_locus * vl, gdl_gvalues_get * get)
{
	return
		_gdl_gview_standard_genotype ((gdl_gview_standard_t *) gv, va->idx, vl, get);
}

int
gdl_gview_standard_allele (const void * gv, const gdl_accession * va, const gdl_locus * vl, size_t p, gdl_gvalues_get * get)
{
	return
		gdl_gview_standard_allele_f ((gdl_gview_standard_t *) gv, va->idx, vl->idx, p, get);
}

int
gdl_gview_standard_gtemplate (const void * gv, const gdl_accession * va, const gdl_locus * vl, gdl_gvalues_get * get)
{
	_gdl_gview_standard_gtemplate ((gdl_gview_standard_t *) gv, va->idx, vl, get);
	return GDL_SUCCESS;
}

gdl_boolean
gdl_gview_standard_has_missing (const void * gv, const gdl_accession * va, const gdl_locus * vl)
{
	return
		gdl_gdatapoints_has_missing (((gdl_gview_standard_t *)gv)->gdata, va->idx, vl->idx);	
}

gdl_boolean
gdl_gview_standard_is_missing (const void * gv, const gdl_accession * va, const gdl_locus * vl, size_t p)
{
	return
		gdl_gdatapoints_is_missing (((gdl_gview_standard_t *)gv)->gdata, va->idx, vl->idx, p);	
}

gdl_boolean
gdl_gview_standard_is_homozygous (const void * gv, const gdl_accession * va, const gdl_locus * vl)
{
	return
		gdl_gdatapoints_is_homozygous (((gdl_gview_standard_t *)gv)->gdata, va->idx, vl->idx);
}

int
gdl_gview_standard_accession_compare (const void * vg, const gdl_accession * a1, const gdl_accession * a2, const gdl_locus_mask * m)
{
	return gdl_gview_standard_accession_compare_f (vg, a1->idx, a2->idx, m);
}

int
gdl_gview_standard_locus_compare (const void * vg, const gdl_locus * l1, const gdl_locus * l2, const gdl_accession_mask * m)
{
	return gdl_gview_standard_accession_compare_f (vg, l1->idx, l2->idx, m);
}

int
gdl_gview_standard_genotype_compare (const void * vg, const gdl_accession * a1, const gdl_locus * l1, const gdl_accession * a2, const gdl_locus * l2)
{
	return gdl_gview_standard_genotype_compare_f (vg, a1->idx, l1->idx, a2->idx, l2->idx);
}

int
gdl_gview_standard_set_gdatapoint_f (void * vg, size_t i, size_t j, gdl_gdatapoint * gd)
{
	return
		gdl_gdatapoints_set (((gdl_gview_standard_t *)vg)->gdata, i, j, gd);
}

int
gdl_gview_standard_genotype_f (const void * gv, size_t i, size_t j, gdl_gvalues_get * get)
{
	gdl_locus * vl =
		gdl_gview_standard_get_locus ((gdl_gview_standard_t *)gv, j);
	return _gdl_gview_standard_genotype ((gdl_gview_standard_t *)gv, i, vl, get);
}

	
int
gdl_gview_standard_allele_f (const void * vg, size_t i, size_t j, size_t k, gdl_gvalues_get * get)
{
	gdl_gdatapoint * p = gdl_gdatapoints_get (((gdl_gview_standard_t *)vg)->gdata, i, j);
	
	get->na = gdl_true;
	
	if (p == NULL)
	{
		return (0);
	}
	else if (p->type == gdl_gpoint_haplo && p->values[k])
    {
    	get->na = gdl_false;
        get->x->size = p->values[k]->size;
	    gdl_gvalues_copy(get->x, p->values[k]);
	    return (0);
    }
    else if (p->type == gdl_gpoint_geno)
    {
    	size_t g;
    	gdl_locus   * locus;
    	gdl_genotype * geno;
    	gdl_allele * allele;
    	gdl_gvalues * gvs;
    	gdl_gvalue  * gv, * gx;
    	
	    gvs = p->values[0];
	    
	    locus = gdl_gview_standard_get_locus (vg, j);
	    
        get->na = gdl_false;
        get->x->size = gvs->size;
        
        for (g = 0; g < gvs->size; g++)
        {
        	gv     = gvs->values[g];
        	geno   = gdl_locus_get_genotype (locus, gv->idx);
        	allele = gdl_genotype_get_allele (geno, k);
        	gx = get->x->values[g];
        	gx->idx   = allele->idx;
        	gx->value = gv->value;
        }
        
	    return (0);
	}
	
	return (1);
}

int
gdl_gview_standard_gtemplate_f (const void * gv, size_t i, const gdl_locus * locus, gdl_gvalues_get * get)
{
	_gdl_gview_standard_gtemplate ((gdl_gview_standard_t *) gv, i, locus, get);
	return GDL_SUCCESS;
}

gdl_boolean
gdl_gview_standard_is_homozygous_f (const void * gv, size_t i, size_t j)
{
	return gdl_gdatapoints_is_homozygous (((gdl_gview_standard_t *)gv)->gdata, i, j);
}

gdl_boolean
gdl_gview_standard_has_missing_f (const void * gv, size_t i, size_t j)
{
	return gdl_gdatapoints_has_missing (((gdl_gview_standard_t *)gv)->gdata, i, j);
}

gdl_boolean
gdl_gview_standard_is_missing_f (const void * gv, size_t i, size_t j, size_t p)
{
	return gdl_gdatapoints_is_missing (((gdl_gview_standard_t *)gv)->gdata, i, j, p);
}

int 
gdl_gview_standard_accession_compare_f (const void * vg, size_t a1, size_t a2, const gdl_locus_mask * m)
{
	size_t l, ll, nl;
	const gdl_gview_standard_t * g = (gdl_gview_standard_t *) vg;
	
	nl = (m == 0) ? g->nl : gdl_entity_mask_size (m);
	
	for (l = 0; l < nl; l++)
	{
		ll = (m == 0) ? l : gdl_entity_mask_idx (m, l);
		gdl_locus * loc = gdl_gview_standard_get_locus (g, ll);
		if (gdl_gview_standard_genotype_compare_f (g, a1, loc->idx, a2, loc->idx))
			return (-1);
	}
	
	return (0);	
}

int
gdl_gview_standard_locus_compare_f (const void * vg, size_t l1, size_t l2, const gdl_accession_mask * m)
{
	size_t l, ll, na;
	const gdl_gview_standard_t * g = (gdl_gview_standard_t *) vg;
	
	na = (m == 0) ? g->na : gdl_entity_mask_size (m);
	
	for (l = 0; l < na; l++)
	{
		ll = (m == 0) ? l : gdl_entity_mask_idx (m, l);
		gdl_accession * a = gdl_gview_standard_get_accession (g, ll);
		if (gdl_gview_standard_genotype_compare_f (g, a->idx, l1, a->idx, l2))
			return (-1);
	}
	
	return (0);
}

int
gdl_gview_standard_genotype_compare_f (const void * vg, size_t a1, size_t l1, size_t a2, size_t l2)
{
	gdl_gview_standard_t * g = (gdl_gview_standard_t *) vg;
	return gdl_gdatapoints_compare (g->gdata, a1, l1, a2, l2);
}

void *
gdl_gview_standard_fread (FILE * stream)
{
	if (stream == 0)
		return NULL;
	else
	{
		size_t p, n, l, status;
		gdl_gview_standard_t * v;
		
		status = fread (&p, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		v = GDL_CALLOC (gdl_gview_standard_t, 1);
		
		v->pl = p;
		
		v->locus = gdl_hashtable_alloc (gdl_entity_interface, 0);
		status   = gdl_hashtable_fread (stream, v->locus);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		v->accession = gdl_hashtable_alloc (gdl_entity_interface, 0);
		status = gdl_hashtable_fread (stream, v->accession);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		v->gdata = gdl_gdatapoints_fread (stream);
		GDL_FREAD_STATUS (v->gdata != NULL, 1);
		
		v->nl = gdl_hashtable_size (v->locus);
		
		if (v->nl)
		{
			gdl_hashtable_itr * itr 
				= gdl_hashtable_iterator (v->locus);
			
			_gdl_gview_standard_grow_locus (v, v->nl);
			
			do
			{
				gdl_locus * l = (gdl_locus *)
					gdl_hashtable_iterator_value (itr);
				v->_locus[l->idx] = l;
			}
			while (gdl_hashtable_iterator_next (itr));
			
			gdl_hashtable_iterator_free (itr);
		}
		
		v->na = gdl_hashtable_size (v->accession);
		
		if (v->na)
		{
			gdl_hashtable_itr * itr 
				= gdl_hashtable_iterator (v->accession);
			
			_gdl_gview_standard_grow_accession (v, v->na);
			
			do
			{
				gdl_accession * a = (gdl_accession *)
					gdl_hashtable_iterator_value (itr);
				v->_accession[a->idx] = a; 
			}
			while (gdl_hashtable_iterator_next (itr));
			
			gdl_hashtable_iterator_free (itr);
		}
				
		return v;
	}	
}

int
gdl_gview_standard_fwrite (FILE * stream, void * gv)
{
	if (stream == 0 || gv == 0)
		return GDL_EINVAL;
	else
	{
		size_t status;
		gdl_gview_standard_t * v = (gdl_gview_standard_t *) gv;
		
		status = fwrite (&v->pl, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = gdl_hashtable_fwrite (stream, v->locus);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_hashtable_fwrite (stream, v->accession);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		status = gdl_gdatapoints_fwrite (stream, v->gdata);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}	
}

static const gdl_gview_type _standard = 
{
	"gdl_gview_standard",
	sizeof (gdl_gview_standard_t),
	&gdl_gview_standard_alloc,
	&gdl_gview_standard_free,
	&gdl_gview_standard_init,
	&gdl_gview_standard_add_locus,
	&gdl_gview_standard_add_accession,
	&gdl_gview_standard_ploidy,
	&gdl_gview_standard_locus_size,
	&gdl_gview_standard_accession_size,
	&gdl_gview_standard_get_locus,
	&gdl_gview_standard_get_accession,
	&gdl_gview_standard_search_locus,
	&gdl_gview_standard_search_accession,
	&gdl_gview_standard_get_new,
	&gdl_gview_standard_set_gdatapoint,
	&gdl_gview_standard_genotype,
	&gdl_gview_standard_allele,
	&gdl_gview_standard_has_missing,
	&gdl_gview_standard_is_missing,
	&gdl_gview_standard_is_homozygous,
	&gdl_gview_standard_gtemplate,
	&gdl_gview_standard_accession_compare,
	&gdl_gview_standard_locus_compare,
	&gdl_gview_standard_genotype_compare,
	&gdl_gview_standard_set_gdatapoint_f,
	&gdl_gview_standard_genotype_f,
	&gdl_gview_standard_allele_f,
	&gdl_gview_standard_has_missing_f,
	&gdl_gview_standard_is_missing_f,
	&gdl_gview_standard_is_homozygous_f,
	&gdl_gview_standard_gtemplate_f,
	&gdl_gview_standard_accession_compare_f,
	&gdl_gview_standard_locus_compare_f,
	&gdl_gview_standard_genotype_compare_f,
	&gdl_gview_standard_fread,
	&gdl_gview_standard_fwrite,
	NULL
};

const gdl_gview_type * gdl_gview_standard = &_standard;
