/*  
 * 	gview/stemp.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:43 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_gpoint.h>
#include <gdl/gdl_gentity.h>

typedef struct {
	const gdl_locus * locus;
    size_t p;
    size_t * nallele;
    size_t ** alleles;
    double ** proba;
    double ptot;
    gdl_gvalues * x;
    size_t * path;
    size_t ng;
} _twk;

static void
_build_genotype (_twk * t)
{
	size_t j;
	double pr = 1;
	gdl_string * name;
	gdl_allele * a;
	gdl_genotype * tmp = gdl_genotype_alloc ();
	
	for (j = 0; j < t->p; j++)
	{
		size_t aidx    = t->alleles[j][t->path[j]];
		a = gdl_locus_get_allele (t->locus, aidx);
		gdl_genotype_add (tmp, a, 0);
		pr *= t->proba[j][t->path[j]];
	}
	name = gdl_string_clone (tmp->name);
	gdl_genotype_free (tmp);
	tmp = gdl_locus_search_genotype (t->locus, name);
	if (!tmp)
	{
		gdl_string_free (name);
		tmp = gdl_genotype_alloc ();
		for (j = t->p; j >= 1; j--)
		{
			a = gdl_locus_get_allele (t->locus, t->alleles[j-1][t->path[j-1]]);
			gdl_genotype_add (tmp, a, 0);
		}
		name = gdl_string_clone (tmp->name);
		gdl_genotype_free (tmp);
		tmp = gdl_locus_search_genotype (t->locus, name);
	}
	if (tmp)
	{
		// Check if this geno has not already been
		// stored..
		for (j = 0; j < t->x->size; j++)
		{
			if (t->x->values[j]->idx == tmp->idx)
				break;
		}
		if (j < t->x->size)
		{
			// Update it
			t->x->values[j]->value += pr;
		}
		else
		{
			// Add it
			t->x->values[t->x->size]->idx   = tmp->idx;
			t->x->values[t->x->size]->value = pr;
			t->x->size++;
		}
		t->ptot += pr;
	}
	gdl_string_free (name);
}

static void
_path_in_alleles (_twk * t, size_t j, size_t jj)
{
	size_t k;
	
	k = (j == jj) ? 1 : 0;
	
	for (;k < t->nallele [j]; k++)
	{
		t->path[j] = k;
		if (j == t->p - 1)
			_build_genotype (t);
		if (j < t->p - 1)
			_path_in_alleles (t, j+1, jj);
	}
	t->path[j] = 0;
}

static int
_extract_genotypes (const gdl_locus * vl, 
                      size_t p,
                      size_t * nallele,
                      size_t ** alleles,
                      double ** proba,
                      gdl_gvalues * x)
{
	size_t j, jj;
	_twk t;
	
	t.locus   = vl;
	t.p       = p;
	t.nallele = nallele;
	t.alleles = alleles;
	t.proba   = proba;
	t.x       = x;
	t.path    = GDL_CALLOC (size_t, p);
	t.ptot    = 0;
	t.x->size = 0;
	
	_build_genotype (&t);
	
	for (jj = t.p; jj > 0; jj--)
	{
		j = jj - 1;
		_path_in_alleles (&t, j, j);
	}
	
	for (jj = 0; jj < t.x->size; jj++)
	{
		t.x->values[jj]->value /= t.ptot;
	}
	
	GDL_FREE (t.path);

	return GDL_SUCCESS;	
}

static int
_gdl_gview_standard_gtemplate (const gdl_gview_standard_t * v, size_t i, const gdl_locus * vl, gdl_gvalues_get * get)
{	
	size_t j, k;
	gdl_gdatapoint * p = gdl_gdatapoints_get (v->gdata, i, vl->idx);
	
	if (p == NULL)
	{
		get->x->size = gdl_locus_genotype (vl);
		for (j = 0; j < get->x->size; j++)
		{
			get->x->values[j]->idx = j;
			get->x->values[j]->value = 1./(double)get->x->size;
		}
	}
	else if (p->type == gdl_gpoint_geno)
	{
		gdl_gvalues * gx = p->values[0];
		gdl_locus   * ol = v->_locus[vl->idx];
		get->x->size     = gx->size;
		if (vl == ol)
		{
			gdl_gvalues_copy(get->x, gx);
		}
		else
		{
			for (j = 0; j < gx->size; j++)
			{
				gdl_gvalue * gxv  = gx->values[j]; 
				gdl_genotype * og = gdl_locus_get_genotype (ol, gxv->idx);
				og = gdl_locus_search_genotype (vl, og->name);
				get->x->values[j]->idx   = og->idx;
				get->x->values[j]->value = gxv->value;
			}
		}
	}
	else
	{
		size_t * nallele  = GDL_CALLOC (size_t, v->pl);
		size_t ** alleles = GDL_CALLOC (size_t *, v->pl);
		double ** proba   = GDL_CALLOC (double *, v->pl);
		
		for (j = 0; j < v->pl; j++)
		{
			gdl_gvalues * x = p->values[j];
			
			if (x)
			{
				nallele[j] = x->size;
				alleles[j] = GDL_CALLOC (size_t, nallele[j]);
				proba[j]   = GDL_CALLOC (double, nallele[j]);
				
				for (k = 0; k < nallele[j]; k++)
				{
					alleles[j][k] = x->values[k]->idx;
					proba[j][k]   = x->values[k]->value;
				}
			}
			else
			{
				nallele[j] = gdl_locus_allele (vl);
				alleles[j] = GDL_CALLOC (size_t, nallele[j]);
				proba[j]   = GDL_CALLOC (double, nallele[j]);
				
				for (k = 0; k < nallele[j]; k++)
				{
					alleles[j][k] = k;
					proba[j][k]   = 1.0/(double)nallele[j];
				}	
			}			
		}
		
		_extract_genotypes (vl, v->pl, nallele, alleles, proba, get->x);
		
		for (j = 0; j < v->pl; j++)
		{
			GDL_FREE (alleles[j]);
			GDL_FREE (proba[j]);
		}
		
		GDL_FREE (alleles);
		GDL_FREE (proba);
		GDL_FREE (nallele);
		
	}
	
	get->na = gdl_false;
	
	return GDL_SUCCESS;
}
