/*  
 * 	gview/wrapper.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:43 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_allele_block.h>
#include <gdl/gdl_clustering.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_mask.h>
#include <gdl/gdl_gview_collector.h>
#include <gdl/gdl_gview_wrapper.h>

struct _gdl_gview_wrapper_type
{
	gdl_string * name;	
};

struct _gdl_gview_wrapper
{
	const gdl_gview_wrapper_type * type;
	const gdl_gview     * gview;
	const gdl_mask     * gmask;
	gdl_clustering      * gclust;
	gdl_gview_collector * gmiss;
	gdl_accession       ** accession;     
	gdl_locus           ** locus;
	gdl_locus           ** template;
	gdl_hashtable       * loctab;
	gdl_hashtable       * acctab;
};

static void
_alloc_template (gdl_gview_wrapper * gw, size_t j)
{
	size_t i, ii, k, na, * tmp;
	gdl_locus     * locus;
	gdl_gvalues_get * gbuf;
	const gdl_gvalues   * gv;
	gdl_boolean has;
	
	locus           = GDL_GVIEW_GET_LOCUS (gw->gview, gw->gmask, j);
	gw->template[j] = gdl_locus_resolve_genotype (locus);
	gbuf            = gdl_gvalues_get_alloc (gdl_locus_genotype (gw->template[j]));
	
	na = gdl_clustering_nclust (gw->gclust);
	
	for (i = 0; i < na; i++)
	{
		ii = gdl_clustering_clust_idx (gw->gclust, i);
		GDL_GVIEW_GET_GENOTYPE_TEMPLATE (gw->gview, gw->gmask, ii, gw->template[j], gbuf);
		has = GDL_GVIEW_HAS_MISSING (gw->gview, gw->gmask, ii, j);
		gv  = gdl_gvalues_get_gvalues (gbuf);
		if (gv)
		{
			for (k = 0; k < gv->size; k++)
			{
				gdl_genotype * geno = gdl_locus_get_genotype (gw->template[j], gv->values[k]->idx);
				tmp = (size_t *)geno->extra;
				if (!tmp)
				{
					tmp = GDL_CALLOC (size_t, 1);
					geno->extra = tmp;
				}
				if (!has && gv->values[k]->value == 1.0)
				{
					(*tmp)++;	
				}
			}
		}
		else
		{
			GDL_ERROR_VOID ("Unexpected null pointer", GDL_EFAULT);
		}
	}
	gdl_gvalues_get_free (gbuf);
}

static void
_alloc_locus (gdl_gview_wrapper * gw, size_t j)
{
	size_t i, k, na, ng, * tmp;
	gdl_locus    * template = gw->template[j];
	gdl_locus    * locus;
	gdl_genotype * g1, * g2;
	gdl_allele * a1, * a2;
	
	gw->locus[j] = locus = gdl_locus_new (template->name);
	gdl_hashtable_add (gw->loctab, template->name, gw->locus[j], 0);
	locus->idx   = j;
	
	ng = gdl_locus_genotype (template);
	
	for (i = 0; i < ng; i++)
	{
		g1 = gdl_locus_get_genotype (template, i);
		tmp  = (size_t *) g1->extra;
		if (tmp)
		{	
			if ((gw->type == gdl_gview_wrapper_local && *tmp)
			    || gw->type == gdl_gview_wrapper_global)
			{
				
				g2 = gdl_entity_clone (g1);
				gdl_locus_add_genotype (locus, &g2, 1);
				*tmp = g2->idx;
				na = gdl_genotype_size (g2);
				for (k = 0; k < na; k++)
				{
					a1 = gdl_genotype_get_allele (g1, k);
					if (!a1->extra)
					{
						a2 = gdl_genotype_get_allele (g2, k);
						a1->extra = &(a2->idx);
					}				
				}
			}
		}
	}
}

static void
_init_locus (gdl_gview_wrapper * gw)
{
	size_t j, nl;
	
	nl = GDL_GVIEW_LOCUS_SIZE (gw->gview, gw->gmask);
	
	gw->template = GDL_MALLOC (gdl_locus *, nl);
	gw->locus    = GDL_MALLOC (gdl_locus *, nl);
	gw->loctab   = gdl_hashtable_alloc (gdl_entity_interface, nl);
	
	for (j = 0; j < nl; j++)
	{
		_alloc_template (gw, j);
		_alloc_locus (gw, j);
	}
}

static void
_init_accession (gdl_gview_wrapper * gw)
{
	size_t j, na;
	
	na = GDL_GVIEW_ACCESSION_SIZE (gw->gview, gw->gmask);
	
	gw->accession = GDL_MALLOC (gdl_locus *, na);
	gw->acctab    = gdl_hashtable_alloc (gdl_entity_interface, na);
	
	for (j = 0; j < na; j++)
	{
		gw->accession[j] = gdl_entity_clone (GDL_GVIEW_GET_ACCESSION (gw->gview, gw->gmask, j));
		gdl_entity_set_idx (gw->accession[j], j);
		gdl_hashtable_add (gw->acctab, gw->accession[j]->name, gw->accession[j], 0);
	}
}

gdl_gview_wrapper *
gdl_gview_wrapper_alloc (const gdl_gview_wrapper_type * T, 
                         const gdl_gview * gview,
                         const gdl_mask * gmask)
{
	gdl_gview_wrapper * gw;
	
	gw = GDL_CALLOC (gdl_gview_wrapper, 1);
	
	gw->type   = T;
	gw->gview  = gview;
	gw->gmask  = gmask;
	gw->gclust = gdl_gview_accession_clustering (gview, gmask);
	gw->gmiss  = gdl_gview_collector_alloc (gdl_gview_collector_missing);
	gdl_gview_collector_perform (gw->gmiss, gview, gmask, gw->gclust);
	
	_init_locus (gw);
	
	_init_accession (gw);
	
	return gw;
}

void
gdl_gview_wrapper_free (gdl_gview_wrapper * gw)
{
	if (gw)
	{
		size_t i, nl = gdl_gview_wrapper_locus_size (gw);
		
		gdl_clustering_free (gw->gclust);
		gdl_gview_collector_free (gw->gmiss);
		for (i = 0; i < nl; i++)
		{
			gdl_locus_free (gw->locus[i]);
			gdl_locus_free (gw->template[i]);
		}
		GDL_FREE (gw->template);
		GDL_FREE (gw->locus);
		nl = gdl_gview_wrapper_accession_size (gw);
		for (i = 0; i < nl; i++)
		{
			gdl_accession_free (gw->accession[i]);
		}
		GDL_FREE (gw->accession);
		gdl_hashtable_free (gw->loctab);
		gdl_hashtable_free (gw->acctab);
		GDL_FREE (gw);
	}	
}

const
gdl_gview * gdl_gview_wrapper_gview (const gdl_gview_wrapper * gw)
{
	return gw->gview;
}

const
gdl_mask * gdl_gview_wrapper_gmask (const gdl_gview_wrapper * gw)
{
	return gw->gmask;
}

const
gdl_clustering * gdl_gview_wrapper_clustering (const gdl_gview_wrapper * gw)
{
	return gw->gclust;
}

gdl_gview_collector *
gdl_gview_wrapper_missing_sites (const gdl_gview_wrapper * gw)
{
	return gw->gmiss;
}

size_t
gdl_gview_wrapper_ploidy (const gdl_gview_wrapper * gw)
{
	return gdl_gview_ploidy (gw->gview);	
}

size_t
gdl_gview_wrapper_accession_size (const gdl_gview_wrapper * gw)
{
	return GDL_GVIEW_ACCESSION_SIZE (gw->gview, gw->gmask);
}

size_t
gdl_gview_wrapper_accession_size_c (const gdl_gview_wrapper * gw)
{
	return gdl_clustering_nclust (gw->gclust);
}

size_t
gdl_gview_wrapper_locus_size (const gdl_gview_wrapper * gw)
{
	return GDL_GVIEW_LOCUS_SIZE (gw->gview, gw->gmask);
}

size_t
gdl_gview_wrapper_locus_allele_size (const gdl_gview_wrapper * gw, size_t i)
{
	return gdl_locus_allele (gw->locus[i]);
}

size_t
gdl_gview_wrapper_accession_mult_c (const gdl_gview_wrapper * gw, size_t i)
{
	return gdl_clustering_clust_size (gw->gclust, i);	
}

size_t
gdl_gview_wrapper_accession_idx_c (const gdl_gview_wrapper * gw, size_t i)
{
	return gdl_clustering_clust_idx (gw->gclust, i);	
}

size_t
gdl_gview_wrapper_accession_cidx_c (const gdl_gview_wrapper * gw, size_t i)
{
	return gdl_clustering_cluster (gw->gclust, i);	
}

gdl_accession *
gdl_gview_wrapper_get_accession (const gdl_gview_wrapper * gw, size_t i)
{
	return gw->accession[i];
}

gdl_accession *
gdl_gview_wrapper_get_accession_c (const gdl_gview_wrapper * gw, size_t ic)
{
	return gdl_gview_wrapper_get_accession (gw, gdl_clustering_clust_idx (gw->gclust, ic)); 
}

gdl_accession *
gdl_gview_wrapper_search_accession (const gdl_gview_wrapper * gw, const gdl_string * name)
{
	return gdl_hashtable_lookup (gw->acctab, name);
}

gdl_locus *
gdl_gview_wrapper_get_locus (const gdl_gview_wrapper * gw, size_t i)
{
	return gw->locus[i];
}

gdl_locus *
gdl_gview_wrapper_search_locus (const gdl_gview_wrapper * gw, const gdl_string * name)
{
	return gdl_hashtable_lookup (gw->loctab, name);
}

int
gdl_gview_wrapper_get_genotype (const gdl_gview_wrapper * gw, size_t i, size_t j, gdl_gvalues_get * gbuf)
{
	gdl_gvalues * gv;
	gdl_locus   * temp = gw->template[j];
	
	if (!GDL_GVIEW_HAS_MISSING (gw->gview, gw->gmask, i, j))
	{
		GDL_GVIEW_GET_GENOTYPE (gw->gview, gw->gmask, i, j, gbuf);
		gv = gbuf->x;
		for (i = 0; i < gv->size; i++)
		{
			gdl_genotype * geno = gdl_locus_get_genotype (temp, gv->values[i]->idx);
			if (geno->extra)
			{
				gv->values[i]->idx = *((size_t*)geno->extra);
			}
			else
			{
				(gv->size)--;
				if (!gv->size)
				{
					gbuf->na = gdl_true;
					break;
				}
				i--;
			}
		}
	}
	else
	{
		size_t ic;
		
		ic = gdl_clustering_cluster (gw->gclust, i);
		
		// look for genotype imputation
		
		gv = (gdl_gvalues * ) gdl_gview_collector_gget_idx (gw->gmiss, ic, j);
		
		if (gv)
		{
			gbuf->na      = gdl_false;
			gbuf->x->size = gv->size;
			gdl_gvalues_copy (gbuf->x, gv);
		}
		else if (gdl_gview_ploidy (gw->gview) == 1)
		{
			gv = (gdl_gvalues * ) gdl_gview_collector_hget_idx (gw->gmiss, ic, j, 0);
			if (gv)
			{
				gbuf->na      = gdl_false;
				gbuf->x->size = gv->size;
				gdl_gvalues_copy (gbuf->x, gv);
			}
			else
			{
				gbuf->na = gdl_true;
			}
		}
	}
	
	return GDL_SUCCESS;
}

static gdl_gvalues *
_get_allele_gvalues_from_genotype (const gdl_gview_wrapper * gw, size_t l, size_t p, const gdl_gvalues * y)
{
	size_t j;
	gdl_hashtable * hash;
	gdl_hashtable_itr * itr;
	gdl_gvalues * x = NULL;
	gdl_gvalue * ax;
	
	if (y && y->size)
	{
	
		hash = gdl_hashtable_alloc (gdl_hash_default, y->size);
		
		for (j = 0; j < y->size; j++)
		{
			gdl_genotype * g = gdl_locus_get_genotype (gw->locus[l], y->values[j]->idx);
			gdl_allele   * a = gdl_genotype_get_allele (g, p);
			ax = (gdl_gvalue *) gdl_hashtable_vlookup (hash, a);
			if (!ax)
			{
				ax = gdl_gvalue_alloc ();
				ax->idx = a->idx;
				gdl_hashtable_vadd (hash, a, ax, 0);
			}
			ax->value += y->values[j]->value;
		}
		
		x = gdl_gvalues_alloc (gdl_hashtable_size (hash));
		
		itr = gdl_hashtable_iterator (hash);
		j   = 0;
		do
		{
			ax = (gdl_gvalue *) gdl_hashtable_iterator_value (itr);
			x->values[j] = ax;
			j++;
		}
		while (gdl_hashtable_iterator_next (itr));
		
		gdl_hashtable_iterator_free (itr);
		
		gdl_hashtable_free (hash);
		
	}
	
	return x;
}

int
gdl_gview_wrapper_get_allele (const gdl_gview_wrapper * gw, size_t i, size_t j, size_t k, gdl_gvalues_get * gbuf)
{
	gdl_gvalues * gv;
	gdl_locus   * temp = gw->template[j];
	
	if (!GDL_GVIEW_IS_MISSING (gw->gview, gw->gmask, i, j, k))
	{
		size_t l, * idx;
		
		GDL_GVIEW_GET_ALLELE (gw->gview, gw->gmask, i, j, k, gbuf);
		gv = gbuf->x;
		
		for (l = 0; l < gv->size; l++)
		{
			gdl_allele * allele = gdl_locus_get_allele (temp, gv->values[l]->idx);
			if (allele->extra)
			{
				idx = (size_t *)allele->extra;
				gv->values[l]->idx = *idx;
			}
			else
			{
				(gv->size)--;
				if (!gv->size)
				{
					gbuf->na = gdl_true;
					break;
				}
				l--;
			}
		}
	}
	else
	{
		// look for allele imputation
		size_t ic;
		
		ic = gdl_clustering_cluster (gw->gclust, i);
		
		gv = (gdl_gvalues * ) gdl_gview_collector_hget_idx (gw->gmiss, ic, j, k);
		
		if (gv)
		{
			gbuf->na      = gdl_false;
			gbuf->x->size = gv->size;
			gdl_gvalues_copy (gbuf->x, gv);
		}
		else
		{
			// look for genotype imputation
			gv = (gdl_gvalues * ) gdl_gview_collector_gget_idx (gw->gmiss, ic, j);
			
			if (gv)
			{
				gv = _get_allele_gvalues_from_genotype (gw, j, k, gv);
				gbuf->x->size = gv->size;
				gdl_gvalues_copy (gbuf->x, gv);
			}
			else
			{
				gbuf->na = gdl_true;
			}
		}
	}
	
	return GDL_SUCCESS;
}

gdl_boolean
gdl_gview_wrapper_is_homozygous (const gdl_gview_wrapper * gw, size_t i, size_t j)
{
	if (GDL_GVIEW_HAS_MISSING (gw->gview, gw->gmask, i, j))
	{
		// look for genotype inputation
		gdl_gvalues * gv = (gdl_gvalues * ) gdl_gview_collector_gget_idx (gw->gmiss, i, j);
		if (gv && gv->size == 1)
		{
			gdl_genotype * geno = gdl_locus_get_genotype (gw->locus[j], gv->values[0]->idx);
			
			return gdl_genotype_is_homozygous (geno);
		}
		else
		{
			size_t k, idx;
			
			for (k = 0; k < gdl_gview_ploidy (gw->gview); k++)
			{
				gdl_gvalues * gv = (gdl_gvalues * ) gdl_gview_collector_hget_idx (gw->gmiss, i, j, k);
				if (gv && gv->size == 1)
				{
					if (k && idx != gv->values[0]->idx)
					{
						return 	gdl_false;
					}
					idx = gv->values[0]->idx;
				}
				else
				{
					return gdl_false;	
				}
			}
			
			return gdl_true;	
		}
	}
	else
	{
		return GDL_GVIEW_IS_HOMOZYGOUS (gw->gview, gw->gmask, i, j);
	}
}

gdl_boolean
gdl_gview_wrapper_has_missing (const gdl_gview_wrapper * gw, size_t i, size_t j)
{
	if (GDL_GVIEW_HAS_MISSING (gw->gview, gw->gmask, i, j))
	{
		// look for genotype inputation
		gdl_gvalues * gv = (gdl_gvalues * ) gdl_gview_collector_gget_idx (gw->gmiss, i, j);
		if (gv)
		{
			return gdl_false;			
		}
		else
		{
			size_t k;
			
			for (k = 0; k < gdl_gview_ploidy (gw->gview); k++)
			{
				gdl_gvalues * gv = (gdl_gvalues * ) gdl_gview_collector_hget_idx (gw->gmiss, i, j, k);
				if (!gv)
				{
					return gdl_true;	
				}
			}
			
			return gdl_false;
		}	
	}
	else
	{
		return gdl_false;
	}
}

gdl_boolean
gdl_gview_wrapper_is_missing (const gdl_gview_wrapper * gw, size_t i, size_t j, size_t p)
{
	if (GDL_GVIEW_IS_MISSING (gw->gview, gw->gmask, i, j, p))
	{
		// look into the collector
		gdl_gvalues * gv = (gdl_gvalues * ) gdl_gview_collector_gget_idx (gw->gmiss, i, j);
		if (gv)
		{
			return gdl_false;
		}
		else
		{
			gv = (gdl_gvalues * ) gdl_gview_collector_hget_idx (gw->gmiss, i, j, p);
			if (gv)
			{
				return gdl_false;
			}
			else
			{
				return gdl_true;
			}
		}
	}
	else
	{
		return gdl_false;
	}
}

int
gdl_gview_wrapper_get_genotype_c (const gdl_gview_wrapper * gw, size_t i, size_t j, gdl_gvalues_get * gbuf)
{
	size_t ii;
	
	ii = gdl_clustering_clust_idx (gw->gclust, i);
	
	return 	gdl_gview_wrapper_get_genotype (gw, ii, j, gbuf);
}

int
gdl_gview_wrapper_get_allele_c (const gdl_gview_wrapper * gw, size_t i, size_t j, size_t k, gdl_gvalues_get * gbuf)
{
	size_t ii;
	
	ii = gdl_clustering_clust_idx (gw->gclust, i);
	
	return gdl_gview_wrapper_get_allele (gw, ii, j, k, gbuf);
}

gdl_gvalues_get *
gdl_gview_wrapper_get_new (const gdl_gview_wrapper * gw)
{
	gdl_gvalues_get * get;
	size_t i, j, amax=0, gmax=0, size, nl;
	
	nl = GDL_GVIEW_LOCUS_SIZE (gw->gview, gw->gmask);
	
	for (i = 0; i < nl; i++)
	{
		gdl_locus * vl = gw->locus[i];
		j = gdl_locus_allele (vl);		
		if (j > amax) amax = j;
		j = gdl_locus_genotype (vl);
		if (j > gmax) gmax = j;
	}
	size = GDL_MAX (gmax, amax);
	
	return gdl_gvalues_get_alloc (size);
}

size_t
gdl_gview_wrapper_missing_size (const gdl_gview_wrapper * gw)
{
	return gdl_gview_collector_size (gw->gmiss);
}

size_t
gdl_gview_wrapper_missing_accession_size (const gdl_gview_wrapper * gw)
{
	return gdl_gview_collector_accession_size (gw->gmiss);	
}

size_t
gdl_gview_wrapper_missing_glocus_size (const gdl_gview_wrapper * gw, size_t r)
{
	return gdl_gview_collector_glocus_size (gw->gmiss, r);	
}

size_t
gdl_gview_wrapper_missing_hlocus_size (const gdl_gview_wrapper * gw, size_t r, size_t h)
{
	return gdl_gview_collector_hlocus_size (gw->gmiss, r, h);
}

size_t
gdl_gview_wrapper_missing_accession_idx_c (const gdl_gview_wrapper * gw, size_t r)
{
	return gdl_gview_collector_accession_idx (gw->gmiss, r);	
}

const size_t *
gdl_gview_wrapper_missing_glocus_idx (const gdl_gview_wrapper * gw, size_t r)
{
	return gdl_gview_collector_glocus_idx (gw->gmiss, r);	
}

const size_t *
gdl_gview_wrapper_missing_hlocus_idx (const gdl_gview_wrapper * gw, size_t r, size_t h)
{
	return gdl_gview_collector_hlocus_idx (gw->gmiss, r, h);
}

static gdl_gvalues *
_init_missing_genotype (const gdl_gview_wrapper * gw, size_t ic, size_t i, size_t l)
{
//	gdl_gvalues_get * gbuf = gdl_gview_wrapper_get_new (gw);
//	gdl_gvalues * x;
//	const gdl_gvalues * y;
//	
//	GDL_GVIEW_GET_GENOTYPE_TEMPLATE (gw->gview, gw->gmask, i, gw->locus[l], gbuf);
//	
//	y = gdl_gvalues_get_gvalues (gbuf);
//	
//	x = gdl_gvalues_clone (y);
//	
//	gdl_gview_collector_gset_idx (gw->gmiss, ic, l, x);
//	
//	gdl_gvalues_get_free (gbuf);
	
	size_t a;
	gdl_gvalues * x;
	
	x = gdl_gvalues_alloc (gdl_locus_genotype (gw->locus[l]));
	for (a = 0; a < x->size; a++)
	{
		x->values[a] = gdl_gvalue_alloc ();
		x->values[a]->idx = a;
		x->values[a]->value = 1.0/x->size;
	}
	
	gdl_gview_collector_gset_idx (gw->gmiss, ic, l, x);
	
	return x;
}

static gdl_gvalues *
_init_missing_allele (const gdl_gview_wrapper * gw, size_t ic, size_t i, size_t l, size_t p)
{
//	size_t j;
//	gdl_hashtable     * hash;
//	gdl_hashtable_itr * itr;
//	gdl_gvalues_get     * gbuf = gdl_gview_wrapper_get_new (gw);
//	gdl_gvalues * x;
//	gdl_gvalue  * ax;
//	const gdl_gvalues * y;
	
//	GDL_GVIEW_GET_GENOTYPE_TEMPLATE (gw->gview, gw->gmask, i, gw->locus[l], gbuf);
//	
//	y = gdl_gvalues_get_gvalues (gbuf);
//	
//	x = _get_allele_gvalues_from_genotype (gw, l, p, y);
//	
//	gdl_gview_collector_hset_idx (gw->gmiss, ic, l, p, x);
//	
//	gdl_gvalues_get_free (gbuf);

	size_t a;
	gdl_gvalues * x;
	
	x = gdl_gvalues_alloc (gdl_locus_allele (gw->locus[l]));
	
	for (a = 0; a < x->size; a++)
	{
		x->values[a] = gdl_gvalue_alloc ();
		x->values[a]->idx = a;
		x->values[a]->value = 1.0/x->size;
	}
	
	gdl_gview_collector_hset_idx (gw->gmiss, ic, l, p, x);
	
	return x;
}

gdl_gvalues *
gdl_gview_wrapper_missing_gget (const gdl_gview_wrapper * gw, size_t r, size_t l)
{
	gdl_gvalues * x = gdl_gview_collector_gget (gw->gmiss, r, l);
	
	if (!x)
	{
		const size_t * j = gdl_gview_collector_glocus_idx (gw->gmiss, r);
		size_t ic = gdl_gview_collector_accession_idx (gw->gmiss, r);
		size_t i  = gdl_clustering_clust_idx (gw->gclust, ic);
		
		x = _init_missing_genotype (gw, ic, i, j[l]);
	}
	
	return x;
}

gdl_gvalues *
gdl_gview_wrapper_missing_hget (const gdl_gview_wrapper * gw, size_t r, size_t h, size_t l)
{
	gdl_gvalues * x = gdl_gview_collector_hget (gw->gmiss, r, h, l);
	
	if (!x)
	{
		const size_t * j = gdl_gview_collector_hlocus_idx (gw->gmiss, r, h);
		size_t ic = gdl_gview_collector_accession_idx (gw->gmiss, r);
		size_t i  = gdl_clustering_clust_idx (gw->gclust, ic);
		
		x = _init_missing_allele (gw, ic, i, j[l], h);
	}
	
	return x;
}

gdl_gvalues *
gdl_gview_wrapper_missing_gget_c (const gdl_gview_wrapper * gw, size_t ridx, size_t lidx)
{
	gdl_gvalues * x = gdl_gview_collector_gget_idx (gw->gmiss, ridx, lidx);
	
	if (!x)
	{
		size_t i = gdl_clustering_clust_idx (gw->gclust, ridx);
		
		x = _init_missing_genotype (gw, ridx, i, lidx);
	}
	
	return x;
}

gdl_gvalues *
gdl_gview_wrapper_missing_hget_c (const gdl_gview_wrapper * gw, size_t ridx, size_t lidx, size_t p)
{
	gdl_gvalues * x = gdl_gview_collector_hget_idx (gw->gmiss, ridx, lidx, p);
	
	if (!x)
	{
		size_t i = gdl_clustering_clust_idx (gw->gclust, ridx);
		
		x = _init_missing_allele (gw, ridx, i, lidx, p);
	}
	
	return x;
}

void
gdl_gview_wrapper_missing_greset (gdl_gview_wrapper * gw, size_t r, size_t l)
{
	gdl_gvalues * x = gdl_gview_collector_gget (gw->gmiss, r, l);
	if (x)
	{
		gdl_gvalues_free (x);
	}
	gdl_gview_collector_gset (gw->gmiss, r, l, NULL);
	gdl_gview_wrapper_missing_gget (gw, r, l);
}

void
gdl_gview_wrapper_missing_hreset (gdl_gview_wrapper * gw, size_t r, size_t h, size_t l)
{
	gdl_gvalues * x = gdl_gview_collector_hget (gw->gmiss, r, h, l);
	if (x)
	{
		gdl_gvalues_free (x);
	}
	gdl_gview_collector_hset (gw->gmiss, r, h, l, NULL);
	gdl_gview_wrapper_missing_hget (gw, r, h, l);
}

void
gdl_gview_wrapper_missing_greset_c (gdl_gview_wrapper * gw, size_t ridx, size_t lidx)
{
	gdl_gvalues * x = gdl_gview_collector_gget_idx (gw->gmiss, ridx, lidx);
	if (x)
	{
		gdl_gvalues_free (x);
	}
	gdl_gview_collector_gset_idx (gw->gmiss, ridx, lidx, NULL);
	gdl_gview_wrapper_missing_gget_c (gw, ridx, lidx);
}

void
gdl_gview_wrapper_missing_hreset_c (gdl_gview_wrapper * gw, size_t ridx, size_t lidx, size_t p)
{
	gdl_gvalues * x = (gdl_gvalues *) gdl_gview_collector_hget_idx (gw->gmiss, ridx, lidx, p);
	if (x)
	{
		gdl_gvalues_free (x);
	}
	gdl_gview_collector_hset_idx (gw->gmiss, ridx, lidx, p, NULL);
	gdl_gview_wrapper_missing_hget_c (gw, ridx, lidx, p);
}

gdl_mask *
gdl_gview_wrapper_gmask_c (const gdl_gview_wrapper * gw)
{
	return gdl_mask_clustering (gw->gmask, GDL_ACCESSION, gw->gclust);
}

gdl_allele_block *
gdl_gview_wrapper_allele_block (const gdl_gview_wrapper * gw, const size_t ambiguous)
{
	size_t i, ic, j, k, l, na, nl, np;
	gdl_gvalues_get * gb;
	const gdl_gvalues * gx;
	gdl_allele_block * block;
	
	np = gdl_gview_wrapper_ploidy (gw);
	na = gdl_gview_wrapper_accession_size (gw);
	nl = gdl_gview_wrapper_locus_size (gw);
	
	block = gdl_allele_block_alloc (na, nl, np);
	
	gb = gdl_gview_wrapper_get_new (gw);
	
	for (i = 0; i < na; i++)
	{
		for (j = 0; j < np; j++)
		{
			for (l = 0; l < nl; l++)
			{
				gdl_gview_wrapper_get_allele (gw, i, l, j, gb);
				gx = gdl_gvalues_get_gvalues (gb);
				if (gx)
				{
					if (gx->size==1 || (gx->size>1 && !ambiguous))
					{ 
						const gdl_gvalue * ggx = gdl_gvalues_get_max (gx);
						gdl_allele_block_set (block, i, l, j, (ggx->idx+1));
					}
					else
					{
						gdl_allele_block_set (block, i, l, j, 0);	
					}
				}
				else
				{
					gdl_allele_block_set (block, i, l, j, 0);
				}
			}	
		}	
	}
	
	gdl_gvalues_get_free (gb);
	
	return block;
}

gdl_allele_block *
gdl_gview_wrapper_allele_block_c (const gdl_gview_wrapper * gw, const size_t ambiguous)
{
	size_t i, ic, j, k, l, na, nl, np;
	gdl_gvalues_get * gb;
	const gdl_gvalues * gx;
	gdl_allele_block * block;
	
	np = gdl_gview_wrapper_ploidy (gw);
	na = gdl_gview_wrapper_accession_size_c (gw);
	nl = gdl_gview_wrapper_locus_size (gw);
	
	block = gdl_allele_block_alloc (na, nl, np);
	
	gb = gdl_gview_wrapper_get_new (gw);
	
	for (i = 0; i < na; i++)
	{
		for (j = 0; j < np; j++)
		{
			for (l = 0; l < nl; l++)
			{
				gdl_gview_wrapper_get_allele_c (gw, i, l, j, gb);
				gx = gdl_gvalues_get_gvalues (gb);
				if (gx)
				{
					if (gx->size==1 || (gx->size>1 && !ambiguous))
					{
						const gdl_gvalue * ggx = gdl_gvalues_get_max (gx);
						gdl_allele_block_set (block, i, l, j, (ggx->idx+1));
					}
					else
					{
						gdl_allele_block_set (block, i, l, j, 0);	
					}
				}
				else
				{
					gdl_allele_block_set (block, i, l, j, 0);
				}
			}
		}	
	}
	
	gdl_gvalues_get_free (gb);
	
	return block;
}

gdl_vector *
gdl_gview_wrapper_genotype_freq (const gdl_gview_wrapper * gw, size_t l, gdl_gvalues_get * g)
{
	return gdl_gview_gmask_genotype_freq (gw->gview, gw->gmask, l, g);
}

gdl_vector *
gdl_gview_wrapper_allele_freq (const gdl_gview_wrapper * gw, size_t l, gdl_gvalues_get * g)
{
	return gdl_gview_gmask_allele_freq (gw->gview, gw->gmask, l, g);
}

void
gdl_gview_wrapper_set_missing_sites (gdl_gview_wrapper * gw, const gdl_gview_collector * missing)
{
	gdl_gview_collector_free (gw->gmiss);
	gw->gmiss = gdl_gview_collector_clone (missing);
}

static const gdl_gview_wrapper_type _local =
{
	"gdl_gview_wrapper_local"
};

static const gdl_gview_wrapper_type _global =
{
	"gdl_gview_wrapper_global"
};

const gdl_gview_wrapper_type * gdl_gview_wrapper_local = &_local;
const gdl_gview_wrapper_type * gdl_gview_wrapper_global = &_global;
