/*  
 * 	hash/iterator.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_hash.h>

struct _gdl_hashtable_itr
{
    gdl_hashtable * h;
    gdl_hashentry * e;
    gdl_hashentry * parent;
    size_t index;
};

gdl_hashtable_itr *
gdl_hashtable_iterator (gdl_hashtable * h)
{
	size_t i, size;
    
    gdl_hashtable_itr *itr = GDL_MALLOC (gdl_hashtable_itr, 1);
    
    if (itr == NULL)
    {
    	GDL_ERROR_VAL ("Failed to create a gdl_hashtable_iterator",
    	               GDL_ENOMEM,
    	               0);
    	return NULL;
    }
    
    itr->h      = h;
    itr->e      = NULL;
    itr->parent = NULL;
    
    size       = h->size;
    itr->index = size;
    
    if (h->n == 0)
    	return itr;

    for (i = 0; i < size; i++)
    {
        if (h->table[i].inuse)
        {
            itr->e     = &h->table[i];
            itr->index = i;
            break;
        }
    }
    
    return itr;
}

void
gdl_hashtable_iterator_free (gdl_hashtable_itr * itr)
{
	itr->h = NULL;
	itr->e = NULL;
	itr->parent = NULL;
	GDL_FREE (itr);
}

char *
gdl_hashtable_iterator_key (gdl_hashtable_itr * i)
{
    return i->e->name;
}

void *
gdl_hashtable_iterator_value (gdl_hashtable_itr * i)
{
    return i->e->value;
}

int
gdl_hashtable_iterator_next (gdl_hashtable_itr * itr)
{
	size_t j, size;
    gdl_hashentry * table;
    gdl_hashentry * next;
    
    if (itr->e == NULL)
    	return 0;

    next = itr->e->next;
    
    if (next != NULL)
    {
        itr->parent = itr->e;
        itr->e      = next;
        return 1;
    }
    
    size = itr->h->size;
    
    itr->parent = NULL;
    
    j = ++(itr->index);
    
    if (size <= j)
    {
        itr->e = NULL;
        return 0;
    }
    
    table = itr->h->table;
    
    next = &table[j];
    
    while (!next->inuse)
    {
    	j++;
        if (j >= size)
        {
            itr->index = size;
            itr->e     = NULL;
            return 0;
        }
        next = &table[j];
    }
    
    itr->index = j;
    itr->e     = next;
    
    return 1;
}
