/*  
 * 	hash/private.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <string.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_util.h>
#include <gdl/gdl_hash.h>

#define MAX_HASH_LEN 8

typedef struct _gdl_hashentry gdl_hashentry;

struct _gdl_hashentry {
    struct _gdl_hashentry * next;
    char * name;
    void * value;
    const void * const_value;
    size_t inuse;
    size_t owner;
};

struct _gdl_hashtable {
	const gdl_data_interface   * type;
    struct _gdl_hashentry      * table;
    size_t size;
    size_t n;
};

static unsigned long
gdl_hash_key (gdl_hashtable * table, const char * name)
{
    unsigned long value = 0L;
    char ch;
    
    if (name != NULL) {
		value += 30 * (*name);
		while ((ch = *name++) != 0) {
	 	   value = value ^ ((value << 5) + (value >> 3) + (unsigned long)ch);
		}
    }
    return (value % table->size);
}

static int
gdl_hash_grow (gdl_hashtable * table, int size)
{
    unsigned long key;
    size_t oldsize, i;
    gdl_hashentry * iter, * next;
    gdl_hashentry * oldtable;

    if (table == NULL)
		return(-1);
    if (size < 8)
        return(-1);
    if (size > 8 * 2048)
		return(-1);

    oldsize  = table->size;
    oldtable = table->table;
    
    if (oldtable == NULL)
        return(-1);
  
    table->table = GDL_CALLOC (gdl_hashentry, size);
    
    if (table->table == NULL) {
		table->table = oldtable;
		return(-1);
    }
    
    table->size = size;
	
    for (i = 0; i < oldsize; i++)
    {
    	if (oldtable[i].inuse == 0)
		{
			continue;
		}
		key = gdl_hash_key (table, oldtable[i].name);
		memcpy(&(table->table[key]), &(oldtable[i]), sizeof(gdl_hashentry));
		table->table[key].next = NULL;
    }

    for (i = 0; i < oldsize; i++)
    {
		iter = oldtable[i].next;
		
		while (iter)
		{
	       
	       next = iter->next;

	  	   key = gdl_hash_key (table, iter->name);
	       
	       if (table->table[key].inuse == 0)
	       {
				memcpy(&(table->table[key]), iter, sizeof(gdl_hashentry));
				table->table[key].next = NULL;
				GDL_FREE (iter);
	       }
	       else
	       {
	       	   iter->next = table->table[key].next;
	    	   table->table[key].next = iter;
	       }

	       iter = next;
		}
    }

    GDL_FREE (oldtable);

    return(0);
}

static void
gdl_hash_internal_table_free (gdl_hashtable * table, gdl_data_free f)
{
	size_t i, n, inside;
    gdl_hashentry * iter;
    gdl_hashentry * next;

    if (table == NULL)
		return;
		
    if (table->table == NULL)
    	return;
    	
	for(i = 0; i < table->size; i++)
	{
	    iter = &(table->table[i]);
	    if (iter->inuse==0)
	    {
	    	continue;
	    }
	    inside = 0;
	    while (iter)
	    {
			next = iter->next;
			GDL_FREE (iter->name);
			if (iter->owner && f != NULL)
			{
			   (f)(iter->value);
			}
			iter->value = NULL;
			if (inside)
			{
				GDL_FREE (iter);
			}
			inside = 1;
			iter = next;		
	    }    
	}
	
	GDL_FREE (table->table);
}

static gdl_data_interface _gdl_hash_type_default =
{
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};
