/*  
 * 	hash/test.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_io.h>
#include <gdl/gdl_test.h>
#include <gdl/gdl_hash.h>

typedef struct data
{
	int i;
	double x;
} test_data;

void
test_data_free (void * data)
{
	test_data * t = (test_data * ) data;
	GDL_FREE (t);
}

void *
test_data_clone (const void * data)
{
	test_data * t = (test_data * ) data;
	test_data * c = GDL_MALLOC (test_data, 1);
	if (c == NULL)
		return (NULL);
	c->i = t->i;
	c->x = t->x;
	return c;
}

int
test_data_fwrite (FILE * stream, const void * data)
{
	size_t status;
	test_data * t = (test_data * ) data;
	
	status = fwrite (&t->i, sizeof (int), 1, stream);
	GDL_FWRITE_STATUS (status, 1);
	status = fwrite (&t->x, sizeof (double), 1, stream);
	GDL_FWRITE_STATUS (status, 1);
	
	return GDL_SUCCESS;
}

void *
test_data_fread (FILE * stream)
{
	size_t status;
	test_data * t = GDL_MALLOC (test_data, 1);
	
	status = fread (&t->i, sizeof (int), 1, stream);
	GDL_FREAD_STATUS (status, 1);
	status = fread (&t->x, sizeof (double), 1, stream);
	GDL_FREAD_STATUS (status, 1);
	
	return t;
}

static const gdl_data_interface test_data_type =
{
  &test_data_free,
  &test_data_clone,
  NULL,
  &test_data_fread,
  &test_data_fwrite
};

gdl_hashtable *
test_hashtable_alloc (void)
{
  gdl_hashtable * table;
  
  table = gdl_hashtable_alloc (&test_data_type, 10);
  
  return table;
}

void
test_add (gdl_hashtable * table)
{
	size_t i;
	
#define TEST_ADD 100

for (i = 0; i < TEST_ADD; i++)
  {
  	 char * name;
  	 test_data * t = GDL_MALLOC (test_data, 1);
  	 t->i = i;
  	 t->x = 2.0*i;
  	 name = GDL_CALLOC (char, 12);
  	 sprintf (name, "%d", i);
  	 if (gdl_hashtable_add (table, name, t, 1)!=0)
  	 	return;
  	 GDL_FREE (name);
  }
  
}

int
test_lookup (gdl_hashtable * table)
{
	size_t i;
	
	for (i = 0; i < TEST_ADD; i++)
  	{
	  	 char * name;
	  	 test_data * t;
	  	 
	  	 name = GDL_MALLOC (char, 12);
	  	 sprintf (name, "%d", i);
	  	 
	  	 t = (test_data *) gdl_hashtable_lookup (table, name);
	  	 
	  	 if (t == NULL)
	  	 	return (1);
	  	 if (t->i != i) 
	  	 	return (1);
	  	 if (t->x != 2.0*i) 
	  	 	return (1);	
	  	 GDL_FREE (name);
  	}
  	return 0;
}

int
test_update (gdl_hashtable * table)
{
	int status;
	size_t i;
	
	for (i = 0; i < TEST_ADD; i++)
  	{
	  	 char * name;
	  	 test_data * t = GDL_MALLOC (test_data, 1);
  	 	 t->i = i+1;
  	 	 t->x = 3.0*i;
	  	 name = GDL_MALLOC (char, 12);
	  	 sprintf (name, "%d", i);
	  	 
	  	 status = gdl_hashtable_update (table, name, t, 1);
	  	 
	  	 if (status != 0)
	  	 {
	  	 	return 1;
	  	 }
	  	 
	  	 t = (test_data *) gdl_hashtable_lookup (table, name);
	  	 
	  	 if (t == NULL)
	  	 	return (1);
	  	 if (t->i != i+1) 
	  	 	return (1);
	  	 if (t->x != 3.0*i) 
	  	 	return (1);
	  	 	
	  	 GDL_FREE (name);
  	}
  	
  	return 0;
}

int
test_remove (gdl_hashtable * table)
{
	int status;
	size_t i;
	
	for (i = 0; i < TEST_ADD; i++)
  	{
	  	 char * name;
	  	 test_data * t;
	  	 
	  	 name = GDL_MALLOC (char, 12);
	  	 sprintf (name, "%d", i);
	  	 
	  	 status = gdl_hashtable_remove (table, name);
	  	 
	  	 if (status != 0)
	  	 {
	  	 	return 1;
	  	 }
	  	 
	  	 t = (test_data *) gdl_hashtable_lookup (table, name);
	  	 
	  	 if (t != NULL)
	  	 	return (1);
	  	 	
	  	 GDL_FREE (name);
  	}
  	  	
  	return (gdl_hashtable_size (table) != 0);
}

int
test_iterator (gdl_hashtable * table)
{
	size_t i = 0;
	
	gdl_hashtable_itr * itr = gdl_hashtable_iterator (table);
	
	do
	{
		size_t j;
		char * name = gdl_hashtable_iterator_key (itr);
		test_data * data = (test_data *) gdl_hashtable_iterator_value (itr);
		
		if (name == NULL)
			return 1;
		if (data == NULL)
			return 1;
		
		j = (size_t) atoi (name);
		
		if (data->i != j) 
			return 1;
		if (data->x != 2.0*j) 
			return 1;
		
		i++;		
	} while (gdl_hashtable_iterator_next (itr));
	
	return (i != gdl_hashtable_size (table));
}

int
test_fread (gdl_hashtable * table)
{
	int status;
	FILE * stream;
	
	stream = gdl_fileopen ("test.dat", "r");
  
    status = gdl_hashtable_fread (stream, table);
            
	gdl_fileclose ("test.dat", stream);
	
	if (status != GDL_SUCCESS)
		return 1;
	
	return test_lookup (table);
}

void
test_hashtable_init (void)
{
  gdl_hashtable * table = test_hashtable_alloc ();
  	
  gdl_test (table == 0,  "gdl_hashtable_alloc test");
  
  gdl_hashtable_free (table);
}

void
test_hashtable_add (void)
{
  gdl_hashtable * table = test_hashtable_alloc ();
  
  test_add (table);
  
  gdl_test (gdl_hashtable_size (table) != TEST_ADD,  "gdl_hashtable_add test");
  
  gdl_hashtable_free (table);
}

void
test_hashtable_lookup (void)
{
  gdl_hashtable * table = test_hashtable_alloc ();
  
  test_add (table);
  
  gdl_test (test_lookup (table) != 0,  "gdl_hashtable_lookup test");
  
  gdl_hashtable_free (table);
}

void
test_hashtable_update (void)
{
  gdl_hashtable * table = test_hashtable_alloc ();
  
  test_add (table);
  
  gdl_test (test_update (table) != 0,  "gdl_hashtable_update test");
  
  gdl_hashtable_free (table);
}

void
test_hashtable_remove (void)
{
  gdl_hashtable * table = test_hashtable_alloc ();
  
  test_add (table);
  
  gdl_test (test_remove (table) != 0,  "gdl_hashtable_remove test");
  
  gdl_hashtable_free (table);
}

void
test_hashtable_iterator (void)
{
  gdl_hashtable * table = test_hashtable_alloc ();
  
  test_add (table);
  
  gdl_test (test_iterator (table) != 0,  "gdl_hashtable_iterator test");
  
  gdl_hashtable_free (table);
	
}

void
test_hashtable_fwrite (void)
{
  FILE * stream;
  gdl_hashtable * table = test_hashtable_alloc ();
  
  test_add (table);
  
  stream = gdl_fileopen ("test.dat", "w");
  
  gdl_test (gdl_hashtable_fwrite (stream, table) != GDL_SUCCESS,
            "gdl_hashtable_fwrite test");
            
  gdl_fileclose ("test.dat", stream);
  
  gdl_hashtable_free (table);
}

void
test_hashtable_fread (void)
{
  gdl_hashtable * table = test_hashtable_alloc ();
  
  gdl_test (test_fread (table) != 0,  "gdl_hashtable_fread test");
  
  gdl_hashtable_free (table);
}

int
main (void)
{
//	test_hashtable_init ();
//	test_hashtable_add ();
//	test_hashtable_lookup ();
//	test_hashtable_update ();
//	test_hashtable_remove ();
//	test_hashtable_iterator ();
    do
    {
  	   //test_hashtable_add ();
  	   //test_hashtable_lookup ();
	   test_hashtable_fwrite ();
	   test_hashtable_fread ();
	   printf ("OK\n");
    } while (0==0);
	exit (gdl_test_summary());
}
