/*  
 *  hstruct/block.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_hstruct.h>
#include <gdl/gdl_hstruct_block.h>

gdl_hstruct_block *
gdl_hstruct_block_alloc (size_t kmax, size_t k, size_t from, size_t to, const gdl_matrix * left, const gdl_matrix * right)
{
	size_t i, j;
	gdl_hstruct_block * b;
	
	b = GDL_CALLOC (gdl_hstruct_block, 1);
	
	b->k      = k;
	b->from   = from;
	b->to     = to;
	b->best   = gdl_vector_uint_alloc (kmax);
	for (i = 0; i < k; i++)
	{
		gdl_vector_uint_set (b->best, i, i);
	}	
	if (left)
	{
		b->left = gdl_matrix_alloc (left->size1, left->size2);
		gdl_matrix_memcpy (b->left, left);		
	}	
	if (right)
	{
		b->right = gdl_matrix_alloc (right->size1, right->size2);
		gdl_matrix_memcpy (b->right, right);
	}
	return b;
}

void
gdl_hstruct_block_free (gdl_hstruct_block * b)
{
	if (b)
	{
		gdl_vector_uint_free (b->best);
		gdl_matrix_free (b->left);
		gdl_matrix_free (b->right);
	}	
}

gdl_hstruct_block *
gdl_hstruct_block_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		gdl_boolean has;
		gdl_hstruct_block * b;
		
		b = GDL_CALLOC (gdl_hstruct_block, 1);
	
		status = fread (&(b->k), sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(b->from), sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(b->to), sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		b->best = gdl_vector_uint_fread (stream);
		GDL_FREAD_STATUS (b->best!=0, 1);
		status = fread (&has, sizeof(gdl_boolean), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		if (has)
		{
			b->left = gdl_matrix_fread (stream);
			GDL_FREAD_STATUS (b->left!=0, 1);
		}
		status = fread (&has, sizeof(gdl_boolean), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		if (has)
		{
			b->right = gdl_matrix_fread (stream);
			GDL_FREAD_STATUS (b->right!=0, 1);
		}
		
		return b;
	}
	
	return NULL;	
}

int
gdl_hstruct_block_fwrite (FILE * stream, const gdl_hstruct_block * b)
{
	if (stream && b)
	{
		int status;
		gdl_boolean has;
		
		status = fwrite (&(b->k), sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(b->from), sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(b->to), sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = gdl_vector_uint_fwrite (stream, b->best);
		GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		has = (b->left) ? gdl_true : gdl_false;
		status = fwrite (&has, sizeof(gdl_boolean), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		if (has)
		{
			status = gdl_matrix_fwrite (stream, b->left);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		has = (b->right) ? gdl_true : gdl_false;
		status = fwrite (&has, sizeof(gdl_boolean), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		if (has)
		{
			status = gdl_matrix_fwrite (stream, b->right);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;	
}

static gdl_hstruct_block *
_gdl_hstruct_block_partition_init (const gdl_hstruct_model_partition * p, size_t kmax)
{
	size_t i, j, nl;
	gdl_matrix * left, * right;
	const gdl_hstruct_config * c1, * c2;
	gdl_hstruct_block * block;
	gdl_hstruct_hmm * hmm;
	
	nl   = gdl_hstruct_model_partition_size (p);
	hmm  = gdl_hstruct_model_partition_get_hmm (p);
	
	right = left  = NULL;
	
	for (i = 0; i < nl; i++)
	{
		c1 = gdl_hstruct_model_partition_get_config (p, i);
		if (c1->k == kmax)
		{
			if (i)
			{
				left = gdl_hstruct_hmm_get_transition_matrix (hmm, i-1);
			}
			for (j = i+1; j < nl; j++)
			{
				c2 = gdl_hstruct_model_partition_get_config (p, j);
				if (c1->k != c2->k) break;
			}
			if (j < nl)
			{
				right = gdl_hstruct_hmm_get_transition_matrix (hmm, j);
			}
			
			block = gdl_hstruct_block_alloc (kmax, c1->k, i, j-1, left, right);
			
			gdl_matrix_free (left);
			gdl_matrix_free (right);
			
			return block;
		}
	}
	
	return NULL;
}

static void 
_gdl_hstruct_block_ligate (size_t kmax, gdl_hstruct_block * b1, gdl_hstruct_block * b2, const gdl_matrix * tr, gdl_boolean forward)
{
	size_t k, j;
	
	if (b2->k < b1->k)
	{
		for (k = b2->k; k < b1->k; k++)
		{
			gdl_vector_const_view v = (forward) ? gdl_matrix_const_row (tr, k) : gdl_matrix_const_column (tr, k);
			j = gdl_vector_max_index (&(v.vector));
			gdl_vector_uint_set (b2->best, k, j);
		}
		for (k = b1->k; k < kmax; k++)
		{
			gdl_vector_const_view v = (forward) ? gdl_matrix_const_row (tr, gdl_vector_uint_get (b1->best, k)) : gdl_matrix_const_column (tr, gdl_vector_uint_get (b1->best, k));
			j = gdl_vector_max_index (&(v.vector));
			gdl_vector_uint_set (b2->best, k, j);	
		}
	}
	else if (b2->k > b1->k)
	{
		for (k = b2->k; k < kmax; k++)
		{
			gdl_vector_const_view v = (forward) ? gdl_matrix_const_row (tr, gdl_vector_uint_get (b1->best, k)) : gdl_matrix_const_column (tr, gdl_vector_uint_get (b1->best, k));
			j = gdl_vector_max_index (&(v.vector));
			gdl_vector_uint_set (b2->best, k, j);
		}
	}
}

static void
_gdl_hstruct_block_partition_backward (const gdl_hstruct_model_partition * p, size_t kmax, gdl_hstruct_block * b, gdl_list * buf)
{
	int i, nl;
	gdl_matrix * left, * right;
	const gdl_hstruct_config * c1, * c2;
	gdl_hstruct_block * b1, * b2;
	gdl_hstruct_hmm * hmm;
	
	nl  = gdl_hstruct_model_partition_size (p);
	hmm = gdl_hstruct_model_partition_get_hmm (p);
	
	b1 = b;
	
	for (i = b1->from - 1; i > 0; i--)
	{
		c1 = gdl_hstruct_model_partition_get_config (p, i);
		c2 = gdl_hstruct_model_partition_get_config (p, i-1);
		if (c1->k != c2->k)
		{
			left  = gdl_hstruct_hmm_get_transition_matrix (hmm, i-1);
			right = gdl_hstruct_hmm_get_transition_matrix (hmm, b1->from-1);
			b2    = gdl_hstruct_block_alloc (kmax, c1->k, i, b1->from-1, left, right);
			_gdl_hstruct_block_ligate (kmax, b1, b2, right, gdl_false);
			// clean
			gdl_matrix_free (left);
			gdl_matrix_free (right);
			// store
			gdl_list_push_front (buf, b2, 0);
			b1 = b2;
		}
	}
	if (b1->from > 0)
	{ 
		// and the last one...
		right = gdl_hstruct_hmm_get_transition_matrix (hmm, b1->from-1);
		b2    = gdl_hstruct_block_alloc (kmax, c2->k, i, b1->from-1, NULL, right);
		_gdl_hstruct_block_ligate (kmax, b1, b2, right, gdl_false);
		// clean
		gdl_matrix_free (right);
		// store
		gdl_list_push_front (buf, b2, 0);
	}
}

static void
_gdl_hstruct_block_partition_forward (const gdl_hstruct_model_partition * p, size_t kmax, gdl_hstruct_block * b, gdl_list * buf)
{
	size_t i, nl;
	gdl_matrix * left, * right;
	const gdl_hstruct_config * c1, * c2;
	gdl_hstruct_block * b1, * b2;
	gdl_hstruct_hmm * hmm;
	
	nl  = gdl_hstruct_model_partition_size (p);
	hmm = gdl_hstruct_model_partition_get_hmm (p);
	
	b1 = b;
	
	for (i = b1->to + 1; i < nl-1; i++)
	{
		c1 = gdl_hstruct_model_partition_get_config (p, i);
		c2 = gdl_hstruct_model_partition_get_config (p, i+1);
		if (c1->k != c2->k)
		{
			left  = gdl_hstruct_hmm_get_transition_matrix (hmm, b1->to);
			right = gdl_hstruct_hmm_get_transition_matrix (hmm, i);
			b2    = gdl_hstruct_block_alloc (kmax, c1->k, b1->to+1, i, left, right);
			_gdl_hstruct_block_ligate (kmax, b1, b2, left, gdl_true);
			// clean
			gdl_matrix_free (left);
			gdl_matrix_free (right);
			// store
			gdl_list_push_back (buf, b2, 0);
			b1 = b2;
		}
	}
	if (b1->to < nl - 1)
	{
		// and the last one...
		left  = gdl_hstruct_hmm_get_transition_matrix (hmm, b1->to);
		b2    = gdl_hstruct_block_alloc (kmax, c2->k, b1->to+1, i, left, NULL);
		_gdl_hstruct_block_ligate (kmax, b1, b2, left, gdl_true);
		// clean
		gdl_matrix_free (left);
		// store
		gdl_list_push_back (buf, b2, 0);
	}
}

static gdl_hstruct_block **
_gdl_hstruct_block_partition_create (const gdl_hstruct_model_partition * p, size_t * size, size_t * kmax)
{
	size_t i;
	gdl_hstruct_block * block, ** blocks;
	gdl_list * buf;
	gdl_list_itr * itr;
	
	buf   = gdl_list_alloc (gdl_list_default);
	
	*kmax = gdl_hstruct_model_partition_get_kmax (p);
	
	block = _gdl_hstruct_block_partition_init (p, *kmax);
	
	gdl_list_push_back (buf, block, 0);
	
	_gdl_hstruct_block_partition_backward (p, *kmax, block, buf);
	
	_gdl_hstruct_block_partition_forward (p, *kmax, block, buf);
	
	*size  = gdl_list_size (buf);
	
	blocks = GDL_MALLOC (gdl_hstruct_block *, *size);
	
	itr = gdl_list_iterator_front (buf);
	i   = 0;
	do
	{
		blocks[i++] = (gdl_hstruct_block *) gdl_list_iterator_value (itr);
	}
	while (gdl_list_iterator_next (itr));
	gdl_list_iterator_free (itr);
	gdl_list_free (buf);
	
	return blocks;
}

gdl_hstruct_block_partition *
gdl_hstruct_block_partition_alloc (const gdl_hstruct_model_partition * p)
{
	gdl_hstruct_block_partition * bp;
	
	bp = GDL_MALLOC (gdl_hstruct_block_partition, 1);
	
	bp->blocks = _gdl_hstruct_block_partition_create (p, &(bp->size), &(bp->kmax));
	
	return bp;
}

void
gdl_hstruct_block_partition_free (gdl_hstruct_block_partition * bp)
{
	if (bp)
	{
		size_t i;
		for (i = 0; i < bp->size; i++)
		{
			gdl_hstruct_block_free (bp->blocks[i]);	
		}
		GDL_FREE (bp->blocks);
		GDL_FREE (bp);
	}
}

size_t
gdl_hstruct_block_partition_size (const gdl_hstruct_block_partition * bp)
{
	return bp->size;
}

size_t
gdl_hstruct_block_partition_kmax (const gdl_hstruct_block_partition * bp)
{
	return bp->kmax;
}

gdl_hstruct_block *
gdl_hstruct_block_partition_get (const gdl_hstruct_block_partition * bp, size_t i)
{
	return bp->blocks[i];
}

gdl_hstruct_block_partition *
gdl_hstruct_block_partition_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i;
		gdl_hstruct_block_partition * bp;
	
		bp = GDL_MALLOC (gdl_hstruct_block_partition, 1);
		
		status = fread (&(bp->size), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(bp->kmax), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		bp->blocks = GDL_MALLOC (gdl_hstruct_block *, bp->size);
		
		for (i = 0; i < bp->size; i++)
		{
			bp->blocks[i] = gdl_hstruct_block_fread (stream);
			GDL_FREAD_STATUS (bp->blocks[i]!=0, 1);
		}
		
		return bp;
	}
	
	return NULL;
}

int
gdl_hstruct_block_partition_fwrite (FILE * stream, const gdl_hstruct_block_partition * bp)
{
	if (stream && bp)
	{
		int status;
		size_t i;
		
		status = fwrite (&(bp->size), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(bp->kmax), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		for (i = 0; i < bp->size; i++)
		{
			status = gdl_hstruct_block_fwrite (stream, bp->blocks[i]);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);
		}
		
		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;	
}

