/*  
 * 	hstruct/config.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_hash.h>
#include <gdl/gdl_list.h>
#include <gdl/gdl_vector_uint.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_hview.h>
#include <gdl/gdl_gmap.h>
#include <gdl/gdl_hstruct_point.h>
#include <gdl/gdl_hstruct_config.h>

gdl_hstruct_config *
gdl_hstruct_config_alloc (size_t k)
{
	gdl_hstruct_config * c;
	
	c = GDL_CALLOC (gdl_hstruct_config, 1);
	
	c->k = k;
	
	return c;
}

void
gdl_hstruct_config_free (gdl_hstruct_config * c)
{
	if (c)
	{
		GDL_FREE (c);
	}	
}

gdl_hstruct_config *
gdl_hstruct_config_clone (const gdl_hstruct_config * c)
{
	gdl_hstruct_config * b;
	
	b = gdl_hstruct_config_alloc (c->k);
	
	b->k = c->k;
	
	b->hot  = c->hot;
	b->mu   = c->mu;
	b->f    = c->f;
	b->rho  = c->rho;	
	
	return b;
}

gdl_hstruct_config *
gdl_hstruct_config_get (const gdl_locus * locus)
{
	return (gdl_hstruct_config *)locus->extra;	
}

void
gdl_hstruct_config_attach (gdl_hstruct_config * pconf, gdl_locus * locus)
{
	if (pconf->rho && *(pconf->rho)) gdl_hstruct_point_attach (*(pconf->rho));
	if (pconf->mu && *(pconf->mu))  gdl_hstruct_point_attach (*(pconf->mu));
	if (pconf->hot && *(pconf->hot)) gdl_hstruct_point_attach (*(pconf->hot));
	if (pconf->f && *(pconf->f))   gdl_hstruct_fpoint_attach (*(pconf->f));
	locus->extra = pconf;
}

gdl_hstruct_config *
gdl_hstruct_config_detach (gdl_locus * locus)
{
	if (locus->extra)
	{
		gdl_hstruct_config * pconf = (gdl_hstruct_config *) locus->extra;
		if (pconf->rho && *(pconf->rho)) gdl_hstruct_point_detach (*(pconf->rho));
		if (pconf->mu &&*(pconf->mu))  gdl_hstruct_point_detach (*(pconf->mu));
		if (pconf->hot && *(pconf->hot)) gdl_hstruct_point_detach (*(pconf->hot));
		if (pconf->f && *(pconf->f))   gdl_hstruct_fpoint_detach (*(pconf->f));
		locus->extra = NULL;
		return pconf;
	}
	return NULL;
}

void
gdl_hstruct_config_set_point (gdl_hstruct_config * c, const gdl_hstruct_point_type * T, void ** vp)
{
	if (T == gdl_hstruct_point_rho)
	{
		c->rho = vp;
	}
	else if (T == gdl_hstruct_point_hot)
	{
		c->hot = vp;
	}
	else if (T == gdl_hstruct_point_mu)
	{
		c->mu = vp;
	}
	else if (T == gdl_hstruct_point_f)
	{
		c->f = vp;
		if (vp && *vp)
		{
			c->k = gdl_hstruct_fpoint_size (*vp);
		}
		else
		{
			c->k = 1;
		}
	}	
}

void *
gdl_hstruct_config_get_point (const gdl_hstruct_config * c, const gdl_hstruct_point_type * T)
{
	if (c->rho && T == gdl_hstruct_point_rho)
	{
		return *(c->rho);
	}
	else if (c->hot && T == gdl_hstruct_point_hot)
	{
		return *(c->hot);
	}
	else if (c->mu && T == gdl_hstruct_point_mu)
	{
		return *(c->mu);
	}
	else if (c->f && T == gdl_hstruct_point_f)
	{
		return *(c->f);
	}
	else
	{
		return NULL;	
	}
}

void **
gdl_hstruct_config_get_point_ptr (const gdl_hstruct_config * c, const gdl_hstruct_point_type * T)
{
	if (T == gdl_hstruct_point_rho)
	{
		return c->rho;
	}
	else if (T == gdl_hstruct_point_hot)
	{
		return c->hot;
	}
	else if (T == gdl_hstruct_point_mu)
	{
		return c->mu;
	}
	else if (T == gdl_hstruct_point_f)
	{
		return c->f;
	}
	else
	{
		return NULL;	
	}
}

size_t
gdl_hstruct_config_point_inuse (const gdl_hstruct_config * c, const gdl_hstruct_point_type * T)
{
	if (c->rho && (*(c->rho) && T == gdl_hstruct_point_rho))
	{
		return gdl_hstruct_point_inuse (*(c->rho));
	}
	else if (c->hot && (*(c->hot) && T == gdl_hstruct_point_hot))
	{
		return gdl_hstruct_point_inuse (*(c->hot));
	}
	else if (c->mu && (*(c->mu) && T == gdl_hstruct_point_mu))
	{
		return gdl_hstruct_point_inuse (*(c->mu));
	}
	else if (c->f && (*(c->f) && T == gdl_hstruct_point_f))
	{
		return gdl_hstruct_fpoint_inuse (*(c->f));
	}
	
	return 0;
}

void
gdl_hstruct_config_set_point_eligible (gdl_hstruct_config * c, const gdl_hstruct_point_type * T, gdl_boolean e)
{
	if (c->rho && (*(c->rho) && T == gdl_hstruct_point_rho))
	{
		gdl_hstruct_point_set_eligible (*(c->rho), e);
	}
	else if (c->hot && (*(c->hot) && T == gdl_hstruct_point_hot))
	{
		gdl_hstruct_point_set_eligible (*(c->hot), e);
	}
	else if (c->mu && (*(c->mu) && T == gdl_hstruct_point_mu))
	{
		gdl_hstruct_point_set_eligible (*(c->mu), e);
	}
	else if (c->f && (*(c->f) && T == gdl_hstruct_point_f))
	{
		gdl_hstruct_fpoint_set_eligible (*(c->f), e);
	}
}

gdl_boolean
gdl_hstruct_config_compare (const gdl_hstruct_config * c1, const gdl_hstruct_config * c2)
{
	if (c1 == c2 && c1 != 0)
	{
		return gdl_true;
	}
	else if (c1 != 0 && c2 != 0)
	{
		if (c1->k != c2->k) return gdl_false;
		if (c1->rho != c2->rho) return gdl_false;
		if (c1->mu != c2->mu) return gdl_false;
		if (c1->hot != c2->hot) return gdl_false;
		if (c1->f != c2->f) return gdl_false;
		return gdl_true;
	}
	return gdl_false;
}

struct _gdl_hstruct_config_collector_entry
{
	size_t size;
	gdl_hstruct_config * c;
};

typedef struct _gdl_hstruct_config_collector_entry gdl_hstruct_config_collector_entry;

static gdl_hstruct_config_collector_entry *
gdl_hstruct_config_collector_entry_alloc (gdl_hstruct_config * c)
{
	gdl_hstruct_config_collector_entry * e;
	
	e = GDL_MALLOC (gdl_hstruct_config_collector_entry, 1);
	
	e->c    = c;
	e->size = 0;
	
	return e;
}

struct _gdl_hstruct_config_collector
{
	gdl_hashtable * table;
};

static gdl_hstruct_config_collector_entry *
_get_collector_entry (const gdl_hstruct_config_collector * c, const gdl_hstruct_config * cf)
{
	return (gdl_hstruct_config_collector_entry *) gdl_hashtable_vlookup (c->table, cf);
}

static void
_rm_collector_entry (const gdl_hstruct_config_collector * c, const gdl_hstruct_config * cf)
{
	gdl_hstruct_config_collector_entry * e = _get_collector_entry (c, cf);
	gdl_hashtable_vremove (c->table, cf);
	GDL_FREE (e);
}

static gdl_hstruct_config_collector_entry *
_add_collector_entry (const gdl_hstruct_config_collector * c, gdl_hstruct_config * cf)
{
	gdl_hstruct_config_collector_entry * e;
	
	e = gdl_hstruct_config_collector_entry_alloc (cf);
	
	gdl_hashtable_vadd (c->table, cf, e, 0);
	
	return e;
}


gdl_hstruct_config_collector *
gdl_hstruct_config_collector_alloc (size_t size)
{
	gdl_hstruct_config_collector * c;
	
	c = GDL_MALLOC (gdl_hstruct_config_collector, 1);
	
	c->table = gdl_hashtable_alloc (gdl_hash_default, size);
	
	return c;
}

void
gdl_hstruct_config_collector_free (gdl_hstruct_config_collector * c)
{
	if (c)
	{
		if (gdl_hashtable_size (c->table))
		{
			gdl_hashtable_itr * itr;
			itr = gdl_hashtable_iterator (c->table);
			do
			{
				void * e = gdl_hashtable_iterator_value (itr);
				GDL_FREE (e);
			}
			while (gdl_hashtable_iterator_next (itr));
			gdl_hashtable_iterator_free (itr);
		}
		gdl_hashtable_free (c->table);
		GDL_FREE (c);
	}	
}

int
gdl_hstruct_config_collector_attach (gdl_hstruct_config_collector * c, gdl_hstruct_config * cf)
{
	gdl_hstruct_config_collector_entry * e;
	
	e = _get_collector_entry (c, cf);
	
	if (e == 0)
	{
		//printf ("NEW COLLECTOR ENTRY %p %p\n", c, cf);
		e = _add_collector_entry (c, cf);
	}
	
	//printf ("ATTACH %p %p %d\n", c, cf, e->size);
	
	(e->size)++;
	
	return e->size;
}

int
gdl_hstruct_config_collector_detach (gdl_hstruct_config_collector * c, gdl_hstruct_config * cf)
{
	gdl_hstruct_config_collector_entry * e;
	
	e = _get_collector_entry (c, cf);
	
	if (e)
	{
		(e->size)--;
   	    //printf ("DETACH %p %p %d\n", c, cf, e->size);
		if (!e->size)
		{
			 //printf ("REMOVE %p %p\n", c, cf);
			_rm_collector_entry (c, cf);
		}
		else
		{
			return e->size;
		}
	}
	else
	{
		return 0;	
	}
}

size_t
gdl_hstruct_config_collector_size (const gdl_hstruct_config_collector * c, const gdl_hstruct_config * cf)
{
	gdl_hstruct_config_collector_entry * e = _get_collector_entry (c, cf);
	
	if (e == 0)
	{
		return 0;
	}
	else
	{
		return e->size;
	}
}

gdl_hstruct_config_itr * 
gdl_hstruct_config_collector_iterator (const gdl_hstruct_config_collector * c)
{
	if (gdl_hashtable_size (c->table))
	{
		return gdl_hashtable_iterator (c->table);
	}
	else
	{
		return NULL;	
	}
}

gdl_hstruct_config *
gdl_hstruct_config_collector_iterator_config (gdl_hstruct_config_itr * itr)
{
	gdl_hstruct_config_collector_entry * e;
	
	e = (gdl_hstruct_config_collector_entry *) gdl_hashtable_iterator_value (itr);
	
	return e->c;
}

gdl_boolean
gdl_hstruct_config_collector_iterator_next (gdl_hstruct_config_itr * itr)
{
	return (gdl_hashtable_iterator_next (itr)) ? gdl_true : gdl_false;
}

void
gdl_hstruct_config_collector_iterator_free (gdl_hstruct_config_itr * itr)
{
	gdl_hashtable_iterator_free (itr);	
}

struct _gdl_hstruct_config_buffer_entry
{
	gdl_hstruct_config * old;
	gdl_hstruct_config * new;
	gdl_boolean flush;
};

typedef struct _gdl_hstruct_config_buffer_entry gdl_hstruct_config_buffer_entry;

static gdl_hstruct_config_buffer_entry *
gdl_hstruct_config_buffer_entry_alloc (gdl_hstruct_config * old)
{
	gdl_hstruct_config_buffer_entry * e;
	
	e = GDL_MALLOC (gdl_hstruct_config_buffer_entry, 1);
	
	e->old     = old;
	e->new     = gdl_hstruct_config_clone (e->old);
	e->flush   = gdl_false;
	
	return e;
}

struct _gdl_hstruct_config_buffer
{
	gdl_hashtable * table;
	gdl_hashtable * itable;
};

static gdl_hstruct_config_buffer_entry *
_add_tester_entry (gdl_hstruct_config_buffer * t, gdl_hstruct_config * old)
{
	gdl_hstruct_config_buffer_entry * e;
	
	e = gdl_hstruct_config_buffer_entry_alloc (old);
	
	gdl_hashtable_vadd (t->table, e->old, e, 0);
	gdl_hashtable_vadd (t->itable, e->new, e, 0);
	
	return e;
}

static gdl_hstruct_config_buffer_entry *
_get_tester_entry (gdl_hstruct_config_buffer * t, const gdl_hstruct_config * old)
{
	gdl_hstruct_config_buffer_entry * e;
	
	e = (gdl_hstruct_config_buffer_entry *) gdl_hashtable_vlookup (t->table, old);
	
	return e;	
}

static gdl_hstruct_config_buffer_entry *
_get_tester_ientry (gdl_hstruct_config_buffer * t, const gdl_hstruct_config * new)
{
	gdl_hstruct_config_buffer_entry * e;
	
	e = (gdl_hstruct_config_buffer_entry *) gdl_hashtable_vlookup (t->itable, new);
	
	return e;	
}

static void
_init_tester_table (gdl_hstruct_config_buffer * t, const gdl_hstruct_config_collector * c)
{
	gdl_hstruct_config * cf;
	gdl_hstruct_config_itr * itr;
	
	itr = gdl_hstruct_config_collector_iterator (c);
	
	if (itr)
	{
		do
		{
			cf  = gdl_hstruct_config_collector_iterator_config (itr);
			_add_tester_entry (t, cf);
		}
		while (gdl_hstruct_config_collector_iterator_next (itr));
		
		gdl_hstruct_config_collector_iterator_free (itr);
	}
}

gdl_hstruct_config_buffer *
gdl_hstruct_config_buffer_alloc (const gdl_hstruct_config_collector * c)
{
	gdl_hstruct_config_buffer * t;
	
	t = GDL_MALLOC (gdl_hstruct_config_buffer, 1);
	
	t->table  = gdl_hashtable_alloc (gdl_hash_default, gdl_hashtable_size (c->table));
	t->itable = gdl_hashtable_alloc (gdl_hash_default, gdl_hashtable_size (c->table));
	
	_init_tester_table (t, c);
	
	return t;
}

static void
_clean_tester_table (gdl_hstruct_config_buffer * t)
{
	gdl_hashtable_itr * itr;
	gdl_hstruct_config_buffer_entry * e;
	
	itr = gdl_hashtable_iterator (t->table);
	do
	{
		e = (gdl_hstruct_config_buffer_entry *) gdl_hashtable_iterator_value (itr);
		if (!e->flush)
		{
			gdl_hstruct_config_free (e->new);
		}
		GDL_FREE (e);
	}
	while (gdl_hashtable_iterator_next (itr));
	
	gdl_hashtable_iterator_free (itr);	
}

void
gdl_hstruct_config_buffer_realloc (gdl_hstruct_config_buffer * t)
{
	if (gdl_hashtable_size (t->table))
	{
		gdl_hashtable_itr * itr;
		gdl_hstruct_config_buffer_entry * e;
		
		itr = gdl_hashtable_iterator (t->table);
		do
		{
			e = (gdl_hstruct_config_buffer_entry *) gdl_hashtable_iterator_value (itr);
			if (e->flush)
			{
				e->flush = gdl_false;
				gdl_hashtable_vremove (t->itable, e->new);
				e->new     = gdl_hstruct_config_clone (e->old);
				gdl_hashtable_vadd (t->itable, e->new, e, 0);
			}
		}
		while (gdl_hashtable_iterator_next (itr));
		
		gdl_hashtable_iterator_free (itr);
	}
}

void
gdl_hstruct_config_buffer_free (gdl_hstruct_config_buffer * t)
{
	if (t)
	{
		if (gdl_hashtable_size (t->table))
		{
			_clean_tester_table (t);			
		}
		gdl_hashtable_free (t->table);
		gdl_hashtable_free (t->itable);
		GDL_FREE (t);
	}	
}

int
gdl_hstruct_config_buffer_set (gdl_hstruct_config_buffer * t, const gdl_hstruct_point_type * T, void ** p)
{
	if (gdl_hashtable_size (t->table))
	{
		gdl_hashtable_itr * itr;
		gdl_hstruct_config_buffer_entry * e;
		
		itr = gdl_hashtable_iterator (t->table);
		do
		{
			e = (gdl_hstruct_config_buffer_entry *) gdl_hashtable_iterator_value (itr);
			gdl_hstruct_config_set_point (e->new, T, p);
		}
		while (gdl_hashtable_iterator_next (itr));
		
		gdl_hashtable_iterator_free (itr);
	}
}

gdl_hstruct_config *
gdl_hstruct_config_buffer_new (gdl_hstruct_config_buffer * t, const gdl_hstruct_config * old)
{
	gdl_hstruct_config_buffer_entry * e;
	
	e = _get_tester_entry (t, old);
	
	return e->new;
}

gdl_hstruct_config *
gdl_hstruct_config_buffer_old (gdl_hstruct_config_buffer * t, const gdl_hstruct_config * new)
{
	gdl_hstruct_config_buffer_entry * e;
	
	e = _get_tester_ientry (t, new);
	
	return e->old;	
}

int
gdl_hstruct_config_buffer_flush (gdl_hstruct_config_buffer * t, const gdl_hstruct_config * new)
{
	gdl_hstruct_config_buffer_entry * e;
	
	e = _get_tester_ientry (t, new);
	
	e->flush = gdl_true;
	
	return GDL_SUCCESS;
}

int
gdl_hstruct_config_buffer_unflush (gdl_hstruct_config_buffer * t, const gdl_hstruct_config * new)
{
	gdl_hstruct_config_buffer_entry * e;
	
	e = _get_tester_ientry (t, new);
	
	e->flush = gdl_false;
	
	return GDL_SUCCESS;
}

gdl_hstruct_config_block *
gdl_hstruct_config_block_alloc (gdl_hstruct_config_collector * collector, size_t from, size_t to)
{
	size_t i, k;
	gdl_hstruct_config_itr * itr;
	gdl_hstruct_config_block * b;	
	
	itr = gdl_hstruct_config_collector_iterator (collector);
	i   = 0;
	do
	{
		gdl_hstruct_config * c = gdl_hstruct_config_collector_iterator_config (itr);
		if (i && k != c->k)
		{
			GDL_ERROR_VAL ("A block must share the same number of ancestral haplotypes", GDL_EINVAL, 0);
		}
		k = c->k;
		i++;
	}
	while (gdl_hstruct_config_collector_iterator_next (itr));
	
	gdl_hstruct_config_collector_iterator_free (itr);
	
	b = GDL_MALLOC (gdl_hstruct_config_block, 1);
	b->collector = collector;
	b->buffer    = gdl_hstruct_config_buffer_alloc (collector);
	b->from      = from;
	b->to        = to;
	b->k         = k;
	
	return b;
}

void
gdl_hstruct_config_block_free (gdl_hstruct_config_block * b)
{	
	if (b)
	{
		gdl_hstruct_config_collector_free (b->collector);
		gdl_hstruct_config_buffer_free (b->buffer);
		GDL_FREE (b);		
	}
}

size_t
gdl_hstruct_config_block_ancestral_size (gdl_hstruct_config_block * b)
{
	return b->k;
}

size_t
gdl_hstruct_config_block_from (gdl_hstruct_config_block * b)
{
	return b->from;	
}

size_t
gdl_hstruct_config_block_to (gdl_hstruct_config_block * b)
{
	return b->to;	
}

gdl_boolean
gdl_hstruct_config_block_has_hotspot (gdl_hstruct_config_block * b)
{
	gdl_hstruct_config_itr * itr;
		
	itr = gdl_hstruct_config_collector_iterator (b->collector);
	do
	{
		gdl_hstruct_config * c = gdl_hstruct_config_collector_iterator_config (itr);
		void * hot = gdl_hstruct_config_get_point (c, gdl_hstruct_point_hot);
		if (hot)
		{
			gdl_hstruct_config_collector_iterator_free (itr);
			return gdl_true;
		}
	}
	while (gdl_hstruct_config_collector_iterator_next (itr));
	gdl_hstruct_config_collector_iterator_free (itr);
	
	return gdl_false;
}

void
gdl_hstruct_config_block_set_point_eligible (gdl_hstruct_config_block * b, const gdl_hstruct_point_type * T, gdl_boolean eligible)
{
	gdl_hstruct_config_itr * itr;
		
	itr = gdl_hstruct_config_collector_iterator (b->collector);
	do
	{
		gdl_hstruct_config * c = gdl_hstruct_config_collector_iterator_config (itr);
		gdl_hstruct_config_set_point_eligible (c, T, eligible);
	}
	while (gdl_hstruct_config_collector_iterator_next (itr));
	gdl_hstruct_config_collector_iterator_free (itr);	
}

gdl_hstruct_config_collector *
gdl_hstruct_config_block_config_collector (gdl_hstruct_config_block * b)
{
	return b->collector;	
}

gdl_hstruct_config_buffer *
gdl_hstruct_config_block_buffer (gdl_hstruct_config_block * b)
{
	return b->buffer;
}

gdl_hstruct_static_config *
gdl_hstruct_static_config_alloc (size_t k)
{
	gdl_hstruct_static_config * c;
	
	c = GDL_CALLOC (gdl_hstruct_static_config, 1);
	
	c->k         = k;
	c->ancestral = GDL_CALLOC (gdl_allele *, k);
	c->mutation  = GDL_CALLOC (gdl_hashtable *, k);
	
	return c;
}

void
gdl_hstruct_static_config_free (gdl_hstruct_static_config * c)
{
	if (c)
	{
		size_t i;
		for (i = 0; i < c->k; i++)
		{
			gdl_allele_free (c->ancestral[i]);
			gdl_hashtable_free (c->mutation[i]);
		}
		GDL_FREE (c->ancestral);
		GDL_FREE (c->mutation);
		GDL_FREE (c);
	}
}

gdl_hstruct_static_config *
gdl_hstruct_static_config_clone (const gdl_hstruct_static_config * c)
{
	if (c)
	{
		size_t i;
		gdl_hstruct_static_config * c2;
		
		c2 = gdl_hstruct_static_config_alloc (c->k);
		
		c2->rho_idx = c->rho_idx;
		c2->mu_idx = c->mu_idx;
		c2->hot_idx = c->hot_idx;
		c2->f_idx = c->f_idx;
		
		for (i = 0; i < c->k; i++)
		{
			if (c->ancestral[i])
			{
				c2->ancestral[i] = gdl_entity_clone (c->ancestral[i]);	
			}	
		}
		
		return c2;
	}
	
	return NULL;	
}

int
gdl_hstruct_static_config_add_mutation (gdl_hstruct_static_config * c, size_t k, size_t allele)
{
	if (!c->mutation[k])
	{
		c->mutation[k] = gdl_hashtable_alloc (gdl_interface_uint, 5);
	}
	int status;
	gdl_string * key;
	size_t * ptr = GDL_MALLOC (size_t, 1);
	
	*ptr = allele;
	key  = gdl_string_sprintf ("%d", k);
	
	status = gdl_hashtable_update (c->mutation[k], key, ptr, 1);
	
	gdl_string_free (key);
	
	return status;
}

gdl_vector_uint *
gdl_hstruct_static_config_get_mutations (const gdl_hstruct_static_config * c, size_t k)
{
	gdl_vector_uint * m = NULL;
	
	if (c->mutation[k])
	{
		size_t i = 0;
		gdl_hashtable_itr * itr;
		
		m = gdl_vector_uint_alloc (gdl_hashtable_size (c->mutation[k]));
		
		itr = gdl_hashtable_iterator (c->mutation[k]);
		
		do
		{
			const size_t * ptr = (size_t *) gdl_hashtable_iterator_value (itr);
			gdl_vector_uint_set (m, i, *ptr);
			i++;
		}
		while (gdl_hashtable_iterator_next (itr));
		
		gdl_hashtable_iterator_free (itr);
	}
	
	return m;
}

gdl_hstruct_static_config *
gdl_hstruct_static_config_extend (const gdl_hstruct_static_config * c, size_t k, size_t f_idx)
{
	if (c && k > c->k)
	{
		size_t i;
		gdl_hstruct_static_config * c2;
		
		c2 = gdl_hstruct_static_config_alloc (k);
		
		c2->rho_idx = c->rho_idx;
		c2->mu_idx  = c->mu_idx;
		c2->hot_idx = c->hot_idx;
		c2->f_idx   = f_idx;
		
		for (i = 0; i < c->k; i++)
		{
			if (c->ancestral[i])
			{
				c2->ancestral[i] = gdl_entity_clone (c->ancestral[i]);
			}	
		}
		
		return c2;
	}
	
	return NULL;	
}

gdl_hstruct_static_config *
gdl_hstruct_static_config_fread (FILE * stream)
{
	if (stream)
	{
		size_t i, k;
		gdl_boolean has;
		int status;
		gdl_hstruct_static_config * c;
		
		status = fread (&k, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		c = gdl_hstruct_static_config_alloc (k);
		
		status = fread (&(c->rho_idx), sizeof (int), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(c->mu_idx), sizeof (int), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(c->hot_idx), sizeof (int), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(c->f_idx), sizeof (int), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		for (i = 0; i < c->k; i++)
		{
			c->ancestral[i] = gdl_entity_fread (stream);
			GDL_FREAD_STATUS (c->ancestral[i]!=0, 1);
			status = fread (&has, sizeof (gdl_boolean), 1, stream);
			GDL_FREAD_STATUS (status, 1);
			if (has)
			{
				c->mutation[i] = gdl_hashtable_alloc (gdl_interface_uint, 0);
				status = gdl_hashtable_fread (stream, c->mutation[i]);
				GDL_FREAD_STATUS (status, GDL_SUCCESS);
			}
		}
		
		return c;
	}
	
	return NULL;	
}

int
gdl_hstruct_static_config_fwrite (FILE * stream, const gdl_hstruct_static_config * c)
{
	if (stream && c)
	{
		size_t i;
		gdl_boolean has;
		int status;
		
		status = fwrite (&(c->k), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(c->rho_idx), sizeof (int), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(c->mu_idx), sizeof (int), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(c->hot_idx), sizeof (int), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(c->f_idx), sizeof (int), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		for (i = 0; i < c->k; i++)
		{
			status = gdl_entity_fwrite (stream, c->ancestral[i]);
			GDL_FWRITE_STATUS (status, GDL_SUCCESS);
			has = (c->mutation[i]) ? gdl_true : gdl_false;
			status = fwrite (&has, sizeof (gdl_boolean), 1, stream);
			GDL_FREAD_STATUS (status, 1);
			if (has)
			{
				status = gdl_hashtable_fwrite (stream, c->mutation[i]);
				GDL_FREAD_STATUS (status, GDL_SUCCESS);
			}
		}		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}

static int
_gdl_hstruct_config_template_compare_point (void ** p1, void ** p2)
{
	if (p1 != p2)
	{
		if (p1 && p2)
		{
			if (*(p1) == *(p2))
			{
				return 0;	
			}
		}
		return 1;
	}
	else
	{
		return 0;
	}
}

int
gdl_hstruct_config_template_compare (const gdl_hstruct_config_template * t, const gdl_hstruct_config * c1, const gdl_hstruct_config * c2)
{
	if (!t && c1 != c2)
	{
		return 1;	
	}
	else if (t->k && c1->k != c2->k)
	{
		return 1;
	}	
	if (t->rho && _gdl_hstruct_config_template_compare_point (c1->rho, c2->rho))
	{ 
		return 1;	
	}
	if (t->hot && _gdl_hstruct_config_template_compare_point (c1->hot, c2->hot))
	{ 
		return 1;	
	}
	if (t->mu && _gdl_hstruct_config_template_compare_point (c1->mu, c2->mu))
	{ 
		return 1;
	}
	if (t->f && _gdl_hstruct_config_template_compare_point (c1->f, c2->f))
	{ 
		return 1;
	}
		
	return 0;
}
