/*  
 * 	hstruct/fpoint.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_hstruct_point.h>

static const double GDL_HSTRUCT_FPOINT_MIN = 1.e-10;

struct _gdl_hstruct_fpoint
{
	size_t n;
	size_t k;
	double * q;
	double * uq;
	double tuq;
	gdl_boolean eligible;
};

gdl_hstruct_fpoint *
gdl_hstruct_fpoint_alloc (size_t k)
{
	size_t i;
	gdl_hstruct_fpoint * t = GDL_MALLOC (gdl_hstruct_fpoint, 1);
	
	t->n   = 0;
	t->k   = k;
	t->q   = GDL_CALLOC (double, k);
	t->uq  = GDL_CALLOC (double, k);
	t->tuq = 0;
	t->eligible = gdl_true;
	
	for (i = 0; i < k; i++)
	{
		(t->q)[i] = 1.0/k;
	}

	return t;
}

gdl_hstruct_fpoint *
gdl_hstruct_fpoint_clone (const gdl_hstruct_fpoint * t)
{
	gdl_hstruct_fpoint * c = NULL;
	
	if (t)
	{
		size_t i;
		
		c = gdl_hstruct_fpoint_alloc (t->k);
		
		for (i = 0; i < c->k; i++)
		{
			c->q[i]  = t->q[i];
			c->uq[i] = t->uq[i];	
		}
		c->tuq = t->tuq; 
		c->n   = t->n;
		c->eligible = t->eligible;
	}
	
	return c;
}

int
gdl_hstruct_fpoint_copy (gdl_hstruct_fpoint * dest, const gdl_hstruct_fpoint * src)
{
	if (dest && src)
	{
		GDL_FREE (dest->q);
		GDL_FREE (dest->uq);
		
		dest->k = src->k;
		dest->n = src->n;
		dest->tuq = src->tuq;
		dest->eligible = src->eligible;
		
		dest->q  = GDL_CALLOC (double, dest->k);
		dest->uq = GDL_CALLOC (double, dest->k);
		
		memcpy (dest->q, src->q, sizeof(double)*dest->k);
		memcpy (dest->uq, src->uq, sizeof(double)*dest->k);
	}
}

void
gdl_hstruct_fpoint_free (gdl_hstruct_fpoint * t)
{
	if (t)
	{
		GDL_FREE (t->q);
		GDL_FREE (t->uq);
		GDL_FREE (t);
	}	
}

void
gdl_hstruct_fpoint_vfree (void * t)
{
   gdl_hstruct_fpoint_free ((gdl_hstruct_fpoint *)t);
}

size_t
gdl_hstruct_fpoint_size (const gdl_hstruct_fpoint * t)
{
	return t->k;
}

gdl_boolean
gdl_hstruct_fpoint_is_eligible (const gdl_hstruct_fpoint * t)
{
	return t->eligible;	
}

void
gdl_hstruct_fpoint_set_eligible (gdl_hstruct_fpoint * t, gdl_boolean b)
{
	t->eligible = b;	
}

double
gdl_hstruct_fpoint_value (const gdl_hstruct_fpoint * t, size_t k)
{
	return t->q[k];
}

void
gdl_hstruct_fpoint_init_collect (gdl_hstruct_fpoint * t)
{
	size_t k;
	
	for (k=0; k<t->k; k++)
	{
		t->uq[k]=0;
	}
	t->tuq=0;
}

void
gdl_hstruct_fpoint_collect (gdl_hstruct_fpoint * t, size_t k, double x)
{
	if (k < t->k)
	{
		t->uq[k] += x;
		t->tuq   += x;
	}
}

void
gdl_hstruct_fpoint_update (gdl_hstruct_fpoint * t, double * abs, double * sq)
{
	double s, err;
	size_t i;
	
	for (s = 0, i = 0; i < t->k; i++)
	{
		err = (t->uq[i])/t->tuq;
		if (err > GDL_HSTRUCT_FPOINT_MIN)
		{
			s += t->uq[i];
		}
		else
		{
			t->uq[i] = GDL_HSTRUCT_FPOINT_MIN;
			printf ("F MIN VALUE %d ==> %g\n", i, t->uq[i]);
			s += t->uq[i];
		}
	}
	
	for (i = 0; i < t->k; i++)
	{
		t->uq[i] /= s;
		
		if (t->q[i])
		{
			err = (t->uq[i]-t->q[i]);
			if (abs)
			{
				(*abs) += fabs(err)/t->k;
			}
			if (sq)
			{
				(*sq)  += err*err/t->k;
			}
		}
		t->q[i]   = t->uq[i];
		t->uq[i]  = 0;
		//printf ("UPDATE::Q[%d] %g (%g)\n", i, t->q[i], err);
	}
	
	t->tuq = 0;
}

void
gdl_hstruct_fpoint_attach (gdl_hstruct_fpoint * t)
{
	t->n++;	
}

void
gdl_hstruct_fpoint_detach (gdl_hstruct_fpoint * t)
{
	if (t->n) t->n--;
}

size_t
gdl_hstruct_fpoint_inuse (const gdl_hstruct_fpoint * t)
{
	if (t)
	{	
		return t->n;
	}
	else
	{
		return 0;	
	}
}

void
gdl_hstruct_fpoint_swap (gdl_hstruct_fpoint * t, size_t k1, size_t k2)
{
	if (t)
	{
		//printf ("SWAP (%p) (%d, %d)\n", t, k1, k2);
		double x = t->q[k1];
		t->q[k1] = t->q[k2];
		t->q[k2] = x;
		x = t->uq[k1];
		t->uq[k1] = t->uq[k2];
		t->uq[k2] = x;
	}	
}

gdl_hstruct_fpoint *
gdl_hstruct_fpoint_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t k;
		gdl_hstruct_fpoint * p;
		
		status = fread (&k, sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		p = gdl_hstruct_fpoint_alloc (k);
		
		status = fread (&(p->n), sizeof (size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(p->eligible), sizeof (gdl_boolean), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&(p->tuq), sizeof (double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (p->q, sizeof (double), p->k, stream);
		GDL_FREAD_STATUS (status, p->k);
		status = fread (p->uq, sizeof (double), p->k, stream);
		GDL_FREAD_STATUS (status, p->k);
		
		return p;
	}
	
	return NULL;
}

int
gdl_hstruct_fpoint_fwrite (FILE * stream, const gdl_hstruct_fpoint * p)
{
	if (stream && p)
	{
		int status;
		
		status = fwrite (&(p->k), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(p->n), sizeof (size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(p->eligible), sizeof (gdl_boolean), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&(p->tuq), sizeof (double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (p->q, sizeof (double), p->k, stream);
		GDL_FWRITE_STATUS (status, p->k);
		status = fwrite (p->uq, sizeof (double), p->k, stream);
		GDL_FWRITE_STATUS (status, p->k);		

		return GDL_SUCCESS;
	}
	
	return GDL_EINVAL;
}

gdl_hstruct_fpoint *
gdl_hstruct_fpoint_sscanf (const gdl_string * str)
{
	if (!str)
	{
		return NULL;
	}
	size_t i, j, k, l, n;
	gdl_string * buf;
	gdl_hstruct_fpoint * f = 0;

	n   = strlen (str);
	buf = gdl_string_alloc (n);
	
	for (i=0;isspace(str[i]) && i < n;i++);
	if (i==n) {gdl_string_free (buf);GDL_ERROR_VAL (gdl_string_sprintf ("Invalid input string [ %s ]", str), GDL_EINVAL, 0);}
	for (j=i;!isspace(str[j]) && j < n;j++);
	strncpy (buf, &str[i], j-i);
	
	k = atoi (buf);
	f = gdl_hstruct_fpoint_alloc (k);
	
	if (k == 1)
	{
		f->q[0]=1.0;
		return f;	
	}
	else
	{
		for (i = 0; i < k; i++)
		{
			for (;isspace(str[j]) && j < n;j++);
			if (j == n) {gdl_string_free (buf);GDL_ERROR_VAL (gdl_string_sprintf ("Invalid input string [ %s ]", str), GDL_EINVAL, 0);}
			for (l = j;!isspace(str[l]) && l < n;l++);
			if (i < k-1 && l == n) {gdl_string_free (buf);GDL_ERROR_VAL (gdl_string_sprintf ("Invalid input string [ %s ]", str), GDL_EINVAL, 0);}
			strncpy (buf, &str[j], l-j);
			f->q[i] = (double) atof (buf);
			j=l;
		}
	}
	
	gdl_string_free (buf);
	
	return f;
}
