/*  
 *  hstruct/fuzzy.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_util.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_allele_block.h>
#include <gdl/gdl_locus_block.h>
#include <gdl/gdl_frequency_block.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gview_wrapper.h>
#include <gdl/gdl_clustering.h>
#include <gdl/gdl_hstruct_fuzzy.h>

struct _gdl_hstruct_fuzzy
{
  	int start;
  	size_t K;
  	size_t n;
  	size_t p;
  	size_t l;
  	size_t nres;
  	size_t * na;
  	double abs_res;
  	double sq_res;
  	gdl_frequencies_block * f;
  	gdl_frequencies_block * uf;
  	gdl_block    * q;
  	gdl_block    * uq;
  	gdl_loci_block * z;
	gdl_gvalues_get   * gbuf;
	// data
	gdl_gview_wrapper * gwrap;
	const gdl_allele_block  * gblock;
	const gdl_rng * rng;
};

static double
_gdl_hstruct_fuzzy_get_q (const gdl_hstruct_fuzzy * state, size_t k, size_t i)
{
	return gdl_block_get (state->q, k, i);
}

static double
_gdl_hstruct_fuzzy_get_f (const gdl_hstruct_fuzzy * state, size_t k, size_t l, size_t a)
{
	return gdl_frequencies_block_get (state->f, k, l, a);
}


static int
gdl_hstruct_fuzzy_update_fallele (gdl_hstruct_fuzzy * state,
                                                         size_t k,
                                                         size_t l, 
                                                         size_t a,
                                                         double z,
                                                         double x)
{
	double p = x*z;
	double f = gdl_frequencies_block_get (state->uf, k, l, a);
	gdl_frequencies_block_set (state->uf, k, l, a, f+p);
	return 0;
}

static int 
gdl_hstruct_fuzzy_update_qz(gdl_hstruct_fuzzy * state,
                                                    size_t k,
                                                    size_t i, 
                                                    double z)
{
	double q = gdl_block_get (state->uq, k, i);
	gdl_block_set (state->uq, k, i, q + z);
	return 0;
}                                              

static double
gdl_hstruct_fuzzy_get_pr (const gdl_hstruct_fuzzy * state,
                                                    size_t k,             
                                                    size_t i,
                                                    size_t j,
                                                    size_t l) 
{
	size_t a;
	double f;
	
	a = gdl_allele_block_get (state->gblock, i, l, j);
	if (a)
	{
		return _gdl_hstruct_fuzzy_get_f (state, k, l, a-1);
	}
	else // missing value 
	{
		double pr=0.0;
		const gdl_gvalues * x;
		
		gdl_gview_wrapper_get_allele_c (state->gwrap, i, l, j, state->gbuf);
	 	x = gdl_gvalues_get_gvalues (state->gbuf);
		if ( x != 0 ) 
		{
			for ( a = 0; a < x->size; a++)
			{
				f   = _gdl_hstruct_fuzzy_get_f (state, k, l, x->values[a]->idx);
				pr += f*x->values[a]->value;
			}		
			return pr;
		}
		else
		{
			return 1.0;
		}
	}
}

static void
gdl_hstruct_fuzzy_set_f (gdl_hstruct_fuzzy * state,
                                                    size_t k,            
                                                    size_t l,
                                                    size_t a,
                                                    double f)
{
	gdl_frequencies_block_set (state->f, k, l, a, f);
}  

static void
gdl_hstruct_fuzzy_set_q (gdl_hstruct_fuzzy * state,
                                                    size_t k,             
                                                    size_t i,
                                                    double q) 
{
	gdl_block_set (state->q, k, i, q);	
}                                                    

static double
gdl_hstruct_fuzzy_get_z (const gdl_hstruct_fuzzy * state,
                                                    size_t k,          
                                                    size_t i,
                                                    size_t j,
                                                    size_t l) 
{
	return gdl_loci_block_get (state->z, k, i, l, j);
}

static void
gdl_hstruct_fuzzy_set_z (gdl_hstruct_fuzzy * state,
                                                    size_t k,          
                                                    size_t i,
                                                    size_t j,
                                                    size_t l,
                                                    double z) 
{
	gdl_loci_block_set (state->z, k, i, l, j, z);
}


static int
gdl_hstruct_fuzzy_iterate_start (gdl_hstruct_fuzzy * state)
{
	size_t i, j, k, l;
	
	// Clean the buffers
	for ( k = 0; k < state->K; k++)
	{
		for (i = 0; i < state->n; i++)
		{
			for (j = 0; j < state->p; j++)
			{
				gdl_block_set (state->uq, k, i, j, 0.);
			}
		}	
	}
	for ( k = 0; k < state->K; k++)
	{
		for (i = 0; i < state->l; i++)
		{
			
			for ( j = 0; j < state->na[i]; j++)
			{
				gdl_frequencies_block_set (state->uf, k, i, j, 0.);
			}	
		}
	}
	
	state->abs_res = 0.;
	state->sq_res  = 0.;
	state->nres    = 0;
	
	return 0;
}

static int
gdl_hstruct_fuzzy_update_z (gdl_hstruct_fuzzy * state,
                                                    size_t i,
                                                    size_t j,
                                                    size_t l)
{
	size_t k;
	double pr, q, z, s = 0;
	
	for (k = 0; k < state->K; k++)
	{
		pr = gdl_hstruct_fuzzy_get_pr (state, k, i, j, l);
		q  = _gdl_hstruct_fuzzy_get_q (state, k, i);
		z  = q*pr;
		s += z;
		gdl_hstruct_fuzzy_set_z (state, k, i, j, l, z);
	}
	for (k = 0; k < state->K; k++)
	{
		z = gdl_hstruct_fuzzy_get_z (state, k, i, j, l);
		gdl_hstruct_fuzzy_set_z (state, k, i, j, l, z/s);
	}
	
	return 0;	
}

static int
gdl_hstruct_fuzzy_update_f (gdl_hstruct_fuzzy * state,
                                                    size_t i,
                                                    size_t j,
                                                    size_t l)
{
	size_t  k, a, na, ni;
	double z;
	
	ni  = gdl_gview_wrapper_accession_mult_c (state->gwrap, i);
	
	a = gdl_allele_block_get (state->gblock, i, l, j);
	
	//printf ("UPDATE F[%d,%d](%d) %d\n", i, l, j, a);
	
	if (a)
	{
		for (k = 0; k < state->K; k++)
		{
			z = gdl_hstruct_fuzzy_get_z (state, k, i, j, l);
			gdl_hstruct_fuzzy_update_fallele(state, k, l, a-1, z, (double)ni);
		}
	}
	else // missing values
	{
		const gdl_gvalues * x;
		gdl_gview_wrapper_get_allele_c (state->gwrap, i, l, j, state->gbuf);
		x = gdl_gvalues_get_gvalues (state->gbuf);
		if ( x != 0 ) 
		{
			for (k = 0; k < state->K; k++)
			{
				z = gdl_hstruct_fuzzy_get_z (state, k, i, j, l);
				for ( a = 0; a < x->size; a++)
				{
					gdl_hstruct_fuzzy_update_fallele(state, k, l, x->values[a]->idx, z, ni*x->values[a]->value);
				}		
			}
		}
	}
	
	return 0;
}

static int
gdl_hstruct_fuzzy_update_q (gdl_hstruct_fuzzy * state,
                                                    size_t i,
                                                    size_t j,
                                                    size_t l)
{
	size_t k;
	
	for (k = 0; k < state->K; k++)
	{
		
		double z = gdl_hstruct_fuzzy_get_z (state, k, i, j, l);
		
		//printf ("UPDATE Q [%d][%d] + %g\n", k, i, z);
		
		gdl_hstruct_fuzzy_update_qz (state, k, i, z);
		
	}
	
	return 0;
}

static void
gdl_hstruct_fuzzy_iterate_swap_f (gdl_hstruct_fuzzy * state)
{
	gdl_frequencies_block * ftmp;
	ftmp      = state->f;
	state->f  = state->uf;
	state->uf = ftmp;
}

static void
gdl_hstruct_fuzzy_iterate_swap_f_partial (gdl_hstruct_fuzzy * state, const int * lidx)
{
	size_t i, j, k;
	double f;
	
	for (i = 0; i < state->l; i++)
	{
		if (lidx[i] < 0)
		{
			for (k = 0; k < state->K; k++)
			{
				for (j = 0; j < state->na[i]; j++)
				{
					 f = gdl_frequencies_block_get (state->f, k, i, j);
					 gdl_frequencies_block_set (state->f, k, i, j, gdl_frequencies_block_get (state->uf, k, i, j));
					 gdl_frequencies_block_set (state->uf, k, i, j, f);
				}
			}	
		}	
	}	
}

static void
gdl_hstruct_fuzzy_iterate_swap_q (gdl_hstruct_fuzzy * state)
{
	gdl_block  * qtmp;
	qtmp      = state->q;
	state->q  = state->uq;
	state->uq = qtmp;
}

static void
gdl_hstruct_fuzzy_iterate_swap_q_partial (gdl_hstruct_fuzzy * state, const int * aidx)
{
	size_t i, k;
	double q;
	
	for (i = 0; i < state->n; i++)
	{
		if (aidx[i] < 0)
		{
			for (k = 0; k < state->K; k++)
			{
				 q = gdl_block_get (state->q, k, i);
				 gdl_block_set (state->q, k, i, gdl_block_get (state->uq, k, i));
				 gdl_block_set (state->uq, k, i, q);
			}	
		}
	}	
}

static int
gdl_hstruct_fuzzy_iterate_end_swap (gdl_hstruct_fuzzy * state, const int * aidx, const int * lidx)
{
	if (!lidx)
	{ 	
		gdl_hstruct_fuzzy_iterate_swap_f (state);
	}
	else
	{
		gdl_hstruct_fuzzy_iterate_swap_f_partial (state, lidx);
	}
	
	if (!aidx)
	{ 	
		gdl_hstruct_fuzzy_iterate_swap_q (state);
	}
	else
	{
		gdl_hstruct_fuzzy_iterate_swap_q_partial (state, aidx);
	}	
	
	return 0;
}

static int
gdl_hstruct_fuzzy_iterate_end_q (gdl_hstruct_fuzzy * state, const int * aidx)
{
	double s, q, e;
	size_t i, k;
	
	for ( i = 0; i < state->n; i++)
	{
		if ((aidx!=0 && aidx[i] < 0) || aidx==0)
		{
			s = 0;
			for (k = 0; k < state->K; k++)
			{
				s += _gdl_hstruct_fuzzy_get_q (state, k, i);
				//printf (">>Q[%d][%d] %g\n", k, i, _gdl_hstruct_fuzzy_get_q (state, k, i));
			}
			for ( k = 0; k < state->K; k++)
			{
				q = _gdl_hstruct_fuzzy_get_q (state, k, i);
				if ( q/s < 1.e-20) q = 0.;
				gdl_hstruct_fuzzy_set_q (state, k, i, q/s);
				e = fabs(q/s - gdl_block_get (state->uq, k, i));
				state->abs_res += e;
				state->sq_res  += e*e;
				(state->nres)++;
			}	
		}	
	}
	
	return 0;
}

static int
gdl_hstruct_fuzzy_iterate_end_f (gdl_hstruct_fuzzy * state, const int * lidx)
{
	double s, f, e;
	size_t i, j, k;
	
	for (k = 0; k < state->K; k++)
	{
		for (i = 0; i < state->l; i++)
		{
			if ((lidx != 0 && lidx[i] < 0) || lidx==0)
			{
				for (s = 0, j = 0; j < state->na[i]; j++)
				{
					s += _gdl_hstruct_fuzzy_get_f (state, k, i, j);
				}
				if (!s) s = 1;
				for ( j = 0; j < state->na[i]; j++)
				{
					f = _gdl_hstruct_fuzzy_get_f (state, k, i, j);
					gdl_hstruct_fuzzy_set_f (state, k, i, j, f/s);
					e = fabs(f/s - gdl_frequencies_block_get (state->uf, k, i, j));
					state->abs_res += e;
					state->sq_res  += e*e;
					(state->nres)++;
				}
			}
		}
	}
	
	return 0;
}

static int
gdl_hstruct_fuzzy_iterate_end (gdl_hstruct_fuzzy * state, const int * aidx, const int * lidx)
{
	int status;
	
	// Swap here the buffer
	
	status  = gdl_hstruct_fuzzy_iterate_end_swap (state, aidx, lidx);
	
	status |= gdl_hstruct_fuzzy_iterate_end_q (state, aidx);
	
	status |= gdl_hstruct_fuzzy_iterate_end_f (state, lidx);
	
	state->start = 1;
	
	return status;
}

static void
_gdl_hstruct_fuzzy_init_rng (gdl_hstruct_fuzzy * state, size_t i, size_t j, size_t l)
{
	size_t k;
	double z, s = 0.;
			
	for (k = 0; k < state->K; k++)
	{
		z = gdl_rng_uniform (state->rng);
		gdl_hstruct_fuzzy_set_z (state, k, i, j, l, z);
		s += z;
	}
	for (k = 0; k < state->K; k++)
	{
		z = gdl_hstruct_fuzzy_get_z (state, k, i, j, l);
		gdl_hstruct_fuzzy_set_z (state, k, i, j, l, z/s);
	}
}

gdl_hstruct_fuzzy *
gdl_hstruct_fuzzy_alloc (gdl_gview_wrapper * data, const gdl_allele_block * seq, const gdl_rng * rng, size_t k)
{
	size_t i, j, l, * tmp;
	gdl_hstruct_fuzzy * state;
	
	state = GDL_CALLOC (gdl_hstruct_fuzzy, 1);
	
   state->gwrap = data;
   state->gblock = seq;
   
   state->start = 1;
	state->rng   = rng;
	state->K     = k;
	state->n     = gdl_gview_wrapper_accession_size_c (state->gwrap);
	state->p     = gdl_gview_wrapper_ploidy (state->gwrap);
	state->gbuf  = gdl_gview_wrapper_get_new (state->gwrap);
	state->l     = gdl_gview_wrapper_locus_size (state->gwrap);
	state->na    = GDL_CALLOC (size_t, state->l);
	for (i = 0; i < state->l; i++)
	{
		gdl_locus * locus = gdl_gview_wrapper_get_locus (state->gwrap, i);
		state->na[i]      = gdl_locus_allele (locus);
	}
	state->f     = gdl_frequencies_block_alloc (state->K, state->l, state->na);
	state->q     = gdl_block_alloc2 (2, state->K, state->n);
	state->uf    = gdl_frequencies_block_alloc (state->K, state->l, state->na);
	state->uq    = gdl_block_alloc2 (2, state->K, state->n);
	state->z     = gdl_loci_block_alloc (state->K, state->n, state->l, state->p);
	
	return state;
}

int
gdl_hstruct_fuzzy_free (gdl_hstruct_fuzzy * state)
{
	if (state == 0) return;
	
	gdl_frequencies_block_free (state->f);
	gdl_block_free (state->q);
	gdl_frequencies_block_free (state->uf);
	gdl_block_free (state->uq);	
	gdl_loci_block_free (state->z);
	GDL_FREE (state->na);
	gdl_gvalues_get_free (state->gbuf);
	
	return GDL_SUCCESS;
}

int
gdl_hstruct_fuzzy_init (gdl_hstruct_fuzzy * state)
{
	int status=0;
	size_t i, j, l, k;
	
	// Randomly init z
	for (i = 0; i < state->n; i++)
	{
		for (j = 0; j < state->p; j++)
		{
			for (l = 0; l < state->l; l++)
			{
				_gdl_hstruct_fuzzy_init_rng (state, i, j, l);
				status |= gdl_hstruct_fuzzy_update_f (state, i, j, l);
				status |= gdl_hstruct_fuzzy_update_q (state, i, j, l);
			}
		}
	}
	
	status |= gdl_hstruct_fuzzy_iterate_end (state, NULL, NULL);
	
	return status;
}

int
gdl_hstruct_fuzzy_init_update (gdl_hstruct_fuzzy * state, const int * aidx, const int * lidx, const void * db, double (*get_q)(const void * db, size_t k, size_t i), double (*get_f)(const void * db, size_t k, size_t l, size_t a))
{
	int status=0;
	size_t i, j, l, k, a;
	
	for (i = 0; i < state->n; i++) // loop on individuals
	{
		if (aidx[i]>=0) 
		{
			// set the previous admixture
			for (k = 0; k < state->K; k++)
			{
				gdl_hstruct_fuzzy_set_q (state, k, i, (*get_q)(db, k, aidx[i]));
			}
		}
		for (j = 0; j < state->p; j++) // loop on phases
		{
			for (l = 0; l < state->l; l++) // loop on loci
			{
				if ((i == 0 && j == 0) && lidx[l] >= 0)
				{
					// set the previous locus allele frequencies
					for (a = 0; a < state->na[l]; a++)
					{ 
						for (k = 0; k < state->K; k++)
						{
							gdl_hstruct_fuzzy_set_f (state, k, l, a, (*get_f)(db, k, lidx[l], a));
						}
					}
				}
				if (aidx[i] < 0 || lidx[l] < 0)
				{
					// rng init
					_gdl_hstruct_fuzzy_init_rng (state, i, j, l);
				}
				else 
				{
					// get the z = Pr (zijl | xijl) 
					gdl_hstruct_fuzzy_update_z (state, i, j, l);
				}
				if (lidx[l] < 0)
				{
					status |= gdl_hstruct_fuzzy_update_f (state, i, j, l);
				}
				if (aidx[i] < 0)
				{
					status |= gdl_hstruct_fuzzy_update_q (state, i, j, l);
				}				
			}
		}		
	}
	
	status |= gdl_hstruct_fuzzy_iterate_end (state, aidx, lidx);
	
	return status;
}

int
gdl_hstruct_fuzzy_iterate (gdl_hstruct_fuzzy * state)
{
	size_t i, j, l;
	int status;
	
	status = gdl_hstruct_fuzzy_iterate_start (state);
	
	for (i = 0; i < state->n; i++)
	{
		for (j = 0; j < state->p; j++)
		{
			for (l = 0; l < state->l; l++)
			{
				status |= gdl_hstruct_fuzzy_update_z (state, i, j, l);
				status |= gdl_hstruct_fuzzy_update_f (state, i, j, l);
				status |= gdl_hstruct_fuzzy_update_q (state, i, j, l);				
			}
		}
	}
	
	status |= gdl_hstruct_fuzzy_iterate_end (state, NULL, NULL);
	
	return status;
}

static int
gdl_hstruct_fuzzy_allele_imputation (gdl_hstruct_fuzzy * state, size_t i, size_t l, gdl_gvalues * x)
{
	if (x)
	{
		size_t j, k;
		double q, f;
		gdl_gvalue * gx;
		
		for (j = 0; j < x->size; j++)
		{
			gx = x->values[j];
			gx->value = 0;
			for (k = 0; k < state->K; k++)
			{
				q = _gdl_hstruct_fuzzy_get_q (state, k, i);
				f = _gdl_hstruct_fuzzy_get_f (state, k, l, gx->idx);
				gx->value += q*f;
			}
			//printf ("ALLELE [%d][%d][%d] %g\n", i, l, gx->idx, gx->value);
		}
	}
}

int
gdl_hstruct_fuzzy_imputation (gdl_hstruct_fuzzy * state)
{
	size_t na, nl, i, ic, j, l, il;
	const size_t * lidx;
	int status;
	gdl_gvalues * x;
	
	na = gdl_gview_wrapper_missing_accession_size (state->gwrap);
	
	for (i = 0; i < na; i++)
	{
		ic = gdl_gview_wrapper_missing_accession_idx_c (state->gwrap, i);
		for (j = 0; j < state->p; j++)
		{
			nl   = gdl_gview_wrapper_missing_hlocus_size (state->gwrap, i, j);
			lidx = gdl_gview_wrapper_missing_hlocus_idx (state->gwrap, i, j);
			for (l = 0; l < nl; l++)
			{
				il = lidx[l];
				//printf ("MISSING DATA [%d, %d, %d]\n", ic, j, il);
				x = gdl_gview_wrapper_missing_hget (state->gwrap, i, j, l);
				gdl_hstruct_fuzzy_allele_imputation (state, ic, il, x);
			}	
		}	
	}
	
	return GDL_SUCCESS;
}

double
gdl_hstruct_fuzzy_loglikelihood (const gdl_hstruct_fuzzy * state)
{
	size_t i, j, k, l;
	double s, p, q, v = 0.;
	
	for (i = 0; i < state->n; i++)
	{
		for (j = 0; j < state->p; j++)
		{
			for (l = 0; l < state->l; l++)
			{
				s = 0;
				for (k = 0; k < state->K; k++)
				{
					p  = gdl_hstruct_fuzzy_get_pr (state, k, i, j, l);
					q  = _gdl_hstruct_fuzzy_get_q (state, k, i);
					s += p*q;
				}
				v += log (s);			
			}
		}
	}
	
	return v;
}

double
gdl_hstruct_fuzzy_residual_sq (const gdl_hstruct_fuzzy * state)
{
	return state->sq_res/state->nres;
}

double
gdl_hstruct_fuzzy_residual_abs (const gdl_hstruct_fuzzy * state)
{
	return state->abs_res/state->nres;
}

double
gdl_hstruct_fuzzy_get_pop_q (const gdl_hstruct_fuzzy * state, size_t k)
{
	size_t i, nn, m;
	double q = 0;

	for (nn = i = 0; i < state->n; i++)
	{
		m = gdl_gview_wrapper_accession_mult_c (state->gwrap, i);
		q += _gdl_hstruct_fuzzy_get_q (state, k, i)*m;
		nn+=m;
	}
	
	return q/(double)nn;
}

double
gdl_hstruct_fuzzy_get_q (const gdl_hstruct_fuzzy * state, size_t k, size_t i)
{
	size_t ii;
	ii = gdl_clustering_cluster (gdl_gview_wrapper_clustering (state->gwrap), i);
	return _gdl_hstruct_fuzzy_get_q (state, k, ii);
}

double
gdl_hstruct_fuzzy_get_f (const gdl_hstruct_fuzzy * state, size_t k, size_t l, size_t a)
{
	return _gdl_hstruct_fuzzy_get_f (state, k, l, a);
}

size_t
gdl_hstruct_fuzzy_get_f_max (const gdl_hstruct_fuzzy * state, size_t pop, size_t loc)
{
	double x, max  = GDL_NEGINF;
	size_t a, amax = 0;
	
	for (a = 0; a < state->na[loc]; a++)
	{
		x = _gdl_hstruct_fuzzy_get_f (state, pop, loc, a);
		if (x > max)
		{
			max  = x;
			amax = a;
		}
	}
	
	return amax;
}

size_t
gdl_hstruct_fuzzy_get_f_size (const gdl_hstruct_fuzzy * state, size_t loc)
{
	return state->na[loc];
}
