/*  
 *  hstruct/partfunc.c
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:44 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_mask.h>
#include <gdl/gdl_gmap.h>
#include <gdl/gdl_hstruct_data.h>
#include <gdl/gdl_hstruct_model_chromosome.h>

static gdl_mask ** 
_gdl_hstruct_split_partition (const gdl_hstruct_data * D, const gdl_chromosome * chrom, void * extra, size_t * np)
{
	size_t part;
	size_t i, j, l, nl;
	const gdl_gview * gview;
	gdl_entity_mask * plocus;
	gdl_mask * cmask, ** pmask, * tmp;
	
	part = *((size_t *)extra);
	
	gview = D->gview;
	
	cmask = gdl_mask_alloc ();
	
	gdl_mask_set (cmask, GDL_LOCUS, gdl_chromosome_get_mask (chrom, gdl_mask_get (D->gmask, GDL_LOCUS)), 1);
	gdl_mask_set (cmask, GDL_ACCESSION, gdl_mask_get_clone (D->gmask, GDL_ACCESSION), 1);
	
	nl = GDL_GVIEW_LOCUS_SIZE (gview, cmask);
	
	*np = nl / part;
	
	if (nl % part >= part/2)
	{
		(*np)++;
	}
	
	pmask = GDL_MALLOC (gdl_mask *, *np);
	
	for (l = i = 0; i < *np; i++)
	{
		pmask[i] = gdl_mask_alloc ();
		
		plocus   = gdl_entity_mask_alloc (part);
		
		for (j = 0; j < part && l < nl; j++, l++)
		{
			gdl_locus * loc = GDL_GVIEW_GET_LOCUS (gview, cmask, l);
			gdl_entity_mask_add (plocus, loc);
		}
		if (i == *np-1)
		{
			for (; l < nl; l++)
			{
				gdl_locus * loc = GDL_GVIEW_GET_LOCUS (gview, cmask, l);
				gdl_entity_mask_add (plocus, loc);
			}
		}
				
		gdl_mask_set (pmask[i], GDL_LOCUS, plocus, 1);
		gdl_mask_set (pmask[i], GDL_ACCESSION, gdl_mask_get_clone (D->gmask, GDL_ACCESSION), 0);
		
		//tmp = gdl_gview_gmask_uninformative (gview, pmask[i], GDL_ACCESSION);
		//gdl_mask_free (pmask[i]);
		//pmask[i] = tmp;
	}
	
	gdl_mask_free (cmask);
	
	return pmask;	
}

static gdl_mask ** 
_gdl_hstruct_distance_partition (const gdl_hstruct_data * D, const gdl_chromosome * chrom, void * extra, size_t * np)
{
	size_t i, j, l, nl;
	gdl_gdistance * d1, * d2, * max;
	const gdl_gview * gview;
	const gdl_locus * l1, * l2;
	gdl_entity_mask * plocus;
	gdl_mask * cmask, ** pmask, * tmp;
	
	max = (gdl_gdistance *) extra;
	
	gview = D->gview;
	
	cmask = gdl_mask_alloc ();
	
	gdl_mask_set (cmask, GDL_LOCUS, gdl_chromosome_get_mask (chrom, gdl_mask_get (D->gmask, GDL_LOCUS)), 1);
	gdl_mask_set (cmask, GDL_ACCESSION, gdl_mask_get_clone (D->gmask, GDL_ACCESSION), 1);
	
	nl = GDL_GVIEW_LOCUS_SIZE (gview, cmask);
	
	for (*np = 1, l = 0; l < nl - 1; l++)
	{
		l1 = GDL_GVIEW_GET_LOCUS (gview, cmask, l);
		l2 = GDL_GVIEW_GET_LOCUS (gview, cmask, l+1);
		d1 = gdl_chromosome_search_distance (chrom, l1, l2);
		d2 = gdl_gdistance_convert (d1, max->type);
		if (d2)
		{
			if (d2->value >= max->value)
			{
				(*np)++;
			}
			gdl_gdistance_free (d2);	
		}
		gdl_gdistance_free (d1);
	}
	
	if (*np > 1)
	{
		pmask = GDL_MALLOC (gdl_mask *, *np);
		
		for (l = i = 0; i < *np; i++)
		{
			pmask[i] = gdl_mask_alloc ();
			
			for (; l < nl-1; l++)
			{
				l1 = GDL_GVIEW_GET_LOCUS (gview, cmask, l);
				l2 = GDL_GVIEW_GET_LOCUS (gview, cmask, l+1);
				d1 = gdl_chromosome_search_distance (chrom, l1, l2);
				d2 = gdl_gdistance_convert (d1, max->type);
				gdl_mask_add (pmask[i], l1);
				if (d2)
				{
					if (d2->value < max->value)
					{
						gdl_gdistance_free (d2);
					}
					else
					{
						l++;
						gdl_gdistance_free (d2);
						break;
					}
				}
				gdl_gdistance_free (d1);
			}
			if (i == *np-1)
			{
				l1 = GDL_GVIEW_GET_LOCUS (gview, cmask, l);
				gdl_mask_add (pmask[i], l1);
			}
			
			gdl_mask_set (pmask[i], GDL_ACCESSION, gdl_mask_get_clone (D->gmask, GDL_ACCESSION), 0);
			//tmp = gdl_gview_gmask_uninformative (gview, pmask[i], GDL_ACCESSION);
			//gdl_mask_free (pmask[i]);
			//pmask[i] = tmp;
		}
		
		gdl_mask_free (cmask);
	}
	else
	{
		*np = 1;
		pmask  = GDL_MALLOC (gdl_mask *, 1);
		*pmask = cmask;
	}
	
	return pmask;	
}

static gdl_mask ** 
gdl_hstruct_heterozygous_partition (const gdl_hstruct_data * D, const gdl_chromosome * chrom, void * extra, size_t * np)
{
	
}

const gdl_hstruct_partition_fonction gdl_hstruct_split_partition = &_gdl_hstruct_split_partition;
const gdl_hstruct_partition_fonction gdl_hstruct_distance_partition = &_gdl_hstruct_distance_partition;
